//
// RemoteOutputStream.java
//
//	OutputStream to a remote object.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.proxy.util;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.comm.*;
import java.io.*;

/**
 * RemoteOutputStream is an OutputStream that transfers data written
 * to a process running on the server.
 */
class RemoteOutputStream extends OutputStream {
    Connection _conn;
    Packet _packet;

    /**
     * Construct a RemoteOutputStream.
     * 
     * @param conn Connection to the server.
     * @param packet Template packet for sending data to the server.
     */
    RemoteOutputStream(Connection conn, Packet packet) {
	_conn = conn;
	_packet = packet;
    }

    /**
     * Write one byte to the server.
     * 
     * @param b byte to write.
     * 
     * @exception IOException if an I/O error occurs.
     */
    public void write(int b) throws IOException {	
	byte data[] = new byte[1];
	data[0] = (byte)b;
	write(data, 0, 1);
    }

    /**
     * Write some bytes to the server.
     * 
     * @param b array of bytes to write.
     * @param off offset into the array at which we are to start
     *            writing.
     * @param len Number of bytes to write.
     * 
     * @exception IOException if an I/O error occurs.
     */
    public void write(byte b[], int off, int len) throws IOException {
	// don't write 0 bytes, because that's how we tell the server
	// that a close has occurred.
	if (len == 0) {
	    return;
	}
	if (off == 0) {
	    _packet.setData(b, len);
	} else {
	    byte data[] = new byte[len - off];
	    System.arraycopy(b, off, data, 0, len);
	    _packet.setData(data, len);
	}
	_conn.sendPacket(_packet);
	_packet.setData(null, 0);
    }

    /**
     * Tell the server to close the stream.  What we're doing here is
     * sending a packet with 0 bytes, which is the server's cue to
     * close the stream.
     * 
     * @exception IOException in an I/O error occurs.
     */
    public void close() throws IOException {
	_conn.sendPacket(_packet);
    }
}
