//
// ArrowIcon.java
//
//	An arrow icon which can point in the 4 directions.
//
//
//  Copyright (c) 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import com.sgi.sysadm.util.*;
import javax.swing.*;
import java.awt.*;

/**
 * ArrowIcon extends Icon. A common usage would be to pass it
 * to JButton. It draws the arrow proportionally within the component.
 *
 * For example,
 * JButton button = new JButton(new ArrowIcon(rs, ArrowIcon.UP));
 * will create  a button with an arrow icon pointing up.
 */
public class ArrowIcon implements Icon {
    
    /**
     * Passed to constructor to create arrow pointing up
     */
    public final static int UP = 1;
    
    /**
     * Passed to constructor to create arrow pointing down
     */
    public final static int DOWN = 2;
    
    /**
     * Passed to constructor to create arrow pointing right
     */
    public final static int RIGHT = 3;
    
    /**
     * Passed to constructor to create arrow pointing left
     */
    public final static int LEFT = 4;

    /**
     * The resource <i>ArrowIcon.iconWidth</i> or
     * <i>&lt;componentName&gt;.iconWidth</i> is an Integer specifying the
     * width of the arrow in points.
     */
    public final static String ICON_WIDTH = "iconWidth";

    /**
     * The resource <i>ArrowIcon.iconHeight</i> or
     * <i>&lt;componentName&gt;.iconHeight</i> is an Integer specifying the
     * height of the arrow in points.
     */
    public final static String ICON_HEIGHT = "iconHeight";

    /**
     * The resource <i>ArrowIcon.iconEnabledColor</i> or
     * <i>&lt;componentName&gt;.iconEnabledColor</i> is the color which the
     * icon will be painted in on enabled Components.
     */
    public final static String ICON_ENABLED_COLOR = "iconEnabledColor";

    /**
     * The resource <i>ArrowIcon.iconDisabledColor</i> or
     * <i>&lt;componentName&gt;.iconDisabledColor</i> is the color which the
     * icon will be painted in on disabled Components.
     */
    public final static String ICON_DISABLED_COLOR = "iconDisabledColor";

    private int _direction;
    private int _width;
    private int _height;
    private Color _enabledColor;
    private Color _disabledColor;
    

    /**
     * Constructor
     * 
     * @param rs             Resource Stack needed for color of arrow
     *                       which is retrieved from a SysadmUI
     *                       (unless overridden by client).
     * 
     * @param componentName  Prefix for looking up resources.
     *
     * @param arrowDirection Takes an int which should be one of following
     *                       constants : ArrowIcon.UP, ArrowIcon.DOWN, 
     *                       ArrowIcon.RIGHT, and ArrowIcon.LEFT.
     */
    public ArrowIcon(ResourceStack rs, String componentName,
		     int arrowDirection) {
	Log.assert(arrowDirection > 0 && arrowDirection < 5, "Bad"+
		   "parameter passed to ArrowIcon");
 	Log.assert(rs != null, "NULL resource stack passed to ArrowIcon");
	_direction = arrowDirection;
	setIconWidth(rs.getPixels(componentName + "." + ICON_WIDTH));
	setIconHeight(rs.getPixels(componentName + "." + ICON_HEIGHT));
        _enabledColor = rs.getColor(componentName + "." +
                                    ICON_ENABLED_COLOR);
        _disabledColor = rs.getColor(componentName + "." +
                                    ICON_DISABLED_COLOR);
    }

    /**
     * Constructor
     * 
     * @param rs             Resource Stack needed for color of arrow
     *                       which is retrieved from a SysadmUI
     *                       (unless overridden by client).
     *
     * @param arrowDirection Takes an int which should be one of following
     *                       constants : ArrowIcon.UP, ArrowIcon.DOWN, 
     *                       ArrowIcon.RIGHT, and ArrowIcon.LEFT.
     */
    public ArrowIcon(ResourceStack rs, int arrowDirection) {
	this(rs, "ArrowIcon", arrowDirection);
    }

    /** 
     * Draw the arrow icon proportional to the component c it is in.
     *
     * @param c Component in which the icon is being drawn.
     * @param g Graphics context.
     * @param x x coordinate to start drawing from. This value is 
     *          not used by the method.
     * @param y y coordinate to start drawing from. This value is 
     *          not used by the method.
     */
    public void paintIcon(Component c, Graphics g, int x, int y) {
        Color col = g.getColor();
        
        if(c.isEnabled()){
	    g.setColor(_enabledColor);
	} 
        else{
	    g.setColor(_disabledColor);
	}

	Dimension dim = c.getSize();
	
	// Based on direction, figure out vertices.
	int[] xpoints = new int[3];
	int[] ypoints = new int[3];
	if (_direction == RIGHT) {
	    xpoints[0] = x;
	    ypoints[0] = y;
	    xpoints[1] = x;
	    ypoints[1] = y + _height - 1;;
	    xpoints[2] = x + _width - 1;
	    ypoints[2] = y + _height / 2;
	} else if (_direction == LEFT) {
	    xpoints[0] = x + _width - 1;
	    ypoints[0] = y;
	    xpoints[1] = xpoints[0];
	    ypoints[1] = y + _height - 1;
	    xpoints[2] = x;
	    ypoints[2] = y + _height / 2;
	} else if (_direction == UP) {
	    xpoints[0] = x;
	    ypoints[0] = y + _height - 1;
	    xpoints[1] = x + _width - 1;
	    ypoints[1] = ypoints[0];
	    xpoints[2] = x + _width / 2;
	    ypoints[2] = y;
	} else if (_direction == DOWN) {
	    xpoints[0] = x;
	    ypoints[0] = y;
	    xpoints[1] = x + _width - 1;
	    ypoints[1] = y;
	    xpoints[2] = x + _width / 2;
	    ypoints[2] = y + _height - 1;
	} else return;

	Polygon polygon = new Polygon(xpoints, ypoints, 3);
	g.drawPolygon(polygon);
	g.fillPolygon(polygon);
	
	g.setColor(col);
    }

    /**
     * Set the color to be used to draw the enabled arrow.
     * 
     * @param color color to be used to draw the enabled arrow.
     */
    public void setEnabledColor(Color color) {
	_enabledColor = color;
    }
    /**
     * Set the color to be used to draw the disabled arrow.
     * 
     * @param color color to be used to draw the disabled arrow.
     */
    public void setDisabledColor(Color color) {
	_disabledColor = color;
    }
    /**
     *  Gets width of icon.
     */
    public int getIconWidth() {return _width;}

    /**
     *  Gets height of icon
     */
    public int getIconHeight() {return _height;}
    
    /**
     * Sets width of icon.
     *
     * @param width Width of icon.
     */
    public void setIconWidth(int width) {
	_width = width;
	// We need to have odd width in order to get isosceles triangles.
	if ((_direction == UP || _direction == DOWN) && (_width & 1) == 0) {
	    _width--;
	}
    }
    
    /**
     * Sets height of icon
     *
     * @param height Height of icon.
     */
    public void setIconHeight(int height) {
	_height = height;
	if ((_direction == LEFT || _direction == RIGHT)
	    && (_height & 1) == 0) {
	    // We need to have odd height in order to get isosceles triangles.
	    _height--;
	}
    }
}
