//
// ComponentTable.java
//
//	A subclass of JTable that supports embedding JComponents into
//	the table and sorting the table by clicking on a column header
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;
 
import com.sgi.sysadm.ui.event.*;
import javax.swing.*;
import java.awt.event.*;
import javax.swing.table.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.util.*;
import java.awt.*;


/** 
 * A subclass of JTable that supports embedding JComponents into the
 * table and sorting the table by clicking on a column header.
 * <p>
 * All the constructors are the same as JTable's.
 */
public class ComponentTable extends JTable 
                            implements MouseListener, MouseMotionListener{

    private Vector _sortListeners = new Vector();
    private Cursor _originalCursor;
    private Cursor _lastCursor;
    private boolean _added = false;
    
    
    /**
     * @see JTable#JTable()
     */
    public ComponentTable(){
	super();
	setupTable();
    }
    
    /**
     * @see JTable#JTable(int, int)
     */
    public ComponentTable(int numColumns, int numRows) {
	super(numColumns, numRows);
	setupTable();
    }
    
    /**
     * @see JTable#JTable(Object[][], Object[])
     */
    public ComponentTable(Object data[][], Object columnNames[]){
	super(data, columnNames);
	setupTable();
    }
    
    /**
     * @see JTable#JTable(TableModel) 
     */
    public ComponentTable(TableModel dm) {
	super(dm);
	setupTable();
    }
    
    /**
     * @see JTable#JTable(TableModel, TableColumnModel)
     */
    public ComponentTable(TableModel dm, TableColumnModel cm) {
	super(dm, cm);
	setupTable();
    }
    
    /**
     * @see JTable#JTable(TableModel, TableColumnModel, ListSelectionModel) 
     */
    public ComponentTable(TableModel dm, 
		   TableColumnModel cm,
		   ListSelectionModel sm) {
	super(dm, cm, sm);
	setupTable();
    }
    
    /**
     * @see JTable#JTable(Vector, Vector) 
     */
    public ComponentTable(Vector data, Vector columnNames){
	super(data, columnNames); 
	setupTable();
    }
    
    /**
     * Override addNotify() in order to initialize table headers.  The
     * goal is to align the headers with the columns underneath.
     */
    public void addNotify() {
	super.addNotify();
	if (_added) {
	    return;
	}
	_added = true;
	int columnCount = getColumnCount();
	TableColumnModel columnModel = getColumnModel();
	for (int ii = 0; ii < columnCount; ii++) {
	    TableColumn col = columnModel.getColumn(ii);
	    // Figure out alignment based on the renderer for this
	    // column.
	    int alignment;
	    try {
		alignment = ((DefaultTableCellRenderer)
		    col.getCellRenderer()).getHorizontalAlignment();
	    } catch (ClassCastException ex) {
		alignment = JLabel.LEFT;
	    } catch (NullPointerException ex) {
		alignment = JLabel.LEFT;
	    }
	    // Line the headers up with the columns underneath
	    // them.
	    try {
		DefaultTableCellRenderer headerRenderer =
		    (DefaultTableCellRenderer)col.getHeaderRenderer();
		headerRenderer.setHorizontalAlignment(alignment);
	    } catch (ClassCastException ex) {
	    }
	}
    }

    /** 
     * Sets up the listeners that we need to implement this table.
     */
    private void setupTable() {
	_originalCursor = getCursor();
	// Add a renderer for values of type JComponent
	setDefaultRenderer(JComponent.class, new TableCellComponentRenderer());

	// Add mouse listeners so that we can send the events to
	// embedded components.
	addMouseListener(this);
	addMouseMotionListener(this);

	// setup a mouse listener on the column header so that we can
	// request a sort.
	MouseAdapter listMouseListener = new MouseAdapter() {
	    public void mouseClicked(MouseEvent event) {
		int column = columnAtPoint(event.getPoint());
		if(event.getClickCount() == 1 && column != -1) {
		    column = convertColumnIndexToModel(column);
		    for (int ii = 0; ii < _sortListeners.size() ; ii++) {
			((TableSortRequestListener)_sortListeners.
			 elementAt(ii)).tableSortRequested(
			     new TableSortRequestEvent(ComponentTable.this,
						       column,
						       event.getModifiers()));
		    } 
		}
	    }
	};
	getTableHeader().addMouseListener(listMouseListener);
    }

    /**
     * Adds a listener that will be notified when the user requests
     * that a table be sorted on a particular column.
     *
     * @param listener The listener to add
     */
    public void addTableSortRequestListener(
	TableSortRequestListener listener) {
	_sortListeners.addElement(listener);
    }
    
    ;
    /**
     * Removes a sort request listener
     * 
     * @param listener The listener to remove
     */
    public void removeTableSortRequestListener(
	TableSortRequestListener listener) {
	_sortListeners.removeElement(listener);
    }
    
    
    //----------------MouseListener methods-----------------
    
    /** 
     * @see java.awt.event.MouseListener#mouseClicked(MouseEvent)
     */
    public void mouseClicked(MouseEvent event) {
	handlePassThrough(event);
    }
     
    /** 
     * Called if the mouse is pressed over the table.
     * Finds the component (if any) over which the mouse was pressed
     * and sends the event to that component.
     * @see java.awt.event.MouseListener#mousePressed(MouseEvent)
     */
    public void mousePressed(MouseEvent event) {
	handlePassThrough(event);
    }
     
    /** 
     * Called if the mouse is released over the table.
     * Finds the component (if any) over which the mouse was released
     * and sends the event to that component. 
     * @see java.awt.event.MouseListener#mouseReleased(MouseEvent)
     */
    public void mouseReleased(MouseEvent event) {
	handlePassThrough(event);
    }

    /** 
     * Called when the mouse enters the table.  Currently ignored.
     *
     * @see java.awt.event.MouseListener#mouseEntered(MouseEvent)
     */
    public void mouseEntered(MouseEvent event) {
	// do nothing
    }
    
    /** 
     * Called when the mouse leaves the table.  Currently ignored.
     *
     * @see java.awt.event.MouseListener#mouseExited(MouseEvent)
     */
    public void mouseExited(MouseEvent event) {
	// do nothing
    }
    
    /**
     * Called when we're resized to set the sizes of the columns to
     * fit the table.  Overridden to do nothing if width is less than
     * 1 to avoid botched columns at initialization.
     * 
     * @see JTable#sizeColumnsToFit(boolean)
     */
    public void sizeColumnsToFit(boolean lastColumnOnly) {
	if (getWidth() < 1) {
	    return;
	}
	super.sizeColumnsToFit(lastColumnOnly);
    }

    //--------------MouseMotionListener methods----------------
    
    /** 
     * Called when the mouse is dragged over the table.  Generates
     * entered, exited, and dragged mouse events and sends them to the
     * embedded components.
     * @see java.awt.event.MouseMotionListener#mouseDragged(MouseEvent)
     */
    public void mouseDragged(MouseEvent event) {
	mouseMovedOrDragged(event);
    }
  
     /** 
     * Called when the mouse is moved over the table.  Generates
     * entered, exited, and moved mouse events and sends them to the
     * embedded components.
     *
     * @see java.awt.event.MouseMotionListener#mouseMoved(MouseEvent)
     */
    public void mouseMoved(MouseEvent event) {
	mouseMovedOrDragged(event);
    }
      
    private JComponent _lastComponent;
    private int _lastRow;
    private int _lastColumn;
 
    private void mouseMovedOrDragged(MouseEvent event) {
	int initialType = event.getID();
	Point currentPoint = event.getPoint();
	int currentRow = rowAtPoint(currentPoint);
	int currentColumn = columnAtPoint(currentPoint);
	JComponent currentComponent = findComponent(currentRow,currentColumn);
	Rectangle oldBounds = null;
	if (currentComponent != null) {
	    oldBounds = currentComponent.getBounds();
	    currentComponent.setBounds(getCellRect(
		currentRow, currentColumn, false));
	}
	if (currentComponent == null && _lastComponent == null) {
	    // from non-component to non-component
	    //do nothing
	} else if (currentComponent != null && _lastComponent == null) {
	    // from non-component to component
	    MouseEvent e = translateEvent(event, 
					  currentRow,
					  currentColumn,
					  MouseEvent.MOUSE_ENTERED,
					  currentComponent);
	    currentComponent.dispatchEvent(e);
	    fireCellChanged(currentRow, currentColumn);
	    setCursor(currentComponent, e);
	} else if (currentComponent == null && _lastComponent != null) {
	    // from component to non-component
	    _lastComponent.dispatchEvent(
		translateEvent(event,
			       currentRow,
			       currentColumn,
			       MouseEvent.MOUSE_EXITED,
			       _lastComponent));
	    fireCellChanged(_lastRow, _lastColumn);
	    setCursor(_originalCursor);
	    _lastCursor = _originalCursor;
	} else if (currentComponent == _lastComponent) {
	    // from component to same component
	    MouseEvent e = translateEvent(event,
					  currentRow,
					  currentColumn,
					  initialType,
					  currentComponent);
	    currentComponent.dispatchEvent(e);
	    fireCellChanged(currentRow, currentColumn);
	    setCursor(currentComponent, e);
	} else {
	    // from component to different component
	    _lastComponent.dispatchEvent(
		translateEvent(event,
			       currentRow,
			       currentColumn,
			       MouseEvent.MOUSE_EXITED,
			       _lastComponent));
	    fireCellChanged(_lastRow, _lastColumn);
	    MouseEvent e = translateEvent(event,
					  currentRow,
					  currentColumn,
					  MouseEvent.MOUSE_ENTERED,
					  currentComponent);
	    currentComponent.dispatchEvent(e);
	    fireCellChanged(currentRow, currentColumn);
	    setCursor(currentComponent, e);
	}
	if (currentComponent != null) {
	    currentComponent.setBounds(oldBounds);
	}
	_lastComponent = currentComponent;
	_lastRow = currentRow;
	_lastColumn = currentColumn;
    }

    //---------------MouseEvent utility methods-------------------
     
    /**
     * Sets the cursor to whatever the component wants it to be (if
     * the component implements DynamicCursor)
     */
    private void setCursor(Component component, MouseEvent e) {
	if (component instanceof DynamicCursor) {
	    Cursor c =
		((DynamicCursor)component).getCursorAt(
		    e.getX(),e.getY());
	    if (c != _lastCursor) {
		setCursor(c == null ? _originalCursor: c);
		_lastCursor = c;
	    }
	    
	}  
    }
    

    /** 
     * Passes a mouse event straight to an embedded component after
     * translating to the component's coordinates.
     *
     * @param event The MouseEvent to pass through.
     */
    private void handlePassThrough(MouseEvent event) {
	Point point = event.getPoint();
	int row = rowAtPoint(point);
	int column = columnAtPoint(point);
	JComponent c = findComponent(row, column);
	if (c != null) {
	    Rectangle oldBounds = c.getBounds();
	    c.setBounds(getCellRect(row, column, false));
	    c.dispatchEvent(translateEvent(event, row, column, c));
	    c.setBounds(oldBounds);
	    fireCellChanged(row, column);
	}
    }

    /**
     * Finds the component that is at the point specified in the
     * MouseEvent
     * 
     * @return The component
     * @param row The row
     * @param column The column, with respect to the table
     */
    private JComponent findComponent(int row, int column) {
	if (row != -1 && column != -1) {
	    Object cell = getValueAt(row,column);
	    if (cell instanceof JComponent){
		return (JComponent)cell;
	    } 
	}
	return null;
    }

    /**
     * Figures out what cell is under the given point and tells the
     * table to redraw that cell.
     *
     * @param row The row
     * @param column The column, with respect to the table.
     */
    private void fireCellChanged(int row, int column) {
	int translatedColumn = convertColumnIndexToModel(column);
	tableChanged(new TableModelEvent(dataModel, 
					 row, 
					 row, 
					 translatedColumn));
    }

    /**
     * Translate an event from table coordinates to the coordinates
     * relative to the component embedded in the table.  It also
     * replaces the event id in the event as specified
     * 
     * @param event The MouseEvent to translate
     * @param row The row to translate the event to
     * @param column The column to translate the event to
     * @param id The mouse event id to use in the new event
     * @param component The component to use as the source
     * @return The translated MouseEvent.
     */
    private MouseEvent translateEvent(MouseEvent event, 
				      int row, 
				      int column,
				      int id,
				      Component component) {
	Rectangle rect = getCellRect(row,
				     column,
				     true);
	Point point = event.getPoint();
	point.translate(0-rect.x, 0-rect.y);
	return new MouseEvent(component,
			      id,
			      event.getWhen(),
			      event.getModifiers(),
			      point.x,
			      point.y,
			      event.getClickCount(),
			      event.isPopupTrigger());
    }

     /**
     * Translate an event from table coordinates to the coordinates
     * relative to the component imdebbed in the table.
     * 
     * @param event The MouseEvent to translate
     * @param row The row to translate the event to
     * @param column The column to translate the event to
     * @param component The component to use as the source
     * @return The translated MouseEvent.
     */
    private MouseEvent translateEvent(MouseEvent event, 
				      int row, int column,
				      Component component) {
	return translateEvent(event, row, column, event.getID(), component);
    }
}
