//
// EditableList.java
//
//	Editable list of elements.
//
//
//  Copyright (c) 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;
import java.util.*;

/**
 * EditableList provides a user interface for editing and displaying a
 * list of information.  The list itself is a scrollable table, and
 * the caller supplies a user interface for adding and modifying
 * entries in the list.  EditableList provides buttons for adding,
 * deleting, and modifying entries in the list.
 * <p>
 * The editor within an EditableList can be any AWT Component.
 * Typically, an editor provides one input field for each column of
 * the EditableList, and each input field has a label above it
 * describing what it is for.  The method
 * <a href=#getEditorLayout>getEditorLayout</a> can
 * be used to obtain a LayoutManager to be passed to the
 * <tt>setLayout</tt> method of a Container to facilitate development
 * of an editor that integrates smoothly with its EditableList.
 * <p>
 * An EditableList can optionally be reorderable, in which case each
 * entry in the list is numbered and buttons are provided for changing
 * the order of the entries.
 * <p>
 * EditableList uses TaskData both for its internal representation of
 * the list and to give access to the data in the list to clients.
 * TaskData is also used for communicating between EditableList and
 * the editor supplied by the caller.
 * <p>
 * The caller specifies the name of a Long Attribute to be used to
 * keep track of the number of rows in the list, and an array of
 * names, one for each column in the table.  When the "Add" button is
 * pressed, the value for each column of the row that is added is
 * obtained by getting the Attribute from TaskData, using the array
 * of names.  For each row in the table, the TaskData is populated by
 * setting string Attributes based on the array of names with the
 * row-number appended.
 * <p>
 * For example, suppose the table contains the following data:
 * <pre>
 *    rogerc     Roger Chickering
 *    kirthiga   Kirthiga Reddy
 *    wessmith   Wesley Smith
 *    ctb        Chris Beekhuis
 * </pre>
 * and the number of rows is stored in the Long Attribute named by
 * <b>numUsers</b>.  Suppose also that the names of the columns are
 * <b>loginName</b> and <b>fullName</b>.  The following code gets the
 * data out of the EditableList:
 * <pre>
 *    int numUsers = (int)taskData.getLong("numUsers");
 *    for (int row = 0; row < numUsers; row++) {
 *        String loginName = taskData.getString("loginName" + row);
 *        String fullName = taskData.getString("fullName" + row);
 *    }
 * </pre>
 * The editor sets the data to be added to the table when the "Add"
 * button is pressed with this code:
 * <pre>
 *    taskData.setString("loginName", "orosz");
 *    taskData.setString("fullName", "Jim Orosz");
 * </pre>
 * TaskDataListeners such as JStringTextComponentListener can be
 * used to synchronize editor components with the TaskData.
 * <p>
 * Attribute names for columns and the number of rows can be specified
 * either programmatically or via a ResourceStack.
 * <p>
 * An EditableList can also have invisible columns.  When the "Add"
 * button is pressed, the editor Attributes for invisible columns
 * are copied to the next row along with the visible Attributes.  When
 * the "Modify" button is pressed, the Attributes for the invisible
 * columns in the row being modified are copied back into the editor
 * TaskData.  The difference between visible and invisible columns is
 * that invisible columns have no presentation in the list's table.
 * <p>
 * Invisible columns are typically used to store information for
 * each row that is different from but related to the visible columns.
 */
public class EditableList extends JPanel implements SwingConstants {

    /**
     * The resource <i>EditableList.hGap</i> or
     * <i>&lt;componentName&gt;.hGap</i> is an Integer specifying the
     * horizontal gaps in points between the various Components (buttons,
     * labels, JTable).
     */
    public final static String HGAP = "hGap";

    /**
     * The resource <i>EditableList.vGap</i> or
     * <i>&lt;componentName&gt;.vGap</i> is an Integer specifying the
     * vertical gaps in points between the various Components (buttons,
     * labels, JTable).
     */
    public final static String VGAP = "vGap";

    /**
     * The resource <i>EditableList.hMargin</i> or
     * <i>&lt;componentName&gt;.hMargin</i> is an Integer specifying the
     * horizontal margin in points.  The horizontal margin is the gap
     * between the table and the editor and the left and right edges
     * of the containing Panel.
     */
    public final static String HMARGIN = "hMargin";

    /**
     * The resource <i>EditableList.vMargin</i> or
     * <i>&lt;componentName&gt;.vMargin</i> is an Integer specifying the
     * vertical margin in points.  The vertical margin is the gap
     * between the table and the editor and the top and bottom edges
     * of the containing Panel.
     */
    public final static String VMARGIN = "vMargin";

    /**
     * The resource <i>EditableList.editorVGap</i> or
     * <i>&lt;componentName&gt;.editorVGap</i> is an Integer
     * specifying the spacing in points between the bottom of the
     * editor and the top of the list.
     */
    public final static String EDITOR_VGAP = "editorVGap";

    /**
     * The resource <i>EditableList.numRowsAttrName</i> or
     * <i>&lt;componentName&gt;.numRowsAttrName</i> is a String
     * specifying the name of the Long Attribute in TaskData that
     * keeps track of the number of rows in the entry table.  This
     * Attribute is used to initialize the list at startup, to keep
     * track of the number of rows as the user edits the list, and to
     * programmatically get values out of the list.
     */
    public final static String NUM_ROWS_ATTR_NAME = "numRowsAttrName";

    /**
     * The resource set <i>EditableList.heading&lt;n&gt;</i> or
     * <i>&lt;componentName&gt;.heading&lt;n&gt;</i> specifies the
     * headings that appear in the table of entries.
     */
    public final static String HEADING = "heading";

    /**
     * The resource set <i>EditableList.columnAttrName&lt;n&gt;</i> or
     * <i>&lt;componentName&gt;.columnAttrName&lt;n&gt;</i> specifies the
     * Attribute name for each column in the list.  A call to
     * setColumns overrides this resource.
     * @see #setColumns
     */
    public final static String COLUMN_ATTR_NAME = "columnAttrName";

    /**
     * The resource set
     * <i>EditableList.invisibleColumnAttrName&lt;n&gt;</i> or
     * <i>&lt;componentName&gt;.invisibleColumnAttrName&lt;n&gt;</i>
     * specifies the Attribute name for invisible columns.  Each row
     * in the list can have invisible columns, whose corresponding
     * TaskData Attributes may be of any type.  A call to setColumns
     * overrides this resource.
     * <p>
     * Invisible columns are added, moved, and deleted along with the
     * visible columns, but are not displayed to the user.  They can
     * be used to store auxiliary information in the TaskData for each
     * row.
     * @see #setColumns
     */
    public final static String INVISIBLE_COLUMN_ATTR_NAME =
                                "invisibleColumnAttrName";

    /**
     * The resource <i>EditableList.changedSinceLastAddAttrName</i> or
     * <i>&lt;componentName&gt;.changedSinceLastAddAttrName</i>
     * specifies the name of a Boolean Attribute in TaskData that gets
     * bound to the state of the editor: the Attribute will be
     * <b>true</b> if the user has made changes and <b>false</b>
     * otherwise.
     * <p>
     * EditableList keeps the <i>CHANGED_SINCE_LAST_ADD_ATTR_NAME</i>
     * Attribute up to date by calling its EditVerifier's
     * <tt>changedSinceLastAdd</tt> method whenever any of the
     * Attributes for the editor change.
     * @see #changedSinceLastAdd
     * @see EditVerifier#changedSinceLastAdd
     */
    public final static String CHANGED_SINCE_LAST_ADD_ATTR_NAME =
                                 "changedSinceLastAddAttrName";

    /**
     * The resource set <i>EditableList.columnWidth&lt;n&gt;</i> or
     * <i>&lt;componentName&gt;.columnWidth&lt;n&gt;</i> specifies the
     * widths in points of the columns in the list.  Any columns
     * with missing <i>columnWidth</i> resources are sized dynamically
     * based on the size of the table.
     */
    public final static String COLUMN_WIDTH = "columnWidth";

    /**
     * The resource <i>EditableList.numberColumnWidth</i> or
     * <i>&lt;componentName&gt;.numberColumnWidth</i> is an Integer
     * specifying the width in points of the table column used for
     * numbering the rows of a reorderable list.
     */
    public final static String NUMBER_COLUMN_WIDTH = "numberColumnWidth";

    /**
     * The resource <i>EditableList.numberFormatString</i> or
     * <i>&lt;componentName&gt;.numberFormatString</i> is a String
     * specifying the format of the strings used to number the rows in
     * a reorderable list.  The string is formatted using
     * MessageFormat.format(), passing one Integer argument.
     */
    public final static String NUMBER_FORMAT_STRING = "numberFormatString";

    /**
     * The resource <i>EditableList.reorderable</i> or
     * <i>&lt;componentName&gt;.reorderable</i> is a Boolean that
     * controls whether or not the list is reorderable.  The resource
     * setting can be overridden by the caller by using the setReorderable()
     * method.
     */
    public final static String REORDERABLE = "reorderable";

    /**
     * The resource <i>EditableList.listWidth</i> or
     * <i>&lt;componentName&gt;.listWidth</i> is an Integer specifying
     * the width in points of the JTable Component used to display the
     * entries in the list.  If all of the columns in the table have
     * widths specified (via the <b>COLUMN_WIDTH</b> resource),
     * <b>LIST_WIDTH</b> will be ignored and the width of the list will
     * be set to the sum of the column widths and the column margins.
     */
    public final static String LIST_WIDTH = "listWidth";

    /**
     * The resource <i>EditableList.listHeight</i> or
     * <i>&lt;componentName&gt;.listHeight</i> is an Integer specifying
     * the height in points of the JTable Component used to display
     * the entries in the list.
     */
    public final static String LIST_HEIGHT = "listHeight";

    /**
     * The resource <i>EditableList.addButtonLabel</i> or
     * <i>&lt;componentName&gt;.addButtonLabel</i> is the String
     * displayed on the Add button.
     */
    public final static String ADD_BUTTON_LABEL = "addButtonLabel";

    /**
     * The resource <i>EditableList.modifyButtonLabel</i> or
     * <i>&lt;componentName&gt;.modifyButtonLabel</i> is the String
     * displayed on the Modify button.
     */
    public final static String MODIFY_BUTTON_LABEL = "modifyButtonLabel";

    /**
     * The resource <i>EditableList.resetButtonLabel</i> or
     * <i>&lt;componentName&gt;.resetButtonLabel</i> is the String
     * displayed on the Reset button.
     */
    public final static String RESET_BUTTON_LABEL = "resetButtonLabel";

    /**
     * The resource <i>EditableList.showModifyButton</i> or
     * <i>&lt;componentName&gt;.showModifyButton</i> is a Boolean
     * that controls whether the Modify button is visible.
     */
    public final static String SHOW_MODIFY_BUTTON = "showModifyButton";

    /**
     * The resource <i>EditableList.showResetButton</i> or
     * <i>&lt;componentName&gt;.showResetButton</i> is a Boolean
     * that controls whether the Reset button is visible.
     */
    public final static String SHOW_RESET_BUTTON = "showResetButton";

    /**
     * The resource <i>EditableList.deleteButtonLabel</i> or
     * <i>&lt;componentName&gt;.deleteButtonLabel</i> is the String
     * displayed on the Delete button.
     */
    public final static String DELETE_BUTTON_LABEL = "deleteButtonLabel";

    /**
     * The resource <i>EditableList.orderHeading</i> or
     * <i>&lt;componentName&gt;.orderHeading</i> is the String
     * displayed above the column of numbers on reorderable lists.
     */
    public final static String ORDER_HEADING = "orderHeading";

    /**
     * The resource
     * <i>EditableList.horizontalAlignment.&lt;row&gt;,&lt;column&gt;</i> or
     * <i>&lt;componentName&gt;.horizontalAlignment.&lt;row&gt;,&lt;column&gt;</i>
     * specifies the horizontal alignment of the editor component in
     * the cell specified by <i>row</i> and <i>column</i>.  Specify
     * "CENTER", "LEFT", or "RIGHT".
     */
    public final static String HORIZONTAL_ALIGNMENT =
	"horizontalAlignment";

    /**
     * The resource
     * <i>EditableList.verticalAlignment.&lt;row&gt;,&lt;column&gt;</i> or
     * <i>&lt;componentName&gt;.verticalAlignment.&lt;row&gt;,&lt;column&gt;</i>
     * specifies the vertical alignment of the editor component in
     * the cell specified by <i>row</i> and <i>column</i>.  Specify
     * "CENTER", "TOP", or "BOTTOM".
     */
    public final static String VERTICAL_ALIGNMENT =
	"verticalAlignment";

    private Component _editorComponent = null;
    private EditVerifier _verifier = null;
    private String _columnAttrs[] = null;
    private String _invisibleColumnAttrs[] = null;
    private String _headings[] = null;
    private String _numRowsAttr = null;
    private boolean _added = false;
    private TaskData _taskData;
    private JTable _table;
    private JScrollPane _scroll;
    private JButton _addButton;
    private JButton _modifyButton;
    private JButton _deleteButton;
    private JButton _upButton;
    private JButton _downButton;
    private JButton _resetButton;
    private int _hGap;
    private int _vGap;
    private int _hMargin;
    private int _vMargin;
    private int _editorVGap;
    private boolean _reorderable;
    private boolean _showModifyButton;
    private boolean _showResetButton;
    private String _numberFormatString;
    private ResourceStack _rs;
    private String _name;
    private String _orderHeading;
    private String _changedAttrName;

    /*
     * ListTable is a subclass of JTable with a sane implementation of
     * sizeColumnsToFit().
     */
    private class ListTable extends JTable {

	ListTable(TableModel model) {
	    super(model);
	}

	/*
	 * Adjust the sizes of the columns so they fit inside the
	 * table.  We spread whatever changes have to be made evenly
	 * among the resizable columns of the table.  We ignore the
	 * minimum and maximum column sizes, which is OK because
	 * EditableList doesn't ever set them.
	 *
	 * @param lastColumnOnly ignored in EditableList.
	 */
	public void sizeColumnsToFit(boolean lastColumnOnly) {
	    TableColumnModel model = getColumnModel();
	    int margin = model.getColumnMargin();
	    int tableWidth = getWidth();
	    int columnWidths = 0;

	    Vector resizableColumns = new Vector();
	    Enumeration enum = model.getColumns();
	    while (enum.hasMoreElements()) {
		TableColumn column = (TableColumn)enum.nextElement();
		columnWidths += column.getWidth() + margin;
		if (column.getResizable()) {
		    resizableColumns.addElement(column);
		}
	    }

	    int numResizableColumns = resizableColumns.size();
	    if (numResizableColumns == 0) {
		return;
	    }

	    int tableDelta = tableWidth - columnWidths;

	    int columnDelta = tableDelta / numResizableColumns;

	    for (int ii = 0; ii < numResizableColumns; ii++) {
		TableColumn column = (TableColumn)resizableColumns.elementAt(ii);
		int width = column.getWidth();
		if (ii == numResizableColumns - 1) {
		    // Put the round-off into the last column.
		    column.setWidth(width + tableDelta);
		} else {
		    column.setWidth(width + columnDelta);
		    tableDelta -= columnDelta;
		}
	    }
	}
    }

    /*
     * TableModel for EditableList's table.  Takes care of keeping the
     * contents of the table synchronized with TaskData.
     * <p>
     * If this is a reorderable list, the 0th column of the table is
     * numbers the list, and the TaskData is used starting at the
     * column number 1.
     */
    private class ListTableModel extends AbstractTableModel {

	/**
	 * Get the class for this column.
	 *
	 * @param column Column to get class for.
	 *
	 * @return class for column.
	 */
	public Class getColumnClass(int column) {
	    return String.class;
	}

	/**
	 * Get the value for a cell in the table.
	 *
	 * @param rowIndex row of cell to get.
	 * @param column column of cell to get.
	 *
	 * @return value of cell.
	 */
	public Object getValueAt(int rowIndex, int column) {
	    if (_reorderable) {
		if (column == 0) {
		    return MessageFormat.format(
			_numberFormatString,
			new Object[] { new Integer(rowIndex + 1) });
		}
		column--;
	    }
	    return _taskData.getString(_columnAttrs[column] + rowIndex);
	}

	/**
	 * Get the number of columns in the table.
	 *
	 * @return number of columns.
	 */
	public int getColumnCount() {
	    return _reorderable ? _columnAttrs.length + 1 : _columnAttrs.length;
	}

	/**
	 * Get the number of rows in the table.
	 *
	 * @return number of rows.
	 */
	public int getRowCount() {
	    return (int)_taskData.getLong(_numRowsAttr);
	}

	/**
	 * Get the header for a column.
	 *
	 * @param column Column to get header of.
	 *
	 * @return header of column.
	 */
	public String getColumnName(int column) {
	    if (_reorderable) {
		if (column == 0) {
		    return _orderHeading;
		}
		column--;
	    }
	    if (_headings == null || column >= _headings.length) {
		return null;
	    }
	    return _headings[column];
	}

	/**
	 * Called when the Add button is pressed.  Copy values from
	 * the editor into a new row of the table.
	 */
	void addRow() {
	    // The selected row does not stay sane when a row is
	    // added.  Sun Java bug 4146132.
	    _table.clearSelection();
	    int count = (int)_taskData.getLong(_numRowsAttr);
	    for (int column = 0; column < _columnAttrs.length;
		 column++) {
		String attrName = _columnAttrs[column];
		_taskData.setString(attrName + count,
				    _taskData.getString(attrName));
	    }
	    for (int column = 0; column < _invisibleColumnAttrs.length;
		 column++) {
		String attrName = _invisibleColumnAttrs[column];
		Attribute attr = _taskData.getAttr(attrName);
		_taskData.setAttr(
		    new Attribute(attrName + count,
				  attr.getTypeString(), attr.getValueString()));
	    }
	    _verifier.resetComponents(EditableList.this);
	    count++;
	    _taskData.setLong(_numRowsAttr, count);
	    fireTableRowsInserted(count - 1, count - 1);
	}

	/**
	 * Called when the Delete button is pressed.  Delete a row.
	 *
	 * @param row Row to delete.
	 */
	void deleteRow(int row) {
	    // The selected row does not stay sane when a row is
	    // added.  Sun Java bug 4146132.
	    _table.clearSelection();

	    int count = (int)_taskData.getLong(_numRowsAttr);
	    count--;
	    for (int ii = row; ii < count; ii++) {
		for (int column = 0; column < _columnAttrs.length;
		     column++) {
		    String attrName = _columnAttrs[column];
		    _taskData.setString(attrName + ii,
					_taskData.getString(attrName + (ii + 1)));
		}
		for (int column = 0; column < _invisibleColumnAttrs.length;
		     column++) {
		    String attrName = _invisibleColumnAttrs[column];
		    Attribute attr = _taskData.getAttr(attrName + (ii + 1));
		    _taskData.setAttr(new Attribute(attrName + ii,
						    attr.getTypeString(),
						    attr.getValueString()));
		}
	    }
	    for (int column = 0; column < _columnAttrs.length;
		 column++) {
		_taskData.removeAttr(_columnAttrs[column] + count);
	    }
	    for (int column = 0; column < _invisibleColumnAttrs.length;
		 column++) {
		_taskData.removeAttr(_invisibleColumnAttrs[column] + count);
	    }
	    _taskData.setLong(_numRowsAttr, count);

	    // JTable has notification and repaint problems with
	    // deleted rows.  Sun Java bug 4129394.
	    update();
	    _table.repaint();
	}

	/**
	 * Called when the Modify button is pressed.  Copy a row into
	 * the editor and then delete it.
	 *
	 * @param row row to modify.
	 */
	void modifyRow(int row) {
	    for (int column = 0; column < _columnAttrs.length;
		 column++) {
		String attrName = _columnAttrs[column];
		_taskData.setString(attrName,
				    _taskData.getString(attrName + row));
	    }
	    for (int column = 0; column < _invisibleColumnAttrs.length;
		 column++) {
		String attrName = _invisibleColumnAttrs[column];
		Attribute attr = _taskData.getAttr(attrName + row);
		_taskData.setAttr(
		    new Attribute(attrName,
				  attr.getTypeString(), attr.getValueString()));
	    }
	    deleteRow(row);
	}

	/**
	 * Move a row up by one row.
	 *
	 * @param row row to move up.
	 *
	 * @return true if row was moved, false otherwise.
	 */
	boolean moveRowUp(int row) {
	    if (row <= 0) {
		return false;
	    }
	    for (int column = 0; column < _columnAttrs.length; column++) {
		String attrName = _columnAttrs[column];
		String value = _taskData.getString(attrName + row);
		_taskData.setString(
		    attrName + row, _taskData.getString(attrName + (row - 1)));
		_taskData.setString(attrName + (row - 1), value);
	    }
	    for (int column = 0; column < _invisibleColumnAttrs.length;
		 column++) {
		String attrName = _invisibleColumnAttrs[column];
		Attribute temp = _taskData.getAttr(attrName + row);
		Attribute attr = _taskData.getAttr(attrName + (row - 1));
		_taskData.setAttr(new Attribute(attrName + row,
						attr.getTypeString(),
						attr.getValueString()));
		_taskData.setAttr(new Attribute(attrName + (row - 1),
						temp.getTypeString(),
						temp.getValueString()));
	    }

	    fireTableRowsUpdated(row - 1, row);
	    return true;
	}

	/**
	 * Move a row down by one row.
	 *
	 * @param row row to move down.
	 *
	 * @return true if row was moved, false otherwise.
	 */
	boolean moveRowDown(int row) {
	    if (row >= (int)_taskData.getLong(_numRowsAttr) - 1) {
		return false;
	    }
	    for (int column = 0; column < _columnAttrs.length; column++) {
		String attrName = _columnAttrs[column];
		String value = _taskData.getString(attrName + row);
		_taskData.setString(
		    attrName + row, _taskData.getString(attrName + (row + 1)));
		_taskData.setString(attrName + (row + 1), value);
	    }
	    for (int column = 0; column < _invisibleColumnAttrs.length;
		 column++) {
		String attrName = _invisibleColumnAttrs[column];
		Attribute temp = _taskData.getAttr(attrName + row);
		Attribute attr = _taskData.getAttr(attrName + (row + 1));
		_taskData.setAttr(new Attribute(attrName + row,
						attr.getTypeString(),
						attr.getValueString()));
		_taskData.setAttr(new Attribute(attrName + (row + 1),
						temp.getTypeString(),
						temp.getValueString()));
	    }
	    fireTableRowsUpdated(row, row + 1);
	    return true;
	}

	/**
	 * Remove all data from the table.  Clear the editor data.
	 * Leave the invisible editor data alone.
	 */
	void clear() {
	    int count = (int)_taskData.getLong(_numRowsAttr);
	    for (int row = 0; row < count; row++) {
		for (int column = 0; column < _columnAttrs.length;
		     column++) {
		    _taskData.removeAttr(_columnAttrs[column] + row);
		}
		for (int column = 0; column < _invisibleColumnAttrs.length;
		     column++) {
		    _taskData.removeAttr(_invisibleColumnAttrs[column] + row);
		}
	    }
	    _taskData.setLong(_numRowsAttr, 0);
	    for (int column = 0; column < _columnAttrs.length;
		 column++) {
		_taskData.setString(_columnAttrs[column], "");
	    }
	    fireTableDataChanged();
	}

	/**
	 * Redraw all the cells in the table.  This is used to refresh
	 * the table when the task data is changed programmatically.
	 */
	void update() {
	    fireTableDataChanged();
	}
    }

    /**
     * EditVerifier is an interface for added control over the
     * interactions between EditableList and its editor.
     */
    public interface EditVerifier {
	/**
	 * Called when the Add button is pressed.  Notify listener
	 * with succeeded or failed based on whether the add should
	 * continue.
	 *
	 * @param list  The list whose Add button has been pressed.
	 * @param listener listener to notify of result.
	 */
	public void okToAdd(EditableList list, ResultListener listener);

	/**
	 * Called when the Modify button is pressed.  Notify listener
	 * with succeeded or failed based on whether the modify should
	 * continue.
	 *
	 * @param list  The list whose Add button has been pressed.
	 * @param row The row to be modified.
	 * @param listener listener to notify of result.
	 */
	public void okToModify(EditableList list, int row,
			       ResultListener listener);

	/**
	 * Called when the Delete button is pressed.  Notify listener
	 * with succeeded or failed based on whether the Delete should
	 * continue.
	 *
	 * @param list  The list whose Add button has been pressed.
	 * @param row  The row to be deleted.
	 * @param listener listener to notify of result.
	 */
	public void okToDelete(EditableList list, int row,
			       ResultListener listener);

	/**
	 * Called to reset the editor UI after an Add.
	 *
	 * @param The list whose editor should be reset.
	 */
	public void resetComponents(EditableList list);

	/**
	 * Called to reset the list when the Reset button is pressed.
	 *
	 * @param list The list to reset.
	 */
	public void resetList(EditableList list);

	/**
	 * Called by EditableList.changedSinceLastAdd() to determine
	 * whether the user had made any changes.
	 *
	 * @param list The list to check for changes.
	 */
	public boolean changedSinceLastAdd(EditableList list);
    }

    /**
     * DefaultEditVerifier implements the default behavior when the
     * version of <tt>EditableList.setEditor</tt> that does not take an
     * EditVerifier argument is used.
     * <p>
     * For each of the asynchronous methods <tt>okToAdd</tt>,
     * <tt>okToModify</tt>, and <tt>okToDelete</tt>,
     * DefaultEditVerifier implements synchronous methods which return
     * a <tt>boolean</tt> rather than taking a ResultListener.  The
     * asynchronous versions are all implemented in terms of their
     * synchronous counterparts.  For example, here is the
     * implementation of the asynchronous version of <tt>okToAdd</tt>:
     * <pre>
     *ResultEvent event = new ResultEvent(list);
     *if (okToAdd(list)) {
     *    listener.succeeded(event);
     *} else {
     *    listener.failed(event);
     *}
     * </pre>
     * It is typically more convenient to extend DefaultEditVerifier
     * than to implement EditVerifier because DefaultEditVerifier
     * provides reasonable default implementations for each of the
     * EditVerifier methods and if your <tt>okToAdd</tt>,
     * <tt>okToModify</tt>, and <tt>okToDelete</tt> methods can be
     * performed synchronously you can implement the synchronous
     * versions instead of the asynchronous versions.
     */
    public static class DefaultEditVerifier implements EditVerifier {

	/**
	 * Determine whether it's ok to let an "Add" operation proceed.
	 * This version notifies <tt>listener</tt> of the result of
	 * the  the synchronous <tt>okToAdd</tt> method.
	 *
	 * @param list The list whose Add button has been pressed.
	 * @param listener listener to notify of result.
	 */
	public void okToAdd(EditableList list, ResultListener listener) {
	    ResultEvent event = new ResultEvent(list);
	    if (okToAdd(list)) {
		listener.succeeded(event);
	    } else {
		listener.failed(event);
	    }
	}

	/**
	 * This synchronous version of <tt>okToAdd</tt> is called by the
	 * asynchronous version.  If all Add verification can be
	 * performed synchronously, this method can be overridden
	 * instead of the asynchronous version.
	 * <p>
	 * <tt>DefaultEditVerifier.okToAdd</tt> returns <tt>true</tt>
	 * if any columns have Strings in them other than "".
	 *
	 * @param list The list whose Add button has been pressed.
	 *
	 * @return true if add should happen, false otherwise.
	 */
	public boolean okToAdd(EditableList list) {
	    for (int column = 0; column < list._columnAttrs.length;
		 column++) {
		if (list._taskData.getString(
		    list._columnAttrs[column]).length() > 0) {
		    return true;
		}
	    }
	    Toolkit.getDefaultToolkit().beep();
	    return false;
	}

	/**
	 * Checks to see if there is already a row in the list has the
	 * same values for each column as the values in the editor.
	 * This can be called from within <tt>okToAdd</tt> to
	 * determine whether the values in the editor are different
	 * from any rows already present.
	 *
	 * @param list Editable list to check.
	 *
	 * @return <tt>true</tt> if a row already exists in <tt>list</tt>
	 *         having the same values as the editor,
	 *         <tt>false</tt> otherwise.
	 */
	public static boolean alreadyExists(EditableList list) {
	    // Description of algorithm
	    // For each row in the list, if there is even 1 cell in the
	    // list's row that does not match the corresponding editor
	    // value, break and go to next row. Else, if the whole row
	    // matches, return true.

	    final String[] columnAttrs = list._columnAttrs; //alias
	    final TaskData taskData = list._taskData; //alias
	    int row = 0;
	    int column = 0;
	    int numCols = columnAttrs.length;
	    int numRows = (int)list._taskData.getLong(list._numRowsAttr);
	    while (row < numRows) {
		for (column = 0; column < numCols;
		     column++) {
		    if (!taskData.getString(columnAttrs[column]).
			equals(taskData.
			       getString(columnAttrs[column] + row))) {
			break;
		    }
		}
		// if this condition is met, then we never broke out
		// of the for loop. This means that the whole row
		// matches what is in the editor.
		if (column == numCols) {
		    return true;
		}
		row++;
	    }

	    return false;
	}

	/**
	 * Determine whether it's ok to let a "Modify" operation proceed.
	 * This version notifies <tt>listener</tt> of the result of
	 * the the synchronous <tt>okToModify</tt> method.
	 *
	 * @param list The list whose Modify button has been pressed.
	 * @param row The row to modify.
	 * @param listener listener to notify of result.
	 */
	public void okToModify(EditableList list, int row,
			       ResultListener listener) {
	    ResultEvent event = new ResultEvent(list);
	    if (okToModify(list, row)) {
		listener.succeeded(event);
	    } else {
		listener.failed(event);
	    }
	}

	/**
	 * Called when the Modify button is pressed.  Return <tt>true</tt> if
	 * the modify should continue, <tt>false</tt> to stop it.
	 * <tt>DefaultEditVerifier.okToModify</tt> always returns <tt>true</tt>.
	 *
	 * @param list The list whose Modify button has been pressed.
	 * @param row The row that is being modified.
	 *
	 * @return <tt>true</tt> if the modify should happen,
	 *         <tt>false</tt> otherwise.
	 */
	public boolean okToModify(EditableList list, int row) {
	    return true;
	}

	/**
	 * Determine whether it's ok to let a "Delete" operation proceed.
	 * This version notifies <tt>listener</tt> of the result of
	 * the the synchronous <tt>okToDelete</tt> method.
	 *
	 * @param list The list whose Delete button has been pressed.
	 * @param row The row to delete.
	 * @param listener listener to notify of result.
	 */
	public void okToDelete(EditableList list, int row,
			       ResultListener listener) {
	    ResultEvent event = new ResultEvent(list);
	    if (okToDelete(list, row)) {
		listener.succeeded(event);
	    } else {
		listener.failed(event);
	    }
	}

	/**
	 * Called when the Delete button is pressed.  Return <tt>true</tt> if
	 * the delete should continue, <tt>false</tt> to stop it.
	 * <tt>DefaultEditVerifier.okToDelete</tt> always returns <tt>true</tt>.
	 *
	 * @param list The list whose Delete button has been pressed.
	 * @param row The row that is to be deleted.
	 *
	 * @return <tt>true</tt> if the delete should happen,
	 *         <tt>false</tt> otherwise.
	 */
	public boolean okToDelete(EditableList list, int row) {
	    return true;
	}

	/**
	 * Called to reset the editor UI after an Add.  Sets the
	 * attribute for each column to "".
	 *
	 * @param The list whose editor should be reset.
	 */
	public void resetComponents(EditableList list) {
	    for (int column = 0; column < list._columnAttrs.length;
		 column++) {
		String attrName = list._columnAttrs[column];
		list._taskData.setString(attrName, "");
	    }
	}

	/**
	 * Called to reset the list when the Reset button is pressed.
	 * DefaultEditVerifier method does nothing.
	 *
	 * @param list The list to reset.
	 */
	public void resetList(EditableList list) {
	}

	/**
	 * Called to determine whether user has made any changes.
	 * DefaultEditVerifier method returns true if any columns have
	 * Strings in them other than "".
	 *
	 * @param list The list to check for changes.
	 *
	 * @return true if changes have been made.
	 */
	public boolean changedSinceLastAdd(EditableList list) {
	    for (int column = 0; column < list._columnAttrs.length;
		 column++) {
		String attrName = list._columnAttrs[column];
		if (list._taskData.getString(attrName).length() > 0) {
		    return true;
		}
	    }
	    return false;
	}
    }

    /**
     * Cell is the constraint class for the LayoutManager returned by
     * the <tt>getEditorLayout</tt> method.  A Cell constraint
     * specified in a call to addComponent() of the Container that
     * uses the LayoutManager returned by <tt>getEditorLayout</tt> can
     * be used to specify the row and column to display that Component
     * in.
     * @see EditableList#getEditorLayout
     */
    public static class Cell {

	/**
	 * Row of cell in which to place the Component associated with
	 * this constraint.
	 */
	int _row;

	/**
	 * Column of cell in which to place the Component associated
	 * with this constraint.
	 */
	int _column;

	/**
	 * Horizontal alignment of this cell.
	 */
	int _horizAlignment;

	/**
	 * Vertical alignment of this cell.
	 */
	int _vertAlignment;

	/**
	 * Construct a Cell.
	 *
	 * @param row row to place component.
	 * @param column column to place component.
	 * @param horizAlignment Horizontal alignment of this cell.
	 *        Use LEFT, CENTER, or RIGHT values from SwingConstants.
	 * @param vertAlignment Vertical alignment of this cell.
	 *        Use BOTTOM, CENTER, or TOP values from SwingConstants.
	 */
	public Cell(int row, int column, int horizAlignment,
		    int vertAlignment) {
	    _row = row;
	    _column = column;
	    _horizAlignment = horizAlignment;
	    _vertAlignment = vertAlignment;
	}

	/**
	 * Construct a Cell.
	 *
	 * @param row row to place component.
	 * @param column column to place component.
	 */
	public Cell(int row, int column) {
	    this(row, column, LEFT, CENTER);
	}

	/**
	 * Construct a cell specifying placement at row 0 and column 0.
	 */
	public Cell() {
	    this(0, 0);
	}
    }

    /**
     * EditorLayout is meant to be used by the editor associated with
     * an EditableList.  EditorLayout keeps columns of Components
     * lined up with the columns in the list being edited.
     * <p>
     * By default, components are laid out from left to right in the
     * order in which they are added.  When the number of components
     * exceeds the number of columns, a new row is started.
     * <p>
     * More fine-grained control over the placement of components may
     * be acheived by specifying constraints of type Cell.
     */
    private class EditorLayout implements LayoutManager2 {
	private Hashtable _constraintMap = new Hashtable();
	private int _numRows;

	/**
	 * Package-protected constructor hides from javadoc.
	 */
	EditorLayout() {
	}

	/**
	 * Add a component to the layout.
	 *
	 * @param comp Component to add
	 * @param constraints if non-null, the Cell in which to place
	 *                    <i>comp</i>.
	 */
	public void addLayoutComponent(Component comp, Object constraints) {
	    Log.assert(constraints == null || constraints instanceof Cell,
		       "constraints must be an EditorLayout.Cell");
	    if (constraints != null) {
		_constraintMap.put(comp, constraints);
		Cell cell = (Cell)constraints;
		if (cell._row + 1 > _numRows) {
		    _numRows = cell._row + 1;
		}
	    }
	}

	/**
	 * @see java.awt.LayoutManager2#maximumLayoutSize
	 */
	public Dimension maximumLayoutSize(Container target) {
	    return new Dimension(Short.MAX_VALUE, Short.MAX_VALUE);
	}

	/**
	 * @see java.awt.LayoutManager2#getLayoutAlignmentX
	 */
	public float getLayoutAlignmentX(Container target) {
	    return 0;
	}

	/**
	 * @see java.awt.LayoutManager2#getLayoutAlignmentY
	 */
	public float getLayoutAlignmentY(Container target) {
	    return 0;
	}

	/**
	 * @see java.awt.LayoutManager#addLayoutComponent
	 */
	public void addLayoutComponent(String name, Component comp) {
	    addLayoutComponent(comp, null);
	}

	/**
	 * @see java.awt.LayoutManager#removeLayoutComponent
	 */
	public void removeLayoutComponent(Component comp) {
	    _constraintMap.remove(comp);
	}

	/**
	 * @see java.awt.LayoutManager#minimumLayoutSize
	 */
	public Dimension minimumLayoutSize(Container parent) {
	    return new Dimension(1, 1);
	}

	/**
	 * @see java.awt.LayoutManager2#invalidateLayout
	 */
	public void invalidateLayout(Container target) {
	}

	/**
	 * @see java.awt.LayoutManager#preferredLayoutSize
	 */
	public Dimension preferredLayoutSize(Container parent) {
	    int rowHeights[] = getRowHeights(parent);
	    int height = 0;
	    for (int ii = 0; ii < rowHeights.length; ii++) {
		height += rowHeights[ii];
		if (ii != rowHeights.length - 1) {
		    height += _vGap;
		}
	    }
	    return new Dimension(_scroll.getSize().width, height);
	}

	/**
	 * Layout the children of <i>parent</i> in a grid with columns
	 * that line up with the columns of our list.
	 *
	 * @param parent Container whose children we are to lay out.
	 */
	public void layoutContainer(Container parent) {
	    int rowHeights[] = getRowHeights(parent);
	    int rowOffsets[] = new int[rowHeights.length];
	    int offset = 0;
	    for (int ii = 0; ii < rowHeights.length - 1; ii++) {
		rowOffsets[ii] = offset;
		offset = rowHeights[ii] + _vGap;
	    }
	    rowOffsets[rowHeights.length - 1] = offset;

	    int width = _scroll.getSize().width;
	    int numColumns = _table.getColumnCount();

	    int cellPad = ((numColumns - 1) * _hGap) / numColumns;

	    offset = 0;
	    int columnOffsets[] = new int[numColumns + 1];
	    for (int ii = 0; ii < numColumns - 1; ii++) {
		columnOffsets[ii] = offset;
		TableColumn tableColumn = _table.getColumnModel().getColumn(ii);
		offset += tableColumn.getWidth() - cellPad + _hGap;
	    }
	    columnOffsets[numColumns - 1] = offset;
	    // A fake column at the end to simplify centering and
	    // right-justification logic below.
	    columnOffsets[numColumns] = _table.getSize().width;

	    Component kids[] = parent.getComponents();
	    for (int ii = 0; ii < kids.length; ii++) {
		Component kid = kids[ii];
		Cell cell = (Cell)_constraintMap.get(kid);
		if (cell == null) {
		    int row, column;
		    if (_reorderable) {
			row = ii / (numColumns - 1);
			column = 1 + ii % (numColumns - 1);
		    } else {
			row = ii / numColumns;
			column = ii % numColumns;
		    }
		    String hAlignStr = _rs.getString(createLookup(
			HORIZONTAL_ALIGNMENT + "." + row + "," + column),
						     "LEFT");
		    String vAlignStr = _rs.getString(createLookup(
			VERTICAL_ALIGNMENT + "." + row + "," + column),
						     "CENTER");
		    int hAlign = ResourceStack.mapStringToByte(
			hAlignStr, new String[] { "LEFT", "CENTER",
						  "RIGHT" },
			new byte[] { (byte)LEFT,
				     (byte)CENTER,
				     (byte)RIGHT });
		    int vAlign = ResourceStack.mapStringToByte(
			vAlignStr, new String[] { "BOTTOM", "CENTER",
						  "TOP" },
			new byte[] { (byte)BOTTOM,
				     (byte)CENTER,
				     (byte)TOP });
		    cell = new Cell(row, column, hAlign, vAlign);
		    _constraintMap.put(kid, cell);
		}
		Dimension size = kid.getPreferredSize();

		int x, y;
		switch (cell._horizAlignment) {
		case LEFT:
		    x = columnOffsets[cell._column];
		    break;
		case CENTER:
		default:
		    x = columnOffsets[cell._column]
			+ (columnOffsets[cell._column + 1] -
			   columnOffsets[cell._column]  - size.width) / 2;
		    break;
		case RIGHT:
		    x = columnOffsets[cell._column + 1] - size.width;
		    break;
		}
		switch (cell._vertAlignment) {
		case TOP:
		    y = rowOffsets[cell._row];
		    break;
		case CENTER:
		default:
		    y = rowOffsets[cell._row]
			+ (rowHeights[cell._row] - size.height) / 2;
		    break;
		case BOTTOM:
		    y = rowOffsets[cell._row]
			+ rowHeights[cell._row] - size.height;
		    break;
		}
		kid.setLocation(x, y);

		TableColumn tableColumn = _table.getColumnModel().
		    getColumn(cell._column);

		if (size.width > tableColumn.getWidth() - cellPad) {
		    size.width = tableColumn.getWidth() - cellPad;
		}
		kid.setSize(size);
	    }
	}

	/**
	 * Get the height of each row.
	 *
	 * @param parent Container we are going to layout.
	 *
	 * @return An array of row heights.
	 */
	private int[] getRowHeights(Container parent) {
	    Component kids[] = parent.getComponents();

	    int numColumns = _table.getColumnCount();
	    if (_reorderable) {
		numColumns--;
	    }
	    int numRows = _numRows;
	    if (numRows == 0) {
		numRows = (kids.length + numColumns - 1) / numColumns;
	    }

	    int rowHeights[] = new int[numRows];
	    int column = 0;
	    for (int ii = 0; ii < kids.length; ii++) {
		Component kid = kids[ii];
		Cell cell = (Cell)_constraintMap.get(kid);
		int row;
		if (cell != null) {
		    row = cell._row;
		} else {
		    row = ii / numColumns;
		}
		int height = kid.getPreferredSize().height;
		if (height > rowHeights[row]) {
		    rowHeights[row] = height;
		}
	    }
	    return rowHeights;
	}
    }

    /**
     * Layout manager for the EditableList Panel.
     */
    private class ListLayout implements LayoutManager {
	public void addLayoutComponent(String name, Component comp) {
	}

	public void removeLayoutComponent(Component comp) {
	}

	/**
	 * Calculate how big the EditableList should be.  Note that we
	 * use our EditableList instance to refer to the Components
	 * we're laying out rather than using parent.getComponents().
	 *
	 * @param parent EditableList we are going to lay out.
	 *
	 * @return Preferred size of EditableList.
	 */
	public Dimension preferredLayoutSize(Container parent) {
	    int width = 0;
	    int height = 0;
	    Dimension size = null;
	    if (_editorComponent != null) {
		size = _editorComponent.getPreferredSize();
		width = size.width;
		height = size.height + _editorVGap;
	    }

	    int buttonWidth = 0;
	    int buttonHeight = 0;
	    int numButtons = 0;
	    if (_addButton != null) {
		size = _addButton.getPreferredSize();
		buttonWidth = size.width;
		buttonHeight = size.height;
		// If the "Add" button is taller than the editor,
		// allocate some extra space for it.
		if (buttonHeight > height) {
		    height = buttonHeight + _vGap;
		}
	    }

	    Component buttons[] = new Component[] { _modifyButton,
						    _deleteButton,
						    _upButton,
						    _downButton,
						    _resetButton };
	    for (int ii = 0; ii < buttons.length; ii++) {
		if (buttons[ii] == null) {
		    continue;
		}
		size = buttons[ii].getPreferredSize();
		if (size.width > buttonWidth) {
		    buttonWidth = size.width;
		}
		if (size.height > buttonHeight) {
		    buttonHeight = size.height;
		}
		numButtons++;
	    }
	    int totalButtonHeight = numButtons > 0 ?
		numButtons * (buttonHeight + _vGap) - _vGap : 0;

	    size = _scroll.getPreferredSize();
	    if (size.width > width) {
		width = size.width;
	    }

	    if (size.height > totalButtonHeight) {
		height += size.height;
	    } else {
		height += totalButtonHeight;
	    }

	    width += _hGap + buttonWidth;

	    Insets insets = parent.getInsets();
	    width += insets.left + insets.right + _hMargin * 2;
	    height += insets.top + insets.bottom + _vMargin * 2;

	    return new Dimension(width, height);
	}

	public Dimension minimumLayoutSize(Container parent) {
	    return new Dimension(1, 1);
	}

	/**
	 * Lay out an EditableList.  Note that we use our EditableList
	 * instance to refer to the Components we're laying out rather
	 * than using parent.getComponents().
	 *
	 * @param parent EditableList to lay out.
	 */
	public void layoutContainer(Container parent) {
	    Dimension parentSize = parent.getSize();

	    // Don't bother doing a layout if no one will see it.
	    // Doing the layout will cause problems because it will
	    // mess up preferred sizes.
	    if (parentSize.width == 0 || parentSize.height == 0) {
		return;
	    }

	    Dimension size;
	    int buttonWidth = 0;
	    int buttonHeight = 0;
	    Component buttons[] = new Component[] { _addButton,
						    _modifyButton,
						    _deleteButton,
						    _upButton,
						    _downButton,
						    _resetButton };
	    for (int ii = 0; ii < buttons.length; ii++) {
		if (buttons[ii] == null) {
		    continue;
		}
		size = buttons[ii].getPreferredSize();
		if (size.width > buttonWidth) {
		    buttonWidth = size.width;
		}
		if (size.height > buttonHeight) {
		    buttonHeight = size.height;
		}
	    }

	    for (int ii = 0; ii < buttons.length; ii++) {
		if (buttons[ii] == null) {
		    continue;
		}
		buttons[ii].setSize(buttonWidth, buttonHeight);
	    }

	    Insets insets = parent.getInsets();
	    int x = insets.left + _hMargin;
	    int y = insets.top + _vMargin;
	    int editorHeight = 0;
	    int editorOffset = 0;
	    int addButtonOffset = 0;

	    // At this point, we only care about the editor's height.
	    // Its width isn't meaningful yet, because that depends
	    // on the width of _scroll which we have not yet determined.
	    if (_editorComponent != null) {
		size = _editorComponent.getPreferredSize();
		editorHeight = size.height;
	    }

	    if (_editorComponent != null && _addButton != null) {
		// Line the bottom of the Add button up with the
		// bottom of the editor.
		if (editorHeight > buttonHeight) {
		    addButtonOffset = editorHeight - buttonHeight;
		} else {
		    editorOffset = (buttonHeight - editorHeight) / 2;
		}
	    }

	    int buttonLeft = parentSize.width - insets.right
		- _hMargin - buttonWidth;

	    int editorGap;
	    if (_editorComponent != null) {
		_editorComponent.setLocation(x, y + editorOffset);
		editorGap = _editorVGap;
	    } else {
		editorGap = _vGap;
	    }
	    if (_addButton != null) {
		_addButton.setLocation(buttonLeft, y + addButtonOffset);
		// This is inside the if statement because
		// buttonHeight might be > 0 even if _addButton is
		// null.
		y += Math.max(buttonHeight, editorHeight) + editorGap;
	    } else {
		y += editorHeight + editorGap;
	    }

            // The ScrollPane is actually in a JPanel, to work around
            // a Swing bug.  So we need to set the location and size
            // of the panel, not the ScrollPane.
	    _scroll.getParent().setLocation(x, y);
	    _scroll.getParent().setSize(parentSize.width - insets.left -
                                        insets.right - _hMargin * 2 - _hGap - buttonWidth,
                                        parentSize.height - y - insets.bottom - _vMargin);

	    // _scroll.validate() is necessary to make _table figure
	    // out its new column widths in time for EditorLayout to
	    // use them.
            _scroll.getParent().validate();
	    _table.sizeColumnsToFit(false);

	    // Now that we're done with _scroll, we can finish up with
	    // _editorComponent.
	    if (_editorComponent != null) {
		_editorComponent.invalidate();
		size = _editorComponent.getPreferredSize();
		_editorComponent.setSize(size);
	    }

	    if (_modifyButton != null) {
		_modifyButton.setLocation(buttonLeft, y);
		y += buttonHeight + _vGap;
	    }

	    if (_upButton != null) {
		_upButton.setLocation(buttonLeft, y);
		y += buttonHeight + _vGap;
	    }

	    if (_downButton != null) {
		_downButton.setLocation(buttonLeft, y);
	    }

	    y = parentSize.height - _vMargin - buttonHeight - insets.bottom;

	    if (_resetButton != null) {
		_resetButton.setLocation(buttonLeft, y);
		y -= buttonHeight + _vGap;
	    }

	    _deleteButton.setLocation(buttonLeft, y);
	}
    }

    /**
     * Construct an EditableList.
     *
     * @param componentName Name of this component for resource
     *                      purposes.
     * @param rs Used to get resource settings that control the
     *           appearance and behavior of this EditableList.
     * @param taskData TaskData to be used for storing the information
     *                 in the table.
     */
    public EditableList(String componentName, ResourceStack rs,
			TaskData taskData) {
	setLayout(new ListLayout());
	_taskData = taskData;
	_rs = rs;
	_name = componentName;

	_hGap = _rs.getPixels(createLookup(HGAP));
	_vGap = _rs.getPixels(createLookup(VGAP));
	_hMargin =  _rs.getPixels(createLookup(HMARGIN));
	_vMargin = _rs.getPixels(createLookup(VMARGIN));
	_editorVGap = _rs.getPixels(createLookup(EDITOR_VGAP));
	_numberFormatString = _rs.getString(createLookup(NUMBER_FORMAT_STRING));
	_reorderable = _rs.getBoolean(createLookup(REORDERABLE));
	_showModifyButton =  _rs.getBoolean(createLookup(SHOW_MODIFY_BUTTON));
	_showResetButton = _rs.getBoolean(createLookup(SHOW_RESET_BUTTON));
	_changedAttrName =
	    _rs.getString(createLookup(CHANGED_SINCE_LAST_ADD_ATTR_NAME), null);
	// Initialize it so client doesn't have to.  Otherwise, if
	// we're created but never shown and the client tried to check
	// this value the client would get an exception
	if (_changedAttrName != null) {
	    _taskData.setBoolean(_changedAttrName, false);
	}
    }

    /**
     * Called to set the Component to be used to edit a row in this
     * EditableTable.
     *
     * @param editorComponent Component to be used to edit a row.
     */
    public void setEditor(Component editorComponent) {
	setEditor(editorComponent, new DefaultEditVerifier());
    }

    /**
     * Called to set the Component to be used to edit a row in this
     * EditableTable.  Also specifies an EditVerifier.
     *
     * @param editorComponent The editor component.
     * @param verifier Verifies edits.
     */
    public void setEditor(Component editorComponent, EditVerifier verifier) {
	Log.assert(!_added, "Attempt to set editor after add to parent");
	Log.assert(_editorComponent == null, "Attempt to add multiple editors");
	Log.assert(verifier != null, "Attempt to add null verifier");
	Log.assert(editorComponent != null,
		   "Attempt to add null editorComponent");
	_editorComponent = editorComponent;
	_verifier = verifier;
    }

    /**
     * Determine whether the user has changed anything in the editor
     * since the last time the "Add" button was pressed.  The
     * <i>CHANGED_SINCE_LAST_ADD_ATTR_NAME</i> property can also be
     * used to determine whether the editor has changed.
     *
     * @return true if changes have been made, false otherwise.
     * @see #CHANGED_SINCE_LAST_ADD_ATTR_NAME
     */
    public boolean changedSinceLastAdd() {
	return _verifier.changedSinceLastAdd(this);
    }

    /**
     * Set up the columns of this EditableList.  Calling this method
     * overrides settings of the <i>COLUMN_ATTR_NAME</i> and
     * <i>INVISIBLE_COLUMN_ATTR_NAME</i> resources.
     *
     * @param numRowsAttr name of long Attribute to be used to keep
     *                     track of the number of columns.
     * @param columnAttrs array of Attribute names, one for each column.
     * @param headings array of Strings to be displayed as headings in
     *                 the table.  These strings should be localized
     *                 by the caller.
     * @see #COLUMN_ATTR_NAME
     * @see #INVISIBLE_COLUMN_ATTR_NAME
     */
    public void setColumns(String numRowsAttr,
			   String columnAttrs[], String headings[]) {
	setColumns(numRowsAttr, columnAttrs, headings,
		   new String[0]);
	_invisibleColumnAttrs = new String[0];
    }

    /**
     * Set up the columns of this EditableList, including invisible
     * columns.
     *
     * @param numRowsAttr name of long Attribute to be used to keep
     *                     track of the number of columns.
     * @param columnAttrs array of Attribute names, one for each column.
     * @param headings array of Strings to be displayed as headings in
     *                 the table.  These strings should be localized
     *                 by the caller.
     * @param invisibleColumnAttrs array of Attribute names, one for
     *                             each invisible column.
     */
    public void setColumns(String numRowsAttr,
			   String columnAttrs[], String headings[],
			   String invisibleColumnAttrs[]) {
	Log.assert(!_added, "Attempt to set columns after add to parent");
	Log.assert(numRowsAttr != null, "Attempt to add null numRowsAttr");
	Log.assert(columnAttrs != null, "Attempt to add null columnAttrs");
	Log.assert(headings == null || columnAttrs.length == headings.length,
		   "Number of headings does not match number of attr names");
	Log.assert(invisibleColumnAttrs != null,
		   "Attempt to add null invisibleColumnAttrs");
	_numRowsAttr = numRowsAttr;
	_columnAttrs = columnAttrs;
	_headings = headings;
	_invisibleColumnAttrs = invisibleColumnAttrs;

	// Initialize if there's nothing there yet.
	if (_taskData.getAttr(_numRowsAttr) == null) {
	    _taskData.setLong(_numRowsAttr, 0);
	}
    }

    /**
     * Set whether the list should be reorderable.  If the list is
     * reorderable, the rows will be numbered and buttons will be
     * provided for moving entries up and down.
     *
     * @param reorderable true if the list should be reorderable.
     */
    public void setReorderable(boolean reorderable) {
	Log.assert(!_added, "Attempt to set reorderable after add to parent");
	_reorderable = reorderable;
    }

    /**
     * Get a LayoutManager that can be used with the editor of this
     * EditableList.  <tt>getEditorLayout</tt> returns a LayoutManager
     * that keeps columns of Components lined up with the columns in
     * this list.
     * <p>
     * By default, components are laid out from left to right in the
     * order in which they are added.  When the number of Components
     * exceeds the number of columns, a new row is started.
     * <p>
     * More fine-grained control over the placement of components can
     * be achieved by specifying constraints of type Cell when adding
     * Components to a Container that uses the LayoutManager returned
     * by <tt>getEditorLayout</tt>.
     *
     * @return LayoutManager associated with this EditableList
     * @see Cell
     */
    public LayoutManager getEditorLayout() {
	return new EditorLayout();
    }

    /**
     * Removes all the row data from the list.  Clears the strings in
     * the editor task data.  Does not do anything to the invisible
     * editor task data.
     */
    public void clearList() {
	ListTableModel model = (ListTableModel)_table.getModel();
	model.clear();
    }

    /**
     * Update the information displayed in the list.  This should be
     * called after the task data is changed.
     */
    public void updateList() {
	ListTableModel model = (ListTableModel)_table.getModel();
	model.update();
    }

    /**
     * Called when we're added to a Container.  Build our UI.
     */
    public void addNotify() {
	super.addNotify();
	if (_added) {
	    return;
	}
	_added = true;
	setBorder(BorderFactory.createEtchedBorder());
	if (_editorComponent != null) {
	    this.add(_editorComponent);
	    _addButton = new JButton(
		_rs.getString(createLookup(ADD_BUTTON_LABEL)));
	    this.add(_addButton);
	}

	if (_columnAttrs == null) {
	    initColumns();
	}

	monitorEditor();

	final ListTableModel model = new ListTableModel();
	_table = new ListTable(model);
	_table.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
	_table.getTableHeader().setReorderingAllowed(false);
	_table.setShowVerticalLines(false);
	_table.setShowHorizontalLines(false);
	FontMetrics fm = Toolkit.getDefaultToolkit().
	    getFontMetrics(_table.getFont());
	_table.setRowHeight(fm.getHeight() + fm.getLeading());

	int totalWidth = 0;
	int margin = _table.getColumnModel().getColumnMargin();

	TableColumnModel columnModel = _table.getColumnModel();
	if (_reorderable) {
	    TableColumn tableColumn = columnModel.getColumn(0);
	    int width = _rs.getPixels(createLookup(NUMBER_COLUMN_WIDTH));
	    tableColumn.setWidth(width);
	    tableColumn.setResizable(false);
	    totalWidth += width + margin;
	}

	int columnCount = _table.getColumnCount();
	if (_reorderable) {
	    columnCount--;
	}
	int resizableColumns = columnCount;
	for (int column = 0; column < columnCount; column++) {
	    try {
		TableColumn tableColumn =
		    columnModel.getColumn(_reorderable ? column + 1 : column);

		// Left-align all the column headers.
		try {
		    DefaultTableCellRenderer headerRenderer =
			(DefaultTableCellRenderer)tableColumn.
			    getHeaderRenderer();
		    headerRenderer.setHorizontalAlignment(JLabel.LEFT);
		} catch (ClassCastException ex) {
		}

		int width = _rs.getPixels(createLookup(COLUMN_WIDTH + column));
		tableColumn.setWidth(width);
		tableColumn.setResizable(false);
		totalWidth += width + margin;
		resizableColumns--;
	    } catch (MissingResourceException ex) {
	    }
	}

	// Set preferred size to the sum of columns widths + margins
	// if there are no resizable columns.  Otherwise use LIST_WIDTH.
	_table.setPreferredScrollableViewportSize(
	    new Dimension(resizableColumns > 0 ?
			  _rs.getPixels(createLookup(LIST_WIDTH)) : totalWidth,
			  _rs.getPixels(createLookup(LIST_HEIGHT))));

	_scroll = new JScrollPane(_table);
	_scroll.setHorizontalScrollBarPolicy(
	    ScrollPaneConstants.HORIZONTAL_SCROLLBAR_NEVER);

        // Put the ScrollPane on a JPanel with a solid color backgound
        // to work around a Swing bug which prevents us from setting
        // the Viewport's background any other way.
        JPanel solidColorPanel = new JPanel();
        solidColorPanel.setBackground(
            (Color)UIManager.getLookAndFeelDefaults().get("Table.background"));
        solidColorPanel.setLayout( new BorderLayout() );
        solidColorPanel.add(_scroll, BorderLayout.CENTER);
	this.add(solidColorPanel);

	if (_editorComponent != null) {
	    Log.assert(_addButton != null,
		       "Somehow _addButton is null");
	    _addButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent event) {
		    _verifier.okToAdd(EditableList.this, new ResultListener() {
			public void succeeded(ResultEvent ev) {
			    model.addRow();
			    _editorComponent.requestFocus();
                            //when row is added to table, the selection goes  
                            //away so we should disable delete, up, down
                            //and modify buttons	
			    _deleteButton.setEnabled(false);
                            if(_upButton != null) {
                                _upButton.setEnabled(false);
                            }
                            if (_downButton != null) {
                                _downButton.setEnabled(false);
                            }
                            if (_modifyButton != null) {
                                _modifyButton.setEnabled(false);
                            }
			}
			public void failed(ResultEvent ev) {
			}
		    });
		}
	    });
	    _deleteButton = new JButton(
		_rs.getString(createLookup(DELETE_BUTTON_LABEL)));
	    this.add(_deleteButton);    
	    _deleteButton.setEnabled(false);
 
	    _deleteButton.addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent event) {
		    final int row = _table.getSelectedRow();
		    if (row == -1) {
			return;
		    }
		    _verifier.okToDelete(EditableList.this, row,
					 new ResultListener() {
			public void succeeded(ResultEvent ev) {
			    model.deleteRow(row);
                            //when row is deleted, there is no row 
                            //selected so we should disable up,down
                            //and modify button
			    if(_upButton != null) { 
                                _upButton.setEnabled(false);
                            }
	                    if (_downButton != null) {
                                _downButton.setEnabled(false);
                            }
                            if (_modifyButton != null) {
                                _modifyButton.setEnabled(false);
                            }
			}
			public void failed(ResultEvent ev) {
			}
		    });
		}
	    });

	    _table.getSelectionModel().addListSelectionListener(
 		new ListSelectionListener() {
    		    public void valueChanged(ListSelectionEvent ev) {
        		_deleteButton.setEnabled(_table.getSelectedRow() >= 0);
                        if (_modifyButton != null) {
                            _modifyButton.setEnabled
                                    (_table.getSelectedRow() >= 0);
                        }
			if(_upButton != null) {
                             _upButton.setEnabled(_table.getSelectedRow() > 0);
                        }
			if (_downButton != null) {
                            _downButton.setEnabled(_table.getSelectedRow() <
                                                   (_table.getRowCount() - 1));
                        }
                    }
	        });

	    if (_showModifyButton) {
		_modifyButton
		    = new JButton(
			_rs.getString(createLookup(MODIFY_BUTTON_LABEL)));
		this.add(_modifyButton);
                _modifyButton.setEnabled(false);

		_modifyButton.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent event) {
			final int row = _table.getSelectedRow();
			if (row == -1) {
			    return;
			}
			_verifier.okToModify(EditableList.this, row,
					     new ResultListener() {
			    public void succeeded(ResultEvent ev) {
				model.modifyRow(row);
				_editorComponent.requestFocus();
                                //when row is modified, there is no row 
                                //selected so we should disable up,down
                                //and delete button
                                if(_upButton != null) {
                                    _upButton.setEnabled(false);
                                }
	                        if (_downButton != null) {
                                    _downButton.setEnabled(false);
                                }
                                _deleteButton.setEnabled(false);
			    }
			    public void failed(ResultEvent ev) {
			    }
			});
		    }
		});
	    }

	    if (_showResetButton) {
		_resetButton = new JButton(
		    _rs.getString(createLookup(RESET_BUTTON_LABEL)));
		this.add(_resetButton);
		_resetButton.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent event) {
			_verifier.resetList(EditableList.this);
		    }
		});
	    }

	    if (_reorderable) {
		_upButton = new JButton(new ArrowIcon(_rs, ArrowIcon.UP));
		this.add(_upButton);
                _upButton.setEnabled(false);
		_upButton.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent event) {
			int row = _table.getSelectedRow();
			if (row != -1 && model.moveRowUp(row)) {
			    ListSelectionModel select =
				_table.getSelectionModel();
			    select.clearSelection();
			    select.setSelectionInterval(row - 1, row - 1);
			}
		    }
		});

		_downButton = new JButton(new ArrowIcon(_rs, ArrowIcon.DOWN));
		this.add(_downButton);
                _downButton.setEnabled(false);
		_downButton.addActionListener(new ActionListener() {
		    public void actionPerformed(ActionEvent event) {
			int row = _table.getSelectedRow();
			if (row != -1 && model.moveRowDown(row)) {
			    ListSelectionModel select =
				_table.getSelectionModel();
			    select.clearSelection();
			    select.setSelectionInterval(row + 1, row + 1);
			}
		    }
		});
	    }
	}
    }

    private String[] createLookup(String resource) {
	return new String[] {_name + "." + resource,
				 "EditableList." + resource};
    }

    /**
     * Initialize the columns of our table.
     */
    private void initColumns() {
	_numRowsAttr = _rs.getString(createLookup(NUM_ROWS_ATTR_NAME),
				     null);
	Log.assert(_numRowsAttr != null,
		   "Missing resource " + _name + "." + NUM_ROWS_ATTR_NAME);

	Vector columnAttrs = new Vector();
	int column = 0;
	String columnAttr;
	while ((columnAttr =
		_rs.getString(createLookup(COLUMN_ATTR_NAME +
					   column++), null)) != null) {
	    columnAttrs.addElement(columnAttr);
	}
	_columnAttrs = new String[columnAttrs.size()];
	columnAttrs.copyInto(_columnAttrs);

	columnAttrs.removeAllElements();
	column = 0;
	while ((columnAttr = _rs.getString(createLookup(HEADING + column++),
					   null)) != null) {
	    columnAttrs.addElement(columnAttr);
	}
	_headings = new String[columnAttrs.size()];
	columnAttrs.copyInto(_headings);

	columnAttrs.removeAllElements();
	column = 0;
	while ((columnAttr = _rs.getString(createLookup(
	    INVISIBLE_COLUMN_ATTR_NAME + column++), null)) != null) {
	    columnAttrs.addElement(columnAttr);
	}
	_invisibleColumnAttrs = new String[columnAttrs.size()];
	columnAttrs.copyInto(_invisibleColumnAttrs);

	_orderHeading = _rs.getString(createLookup(ORDER_HEADING), null);
    }

    /**
     * Class used by monitorEditor() to monitor the fields in the
     * editor for user changes.
     */
    private class ColumnBinder extends TaskDataBinder {
	public void taskDataChanged(TaskDataEvent event) {
	    EditableList.this._taskData.setBoolean(
		_changedAttrName, changedSinceLastAdd());
	}
    }

    /**
     * Start monitoring the editor attributes.  Everytime they change,
     * we'll update the changedSinceLastAdd TaskData Attribute.
     */
    private void monitorEditor() {
	if (_changedAttrName == null) {
	    return;
	}
	for (int column = 0; column < _columnAttrs.length;
	     column++) {
	    String attrName = _columnAttrs[column];
	    _taskData.addTaskDataBinder(attrName, new ColumnBinder());
	}
	for (int column = 0; column < _invisibleColumnAttrs.length;
	     column++) {
	    String attrName = _invisibleColumnAttrs[column];
	    _taskData.addTaskDataBinder(attrName, new ColumnBinder());
	}
    }

    /**
     * enable this EditableList (and all its components).
     *
     * @param enable enable if <tt>true</tt>, disable if <tt>false</tt>.
     */
    public void setEnabled(boolean enable) {
	// XXX This really seems like a bug in JPanel.
	super.setEnabled(enable);
	int count = getComponentCount();
	for (int ii = 0; ii < count; ++ii) {
	    Component c = getComponent(ii);
	    c.setEnabled(enable);
	}
    }
}
