//
// FilteredTextField.java
//
//	A RTextField that translates input spaces to some other char.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import javax.swing.*;
import javax.swing.text.*;
import java.awt.*;

/**
 * FilteredTextField is an RTextField that accepts all input characters
 * except whitespace and any specified in the constructor.  Disallowed
 * whitespace characters include Unicode space, Java whitespace, and ISO
 * control characters.  Disallowed characters will be transformed to
 * UNDERSCORE (_) unless otherwise specified in the constructor.
 *
 * @author Rebecca Underwood
 * @author John Relph
 * @version $Revision: 1.5 $ $Date: 2000/07/24 21:58:15 $
 *
 * @see com.sgi.sysadm.ui.RTextField
 * @see java.lang.Character#isSpaceChar(char)
 * @see java.lang.Character#isISOControl(char)
 * @see java.lang.Character#isWhitespace(char)
 */

public class FilteredTextField extends RTextField
{
    private char _replacementChar = '_';

    private String _disallowed = "";

    /**
     * Passing in the special character FilteredTextField.BEEP as the
     * designated replacement character will cause the text field to
     * beep instead of converting disallowed characters.
     */
    public static final char BEEP = '\0';

    // FilteredDocument enforces the refusal of space characters.
    //
    private class FilteredDocument extends PlainDocument {

	public void insertString(int offs, String str, AttributeSet a)
	    throws BadLocationException {
	    StringBuffer newStr = new StringBuffer(str);
	    for (int i = 0; i < str.length(); i++) {
		char ch = str.charAt(i);

		if (Character.isSpaceChar(ch) ||
		    Character.isWhitespace(ch) ||
		    Character.isISOControl(ch) ||
		    _disallowed.indexOf(ch) != -1) {
		    if (_replacementChar == BEEP) {
			Toolkit.getDefaultToolkit().beep();
			return;
		    } else {
			newStr.setCharAt(i, _replacementChar);
		    }
		}
	    }
	    super.insertString(offs, new String(newStr), a);
	}
    }

    /**
     * Create a FilteredTextField with the specified number of columns,
     * disallowed characters, and replacement character.
     *
     * @param columns Number of columns of text.
     * @param disallowed Characters to disallow.
     * @param ch Replacement character.
     *		Passing in FilteredTextField.BEEP for this
     *		parameter will cause the FilteredTextField to beep
     *		and refuse disallowed characters and whitespace;
     *		no replacement of those characters will be done.
     */
    public FilteredTextField(int columns, String disallowed, char ch) {
	super(columns);
	_replacementChar = ch;
	_disallowed = disallowed;
	setDocument(new FilteredDocument());
    }

    /**
     * Create a FilteredTextField with the specified disallowed characters
     * and replacement character.
     *
     * @param disallowed Characters to disallow.
     * @param ch Replacement character.
     *		Passing in FilteredTextField.BEEP for this
     *		parameter will cause the FilteredTextField to beep
     *		and refuse disallowed characters and whitespace;
     *		no replacement of those characters will be done.
     */
    public FilteredTextField(String disallowed, char ch) {
	super();
	_replacementChar = ch;
	_disallowed = disallowed;
	setDocument(new FilteredDocument());
    }

    /**
     * Create a FilteredTextField with the specified number of columns
     * and disallowed characters.  Whitespace and disallowed characters
     * will be replaced with underscore ('_').
     *
     * @param columns Number of columns of text.
     * @param disallowed Characters to disallow.
     */
    public FilteredTextField(int columns, String disallowed) {
	super(columns);
	_disallowed = disallowed;
	setDocument(new FilteredDocument());
    }

    /**
     * Create a FilteredTextField with the specified number of columns.
     * Whitespace will be replaced with underscore ('_');
     *
     * @param columns Number of columns of text.
     */
    public FilteredTextField(int columns) {
	super(columns);
	setDocument(new FilteredDocument());
    }

    /**
     * Create a FilteredTextField with the specified disallowed characters.
     * Whitespace and the specified disallowed characters will be replaced
     * with underscore ('_').
     *
     * @param disallowed Characters to disallow.
     */
    public FilteredTextField(String disallowed) {
	super();
	_disallowed = disallowed;
	setDocument(new FilteredDocument());
    }

    /**
     * Create a FilteredTextField with the specified number of columns and
     * replacement character.  Whitespace will be replaced with the
     * character <TT>ch</TT>.
     *
     * @param columns Number of columns of text.
     * @param ch      Replacement character.
     *                Passing in FilteredTextField.BEEP for this
     *                parameter will cause the FilteredTextField to refuse
     *                whitespace and beep instead.
     */
    public FilteredTextField(int columns, char ch) {
	super(columns);
	_replacementChar = ch;
	setDocument(new FilteredDocument());
    }
}
