//
// Form.java
// 
//	Base class for the Form interface of a Task.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import java.awt.*;
import java.text.*;
import java.util.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;

/**
 * Form is the base class for the Form interface of a Task.  In
 * typical usage, a subclass overrides Form.createUI() to add
 * Task-specific components.
 * <P>
 * For more information about writing Tasks in Rhino, see the
 * <A HREF="../tutorials/HowToWriteATask.html">How to write a Task</A>
 * tutorial.
 *
 * @see com.sgi.sysadm.ui.Task
 */
public class Form extends TaskPage {
    private ResourceStack _rs;
    private TaskContext _taskContext;
    private String _shortName;
    private String _pageTitle;
    private String _serverName;
    private boolean _createdUI;
    
    /**
     * The property <i>Form.icon</i> describes the
     * <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH relative</A>
     * name of the <A HREF="glossary.html#IconImageFile">icon image
     * file</A> to be used on the Form page.  This property is used by
     * Form.createUI() to load the Form page icon.
     * <P>
     * A default value for this property is provided in TaskContextP.properties
     * and it should be overridden in either {package}.PackageP.properties or
     * {package}.{taskname}P.properties.
     */
    public static final String ICON = "Form.icon";

    /**
     * The property <i>Form.title</i> is a string that is displayed at the
     * top of the Form next to the Form icon.  A default value for this
     * property is provided in TaskContextP.properties and it should be
     * overridden in {package}.{taskname}P.properties.
     */
    public static final String TITLE = "Form.title";
    
    /**
     * The property <I>Form.introText</I> is a String containing
     * the introductory text for the Form.  If this property exists,
     * the text will be displayed in a RichTextArea at the top of the
     * Form.
     *
     * @see com.sgi.sysadm.ui.richText.RichTextArea
     */
    public static final String INTRO_TEXT = "Form.introText";
    
    /**
     * The property <i>Form.dynamicSize</i> is a Boolean that
     * specifies whether or not the Form should determine its height
     * dynamically based on Task.TASK_WIDTH_IN_POINTS.  If false, the
     * height is based on Task.TASK_HEIGHT_IN_POINTS if present, or
     * the <A HREF="glossary.html#GoldenRatio">golden ratio</A>.
     */
    public static final String DYNAMIC_SIZE = "Form.dynamicSize";

    /**
     * Constructor.
     *
     * @param taskContext Context of task.
     */
    public Form(TaskContext taskContext) {
	super(taskContext.getResourceStack());
    	_taskContext = taskContext;
	_rs = _taskContext.getResourceStack();
	_shortName = _rs.getString(Task.SHORT_NAME);
	_pageTitle = _rs.getString(TITLE);
	_serverName = _taskContext.getServerName();

	setupSizing(_rs, DYNAMIC_SIZE, Task.WIDTH_IN_POINTS,
		    Task.HEIGHT_IN_POINTS);
    }
    
    /**
     * Called by the Form base class when the Form interface becomes
     * visible for the first time.  Form.createUI() sets the title of
     * the Form page using the property Form.TITLE and sets the icon
     * of the Form page using the property Form.ICON.
     * <P>
     * Subclasses should override this method to create Task-specific
     * Form interface components.  Call super.createUI() before
     * creating any components to ensure that the Form title and icon
     * appear at the top of the Form.
     */
    protected void createUI() {
	setTitle(_pageTitle);
	setIcon(ICON);

	// Set the introductory text area for the form, if specified
	// in the properties file.
	String introString = null;
	try {
	    introString = _rs.getString(INTRO_TEXT);
	    RichTextArea introTextArea = new RichTextArea("intro", _rs);
	    introTextArea.setText(introString);
	    setIntroText(introTextArea);
	} catch (MissingResourceException exception) {
	    // Do nothing, the introductory text is optional.
	}
    }
    
    /**
     * Called by the Task base class each time the Form interface is to be
     * displayed (for example when the User switches from Guide to
     * Form).  The first time Form.showForm() is called, Form.createUI()
     * is called to create the user interface components.  Form.showForm()
     * enables the OK button and sets the Task title using
     * TaskContext.getTitleString().
     */
    public void showForm() {
	if (!_createdUI) {
	    createUI();
	    _createdUI = true;
	}
	
	_taskContext.getTaskControlPanel().setOKEnabled(true);
	_taskContext.setTaskTitle(_taskContext.getTitleString());
    }
}



