//
// FtrIcon.java
//
//	Base class for FtrIcons, which are generated by the
//	ftrjava program.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import java.lang.reflect.*;
import java.awt.*;
import java.awt.image.*;
import java.util.*;
import java.io.*;
import javax.swing.*;
import com.sgi.sysadm.util.*;

/**
 * FtrIcon is the base class for Icons generated by the
 * $TOOLROOT/usr/sbin/ftrjava program.  ftrjava parses .ftr files and
 * generates java source code for rendering File Typing Rule (FTR) icons.
 * <p>
 * An .ftr file defines a file type.  The Indigo Magic desktop uses
 * FTRs to determine the types of files and to display
 * appropriate icons for files.  ftrjava ignores the file typing part
 * of an FTR and turns the icon part of the FTR into a .java file
 * implementing a subclass of FtrIcon.
 * <p>
 * Here is an example of a filetyping rule (Clock.ftr):
 * <pre>
 *TYPE Clock
 *    ICON {
 *        include("iconlib/Clock.fti");
 *    }
 *</pre>
 * The ftrjava program will translate Clock.ftr into Clock.java, which
 * will define a class named Clock as a subclass of FtrIcon.  The
 * Clock class will render the icon defined in Clock.fti.  Clock.fti
 * is an icon created using the <i>iconsmith</i>(1) Irix program.
 * <p>
 * FtrIcon's <tt>set</tt> and <tt>get</tt> methods can be used to
 * provide conditional rendering:
 * <pre>
 *TYPE Cluster
 *    ICON {
 *        if (get("CAM_STATUS").equals("ACTIVE")) {
 *            include("iconlib/clusterok.fti");
 *        } else if (get("CAM_STATUS").equals("INACTIVE")) {
 *            include("iconlib/clusternotok.fti");
 *        } else {
 *            include("iconlib/cluster.fti");
 *        }
 *    }
 *</pre>
 * The conditional rendering is controlled by calling <tt>set</tt> on
 * an instance of FtrIcon.  Since ftrjava translates .ftr files to
 * java, any legal java code may be put into the ICON rule of a .ftr
 * file.  Additionally, the <tt>-i</tt> and <tt>-b</tt> options to
 * ftrjava may be used to import a package and implement interfaces so
 * that static final variables from other classes can be used in .ftr
 * files.  This may be used to avoid hard-coding strings such as in
 * the Cluster example above.
 * <p>
 * Only the public methods of FtrIcon are of interest to a developer
 * writing java code.  The protected methods and variables exist to
 * facilitate the translation of .fti files into java code.  FtrIcons
 * are typically rendered by JLabels.
 * @see JLabel
 */
public abstract class FtrIcon implements Icon {
    /**
     * <tt>true</tt> if the icon should be rendered in an opened
     * state.  This variable is only of interest to subclasses
     * generated by the ftrjava program.
     */
    protected boolean opened;
    /**
     * <tt>iconcolor</tt> is a special color which is drawn
     * differently depending on whether an icon is open or located.
     * This variable is only of interest to subclasses generated by
     * the ftrjava program.
     */
    protected final Object iconcolor = new Object();
    /**
     * color for rendering shadows.  This variable is only of interest
     * to subclasses generated by the ftrjava program.
     */
    protected final int shadowcolor = 8;
    /**
     * color for rendering outlines.  This variable is only of
     * interest to subclasses generated by the ftrjava program.
     */
    protected final int outlinecolor = 0;

    private Color _currentColor;
    private Vector _xCoordList = new Vector();
    private Vector _yCoordList = new Vector();
    private int _xCoords[];
    private int _yCoords[];
    private int _width = 100;
    private int _height = 100;
    private Graphics _graphics;
    private Graphics _mask;
    
    private boolean _selected = false;
    private boolean _located = false;
    private Hashtable _attrs = new Hashtable();

    private boolean _hasChanged = true;
    private Image _lastImage;
    
    private static final float FTR_WIDTH = 100;
    private static final float FTR_HEIGHT = 100;
    private static final String CLASS_NAME = "FtrIcon";

    // Cache the scale factor so that we don't repeat the computation
    private float _xScale = _width / FTR_WIDTH;
    private float _yScale = _height / FTR_HEIGHT;

    private static final Color DEFAULT_ICON_COLOR = new Color(255, 255, 255);
    private Color _iconColor = DEFAULT_ICON_COLOR;

    // These are indices into the table which is initialized below.
    // DEFAULT should be dithered between 7 and 15 to match SGI
    // desktop ftr rendering.  Instead, we create an extra color.
    private static final int SELECTED_ICON_COLOR = 3; // yellow
    private static final int LOCATED_ICON_COLOR = 7;  // white

    // Determine if the icons will be displayed with a index color
    // model or a direct color model.
    //
    // If it's a direct model, then only set this flag if the pixel depth
    // is 12 or greater.  That's necessary because if there are less
    // than 4 bits per pixel, then our direct color model method won't work.
    private static boolean _indexColorMode;
    private static boolean _checkedForTransparency = false;
    private static boolean _canUseTransparency = false;

    static {
	ColorModel cm = Toolkit.getDefaultToolkit().getColorModel();
	_indexColorMode = ! (cm instanceof DirectColorModel &&
			     cm.getPixelSize() >= 12);
    }
    
    private static final int TRANSPARENT_INT = 0x00000010;
    private static final Color TRANSPARENT_COLOR = new Color(0x00101010);
    private static final int TRANSPARENT_MASK = 0x000000f0;

    /*
     * Web-safe colors.  The web-safe colors are supported by both
     * netscape and internet explorer without colormap flashing, and
     * are defined as the RGB combinations in which each of R, G, and
     * B can take on the values 0, 51, 102, 153, 204, 255.
     * 
     * A WebColor has _red, _green, and _blue values of [0..5], and
     * multiplies each by 51 to convert to a Color.
     */
    private static class WebColor {
	WebColor(int red, int green, int blue) {
	    _red = red;
	    _green = green;
	    _blue = blue;
	}
	static WebColor dither(WebColor color1, WebColor color2) {
	    return new WebColor((color1._red + color2._red + 1)/2,
				(color1._green + color2._green + 1)/2,
				(color1._blue + color2._blue + 1)/2);
	}
	Color getColor() {
	    return new Color(_red * 51, _green * 51, _blue * 51);
	}
	int _red;
	int _green;
	int _blue;
    }
    private static WebColor _colorMap[] = new WebColor[] {
	new WebColor(0, 0, 0),
	new WebColor(5, 0, 0),
	new WebColor(0, 5, 0),
	new WebColor(5, 5, 0),
	new WebColor(0, 0, 5),
	new WebColor(5, 0, 5),
	new WebColor(0, 5, 5),
	new WebColor(5, 5, 5),
	new WebColor(2, 2, 2),
	new WebColor(4, 2, 2),
	new WebColor(2, 4, 2),
	new WebColor(3, 3, 1),
	new WebColor(2, 2, 4),
	new WebColor(3, 1, 3),
	new WebColor(1, 3, 3),
	new WebColor(4, 4, 4),
    };

    // A hashtable of the icons we've already rendered.
    private static Hashtable _iconHash = new Hashtable();
    
    /**
     * A class that holds all of the details about an FtrIcon.  If the
     * FtrDescriptors of two FtrIcon objects are the same, then
     * they'return drawing the same icon.
     */
    private static class FtrDescriptor {
        private String _className;
        private boolean _selected;
        private boolean _located;
        private int _width;
        private int _height;
        private Color _iconColor;
        private Hashtable _attrs;
       
        FtrDescriptor(String className, boolean selected, 
                      boolean located, int width, int height, 
                      Color iconColor,
                      Hashtable attrs) {
            _className = className;
            _selected = selected;
            _located = located;
            _width = width;
            _height = height;
            _iconColor = new Color(iconColor.getRGB());
            _attrs = (Hashtable)attrs.clone();
        }
        
        
        public String toString() {
            return 
            "\n\tClass: " + _className +
            "\n\tselected: " + _selected +
            "\n\tlocated: " + _located +
            "\n\twidth: " + _width +
            "\n\theight: " + _height +
            "\n\ticonColor: " + _iconColor +
            "\n\tattrs: " + _attrs;
        }

        /**
	 * Test to see if two FtrDescriptors are equal
	 */
        public boolean equals(Object obj) {
            if (obj == null) {
                return false;
            }
            FtrDescriptor other;
            try {
                other = (FtrDescriptor)obj;
            } catch (ClassCastException e) {
                return false;
            }
            if ((!other._className.equals(_className)) ||
                other._selected != _selected ||
                other._located != _located ||
                other._width != _width  ||
                other._height != _height ||
                !other._iconColor.equals(_iconColor) ||
                other._attrs.size() != _attrs.size()) {
                return false;
            }
	    // If all those tests passed, then all that's left is to
	    // check the actual attributes themselves.
            Enumeration enum = other._attrs.keys();
            while (enum.hasMoreElements()) {
                Object key = enum.nextElement();
                if (!other._attrs.get(key).equals(_attrs.get(key))) {
                    return false;
                }
            }
            return true;
        }
        
	// Try to give a unique hashcode to each FtrDesciptor without
	// taking too much time.
        public int hashCode()  {
            return (_className.hashCode() + _width + _height +
		    _attrs.size() + (_selected ? 2 : 0) + 
		    (_located ? 1 : 0));
        }
    }
    
    
    /**
     * Dynamically load a subclass of FtrIcon.
     * 
     * @param className Name of the class to load.
     * @exception IllegalArgumentException Thrown if
     *            <tt>className</tt> is not the name of a subclass of
     *            FtrIcon.
     * 
     * @return FtrIcon instance corresponding to <tt>className</tt>.
     */
    public static FtrIcon createIcon(String className) {
	try {
	    Class iconClass = SysUtil.forName(className);
	    Constructor ctor = iconClass.getConstructor(null);
	    return  (FtrIcon)ctor.newInstance(null);
	} catch (Exception ex) {
	    throw new IllegalArgumentException(ex.getMessage());
	}
    }

    /**
     * Set the selected flag.  The selected and located flags
     * determine the color of the icon.
     * 
     * @param selected true if this icon should be draw selected.
     */
    public void setSelected(boolean selected) {
	if (_selected != selected) {
	    _selected = selected;
	    _hasChanged = true;
	}
    }

    /**
     * Set the located flag.  The selected and located flags determine
     * the color of the icon.
     * 
     * @param located true if this icon should be draw located.
     */
    public void setLocated(boolean located) {
	if (_located != located) {
	    _located = located;
	    _hasChanged = true;
	}
    }

    /**
     * Set the size which should be used for rendering.
     * 
     * @param width horizontal size.
     * @param height vertical size.
     */
    public void setSize(int width, int height) {
	if (_width != width || _height != height) {
	    _width = width;
	    _height = height;
	    
	    // recalculate the scales
	    _xScale = width / FTR_WIDTH;
	    _yScale = height / FTR_HEIGHT;
	    _hasChanged = true;
	}
    }

    /**
     * Perform a test to see if our transparency scheme is going to
     * work.  We draw a white pixel into the graphics, then read it
     * back to see what we get.  If we get "0xff000000" (black) back,
     * then assume that we can't cache the FtrIcons and just draw the
     * icon each time.  This works around a problem when the GUI is
     * running on IRIX and being remote displayed to Linux.
     */
    private void checkTransparency(Component component) {
	if (_checkedForTransparency) {
	    return;
	}
	_checkedForTransparency = true;
	Image image = component.createImage(1,1);
	Graphics g = image.getGraphics();
	g.setColor(Color.white);
	g.drawLine(0, 0, 1, 0);	    
	
	int pix[] = new int[1];
	PixelGrabber pg = new PixelGrabber(
	    image, 0, 0, 1, 1, pix, 0, 1);
	try {
	    pg.grabPixels();
	}
	catch (InterruptedException e) {
	    Log.error(CLASS_NAME, "interrupted waiting for pixels!");
	    return;
	}
	if (pix[0] == 0xff000000) {
	    Log.debug(CLASS_NAME, "Can't use transparency");
	    _canUseTransparency = false;
	} else {
	    Log.debug(CLASS_NAME, "Can use transparency");
	    _canUseTransparency = true;	
	}
    }
    
    /**
     * Icon method that gets called at render time.
     * 
     * @param component component in which we are drawing.
     * @param graphics for drawing.
     * @param x horizontal location.
     * @param y vertical location.
     * @see Icon#paintIcon
     */
    public void paintIcon(Component component, Graphics graphics,
			  int x, int y) {
	checkTransparency(component);
	if (!_canUseTransparency) {
	    Rectangle clipRect = graphics.getClipBounds();
	    
	    // Create a new Graphics object with the origin at x,y.
	    // That way the draw() method can always assume it's
	    // drawing from 0,0.
	    _graphics = graphics.create(x, y, 
					clipRect.width - x,
					clipRect.height - y);
	    draw();
	    return;
	}
	
        Image image;
        if (!_hasChanged) {
            image = _lastImage;
        } else {
            FtrDescriptor desc = new FtrDescriptor(
                this.getClass().getName(),
                _selected, _located, _width, _height,
                _iconColor, _attrs);
            if ((image = (Image)_iconHash.get(desc)) == null) {
                // Create the image and store it in the cache.  The
                // hard thing here is that we need the FtrIcon to have
                // transparent pixels, and there's no way to directly
                // draw transparent pixels.  So, we'll use one
                // of two methods, depending on whether we're using a
                // index color model or a direct color model.
                // If direct:
                //
                // 1) Get an Image from the Component where we're
                //    going to draw the icon
                // 2) Get the Graphics associated with the image
                // 3) Fill the Graphics with a special Color that
                //    can't be used by the subclass.
                // 4) Draw the icon onto the Graphics
                // 5) Access the Image as an Integer array
                // 6) Loop through the Integers in the Image, changing
                //    all the special color pixels to be transparent
                // 7) Create a new image from the new pixels
                // 8) Draw the new image on the graphics
                //
                // If index:
                // 1) Get two images from Component
                // 2) Get two Graphics from the images
                // 3) Paint a pure white backgroud on the second Graphics
                // 4) Draw the FtrIcon normally on the first Graphics, and
                //    draw a pure black version of the icon on the
                //    second Graphics
                // 5) Access the Images as Integer arrays.
                // 6) Loop through the Integers in the first and
                //    second Graphics, setting the pixels in the first
                //    Graphics to transparent iff the pixel was white in
                //    the second Graphics
                // 7) Create a new image from the new pixels
                // 8) Draw the new image on the graphics
                //
                // The index method would work for both modes, but
                // it's slower, so we use the direct method if possible.

                image = component.createImage(_width, _height);
                Image maskImage = null;
                if (_indexColorMode) {
                    maskImage = component.createImage(_width, _height);
                    _mask = maskImage.getGraphics();
                }
                _graphics = image.getGraphics();

                if (_indexColorMode) {
                    _mask.setColor(Color.white);
                    _mask.fillRect(0,0,_width,_height);
                    _mask.setColor(Color.black);
                } else {
                    // TRANSPARENT_COLOR is not a websafe color, so no FTR
                    // can use it. Fill the image with this color. 
                    _graphics.setColor(TRANSPARENT_COLOR);
                    _graphics.fillRect(0,0,_width,_height);
                }
                draw();

		// Now, get access to the array of pixels that compose
		// the image.
                int pix[] = new int[_width * _height];
                int maskPix[] = null;
                PixelGrabber pg = new PixelGrabber(
                    image, 0,0, _width, _height, pix, 0, _width);
                PixelGrabber maskPg = null;
                if (_indexColorMode) {
                    maskPix = new int[_width * _height];
                    maskPg  = new PixelGrabber(
                        maskImage, 0,0, _width, _height, maskPix, 0, _width);
                }
                try {
                    pg.grabPixels();
                    if (_indexColorMode) {
                        maskPg.grabPixels();
                    }
                    
                } catch (InterruptedException e) {
                    Log.error(CLASS_NAME, "interrupted waiting for pixels!");
                    return;
                }
                
                if (_indexColorMode) {
                    
                    // Loop through the pixels in the mask.  If the
                    // pixel is white, then set the pixel in the real
                    // image to be transparent.
                    for (int j = 0; j < _width * _height; j++) {
                        if (maskPix[j] == 0xffffffff) {
                            pix[j] = 0;
                        }
                    }
                } else {

                    // Loop through all the pixels.  If any of them are
                    // our special TRANSPARENT color,
                    // then change that pixel to 0x00000000, which is
                    // transparent.
                    for (int j = 0; j < _width * _height; j++) {
                        if ((pix[j] & TRANSPARENT_MASK) == TRANSPARENT_INT) {
                            pix[j] = 0;
                        }
                    }
                }

		// Create a new image out of the modified array.
                MemoryImageSource mis = 
                new MemoryImageSource(_width,_height,pix,0,_width);
                image = component.createImage(mis);
                _iconHash.put(desc, image);
	    }
            
        }
        
        graphics.drawImage(image, x, y, null);
        _hasChanged = false;
        _lastImage = image;
        }

    /**
     * Set the icon color to <tt>iconColor</tt>.
     * 
     * @param iconColor New icon color.
     */
    public void setIconColor(Color iconColor) {
	if (!_iconColor.equals(iconColor)) {
	    _iconColor = iconColor;
	    _hasChanged = true;
	}
    }

    /**
     * Get the current icon color.
     * 
     * @return current icon color.
     */
    public Color getIconColor() {
	return _iconColor;
    }

    /**
     * Set the icon color back to its default value.
     */
    public void setDefaultIconColor() {
	setIconColor(DEFAULT_ICON_COLOR);
    }

    /**
     * Icon method that gets called to query the height of the icon.
     * 
     * @return vertical size.
     * @see Icon#getIconHeight
     */
    public int getIconHeight() {
	return _height;
    }

    /**
     * Icon method that gets called to query the width of the icon.
     * 
     * @return horizontal size.
     * @see Icon#getIconWidth
     */
    public int getIconWidth() {
	return _width;
    }

    /**
     * Interface for setting attributes to affect rendering.
     * 
     * @param key key of attr to set.
     * @param value value of attr to set.
     */
    public void set(String key, String value) {
	if (!value.equals(_attrs.put(key, value))) {
	    _hasChanged = true;
	}
    }
    
    /**
     * Interface for setting attributes to affect rendering.
     * 
     * @param attr The attribute to set
     */
    public void set(Attribute attr) {
	set(attr.getKey(), attr.stringValue());
    }
    
    /**
     * Interface for setting a bundle of attributes that affect
     * rendering.  Each attribute must be of type string.
     * @param bundle An AttrBundle
     */
    public void set(AttrBundle bundle) {
	Enumeration enum = bundle.getAttrEnum();
	while (enum.hasMoreElements()) {
	    set((Attribute)enum.nextElement());
	}
    }

    /**
     * Get attribute to affect rendering.
     * 
     * @param key key of attr to get.
     * 
     * @return value for attr.
     */
    public String get(String key) {
	String val = (String)_attrs.get(key);
	return val == null ? "" : val;
    }

    /**
     * Set the current color.  This method is only of interest to
     * subclasses generated by the ftrjava program.
     * 
     * @param color The current color.  Should be set to <tt>iconcolor</tt>.
     */
    protected void color(Object color) {
	if (color == iconcolor) {
	    Color clr;
	    if (_selected) {
		clr = _colorMap[SELECTED_ICON_COLOR].getColor();
	    } else if (_located) {
		clr = _colorMap[LOCATED_ICON_COLOR].getColor();
	    } else {
		clr = _iconColor;
	    }
	    _graphics.setColor(clr);
	} else {
	    Log.fatal("illegal call to color()");
	}
    }

    /**
     * Set the current color.  This method is only of interest to
     * subclasses generated by the ftrjava program.
     * 
     * @param color The current color.  Indexes into our colormap.
     */
    protected void color(int color) {
	if (color > 0) {
	    _graphics.setColor(_colorMap[color %
					_colorMap.length].getColor());
	} else {
	    _graphics.setColor(WebColor.dither(
		_colorMap[(-color) % _colorMap.length],
		_colorMap[(-color) >> 4]).getColor());
	}
    }

    /**
     * Start drawing a polygon.  This method is only of interest to
     * subclasses generated by the ftrjava program.
     */
    protected void bgnpolygon() {
	clearCoords();
    }

    /**
     * Finish drawing a polygon.  This method is only of interest to
     * subclasses generated by the ftrjava program.
     */
    protected void endpolygon() {
	setCoords();
	_graphics.fillPolygon(_xCoords, _yCoords, _xCoords.length);
        if (_indexColorMode) {
            _mask.fillPolygon(_xCoords, _yCoords, _xCoords.length);
        }
    }

    /**
     * Start drawing a series of points.  This method is only of interest to
     * subclasses generated by the ftrjava program.
     */
    protected void bgnpoint() {
	clearCoords();
    }

    /**
     * Finish drawing a series of points.  This method is only of interest to
     * subclasses generated by the ftrjava program.
     */
    protected void endpoint() {
	int numCoords = _xCoordList.size();
	Log.assert(numCoords == _yCoordList.size(),
		   "Coordinate vectors are of unequal size.");
	for (int ii = 0; ii < numCoords; ii++) {
	    int x = ((Integer)_xCoordList.elementAt(ii)).intValue();
	    int y = ((Integer)_yCoordList.elementAt(ii)).intValue();
	    _graphics.fillRect(x, y, 1, 1);
            if (_indexColorMode) {
                _mask.fillRect(x, y, 1, 1);
            }
            
	}
    }

    /**
     * Start drawing a series of connected lines.  This method is only
     * of interest to subclasses generated by the ftrjava program.
     */
    protected void bgnline() {
	clearCoords();
    }

    /**
     * Finish drawing a series of connected lines.  This method is
     * only of interest to subclasses generated by the ftrjava
     * program.
     */
    protected void endline() {
	setCoords();
	_graphics.drawPolyline(_xCoords, _yCoords, _xCoords.length);
        if (_indexColorMode) {
            _mask.drawPolyline(_xCoords, _yCoords, _xCoords.length);
        }
        
    }

    /**
     * Start drawing a series of closed lines.  This method is only of
     * interest to subclasses generated by the ftrjava program.
     */
    protected void bgnclosedline() {
	clearCoords();
    }

    /**
     * Finish drawing a series of closed lines.  This method is only
     * of interest to subclasses generated by the ftrjava program.
     */
    protected void endclosedline() {
	setCoords();
	_graphics.drawPolygon(_xCoords, _yCoords, _xCoords.length);
        if (_indexColorMode) {
            _mask.drawPolygon(_xCoords, _yCoords, _xCoords.length);
        }
    }

    /**
     * Start drawing an outlined polygon.  This method is only of
     * interest to subclasses generated by the ftrjava program.
     */
    protected void bgnoutlinepolygon() {
	bgnclosedline();
    }

    /**
     * Finish drawing an outlined polygon.  This method is only of
     * interest to subclasses generated by the ftrjava program.
     * 
     * @param color Color for the outline.
     */
    protected void endoutlinepolygon(int color) {
	endpolygon();
	Color saveColor = _graphics.getColor();
	color(color);
	endclosedline();
	_graphics.setColor(saveColor);
    }

    /**
     * Finish drawing an outlined polygon.  This method is only of
     * interest to subclasses generated by the ftrjava program.
     * 
     * @param color Color for the polygon.
     */
    protected void endoutlinepolygon(Object color) {
	endpolygon();
	Color saveColor = _graphics.getColor();
	color(color);
	_graphics.setColor(saveColor);
	endclosedline();
    }

    /**
     * Add a vertex to the list of vertices for the current shape.
     * This method is only of interest to subclasses generated by the
     * ftrjava program.
     * 
     * @param x horizontal coordinate.
     * @param y vertical coordinate.
     */
    protected void vertex(double x, double y) {
	_xCoordList.addElement(
	    new Integer((int)(x * _xScale)));
	_yCoordList.addElement(
	    new Integer((int)((FTR_HEIGHT - y) * _yScale)));
    }

    /**
     * Called to draw the icon.  Subclass calls the <tt>bgn*</tt>
     * <tt>end*</tt>, and <tt>vertex</tt> methods.  This method is
     * only of interest to subclasses generated by the ftrjava
     * program.
     */
    protected abstract void draw();

    /**
     * Get rid of accumulated coordinates.
     */
    private void clearCoords() {
	_xCoordList.removeAllElements();
	_yCoordList.removeAllElements();
    }

    /**
     * Set the _xCoords and _yCoords arrays to the set of accumulated
     * coordinates.
     */
    private void setCoords() {
	Log.assert(_xCoordList.size() == _yCoordList.size(),
		   "Coordinate vectors are of unequal size.");
	_xCoords = new int[_xCoordList.size()];
	_yCoords = new int[_yCoordList.size()];
	for (int ii = 0; ii < _xCoords.length; ii++) {
	    _xCoords[ii] = ((Integer)_xCoordList.elementAt(ii)).intValue();
	    _yCoords[ii] = ((Integer)_yCoordList.elementAt(ii)).intValue();
	}
    }
/*
  Intentially commented out.  Only used for debugging.
    private void debugTransparency(Component component) {
	Image image = component.createImage(256,1);
	Graphics g = image.getGraphics();
	for (int ii = 0; ii < 256; ii++) {
	    g.setColor(new Color(0,0,ii));
	    g.drawLine(ii, 0, ii + 1, 0);	    
	}

	int pix[] = new int[256];
	PixelGrabber pg = new PixelGrabber(
	     image, 0, 0, 256, 1, pix, 0, 256);
	try {
	    pg.grabPixels();
	}
	catch (InterruptedException e) {
                    System.err.println("interrupted waiting for pixels!");
                    return;
	}
	ColorModel cm = component.getToolkit().getColorModel();
	for (int j = 0; j < 256; j++) {
	    System.out.println(" " + j + ": " + Integer.toHexString(pix[j]));
	    System.out.println("*" + j + ": " +
			       Integer.toHexString(cm.getRGB(pix[j])));
	}
	System.out.println("ColorModel: " + cm);
	DirectColorModel dcm = (DirectColorModel)cm;
	System.out.println("Alpha: " +
			   Integer.toHexString(dcm.getAlphaMask()));
	System.out.println("Red: " +
			   Integer.toHexString(dcm.getRedMask()));
	System.out.println("Green: " +
			   Integer.toHexString(dcm.getGreenMask()));
	System.out.println("Blue: " +
			   Integer.toHexString(dcm.getBlueMask()));
    }
*/    
}
