//
// GenericItemRenderer
//
//	A class that can turn an Item into an Object
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.util.*;
import java.util.*;

/**
 * A class that can turn an Item into something else (For example a
 * String or a Icon).  Exactly what the Item is turned into is
 * determined by subclasses.
 */
public abstract class GenericItemRenderer {
    /**
     * The ResourceStack passed to the constructor
     */
    private ResourceStack _rs;
    
    /**
     * The HostContext passed to the constructor
     */
    private HostContext _hc;
    
    /**
     * The fullCategoryName passed to the constructor
     */
    private String _name;
    
    private Hashtable _listeners = new Hashtable();
    private Category _category;
    private Hashtable _items = new Hashtable();
    private CategoryListener _catListener = new CatListener();
    private int _numListeners;
    
    /**
     * A Object that is passed as <tt>context</tt> to
     * <tt>createRenderedObject</tt> when the Item is being rendered
     * for an ItemView.
     */
    public static final Object ITEM_VIEW = new Object();

    /**
     * A Object that is passed as <tt>context</tt> to
     * <tt>createRenderedObject</tt> when the Item is being rendered
     * for an ResultView.
     */
    public static final Object RESULT_VIEW = new Object();

    /**
     * A Object that is passed as <tt>context</tt> to
     * <tt>createRenderedObject</tt> when the Item is being rendered
     * for an ItemTable.
     */
    public static final Object ITEM_TABLE = new Object();


    /**
     * A Object that is passed as <tt>context</tt> to
     * <tt>createRenderedObject</tt> when the Item is being rendered
     * for a TreeView.
     */
    public static final Object TREE_NODE = new Object();

    /**
     * A string that is passed as <tt>itemSelector</tt> to
     * <tt>addRenderedObjectListener</tt> and
     * <tt>removeRenderedObjectListener</tt>
     * to signal that the caller is requesting that the rendered
     * object contain information about the Category in general,
     * instead of a particular Item.
     */
    public static final String CATEGORY_ONLY = new String();
    
    /**
     * The constructor.  The arguments passed to this constructor are
     * stored in protected member variables for use by subclasses.
     *
     * @param fullCategoryName The name of the Category that this
     *                         should render Items of.  
     * @param rs A ResourceStack that is used to look up properties.
     * @param hc The HostContext that is used to obtain Categories.
     */
    protected GenericItemRenderer(String fullCategoryName,
				  ResourceStack rs,
				  HostContext hc) {
	_rs = rs;
	_hc = hc;
	_name = fullCategoryName;
	_category =
	    hc.getCategory(ResourceStack.getClassName(fullCategoryName));
     }
    
    /**
     * Adds a RenderedObjectListener to the renderer
     * @param itemSelector The Item the listener is interested in.
     *                     The caller can pass CATEGORY_ONLY to
     *                     indicate that it is interested in the
     *                     Category, rather than a particular Item.
     * @param listener The listener
     * @param context An Object that is passed to the renderer to
     *                control how the object is to be rendered.  The
     *                GenericItemRenderer subclass and the caller of
     *                this method can create a protocol about how the
     *                <tt>context</tt> will be used. When the
     *                infrastructure uses ItemRenderers, it uses
     *                ITEM_VIEW, RESULT_VIEW, ITEM_TABLE, and TREE_NODE as
     *                contexts.
     *
     * @see #CATEGORY_ONLY
     * @see #ITEM_VIEW
     * @see #RESULT_VIEW
     * @see #ITEM_TABLE
     * @see #TREE_NODE
     */
    protected void addRenderedObjectListener(String itemSelector,
					     RenderedObjectListener listener,
					     Object context) {
	if (_numListeners == 0) {
	    _category.addCategoryListener(_catListener,
					  NotificationFilter.ALL_ITEMS);
	}
	_numListeners++;
	Vector v = (Vector)_listeners.get(itemSelector);
	if (v == null) {
	    v = new Vector();
	    _listeners.put(itemSelector, v);
	}
	v.addElement(new ListenerStruct(listener, context));
	listener.renderedObjectChanged (
	    itemSelector,
	    createRenderedObject((Item)_items.get(itemSelector), context));
    }
    
    /**
     * Removes a RenderedObjectListener
     * @param itemSelector The Item that the listener is no longer
     *                     interested in. The listener will continue
     *                     to receive notification about other Items
     *                     that it has registered interest
     *                     in. CATEGORY_ONLY is a legal value, if it
     *                     was passed to a corresponding
     *                     <tt>addRenderedObjectListener</tt> call.
     * @param listener The listener to remove.
     *
     * @see #CATEGORY_ONLY
     */
    protected void removeRenderedObjectListener(String itemSelector,
						RenderedObjectListener listener) {
	Vector v = (Vector)_listeners.get(itemSelector);
	if (v != null) {
	    for (int ii = 0; ii < v.size(); ii++) {
		if (((ListenerStruct)v.elementAt(ii)).listener == listener) {
		    v.removeElementAt(ii);
		}
	    }
	}
	_numListeners--;
	if (_numListeners == 0) {
	    _category.removeCategoryListener(_catListener);
	}
    }
    
    /**
     * Notifies the listeners that the rendered object has changed
     * @param item The Item that changed
     * @param removed true if the Item has been deleted, else false
     */
    protected void notifyListeners(Item item, boolean removed) {
	String itemSelector = item.getSelector();
	Vector listeners = (Vector) _listeners.get(itemSelector);
	if (listeners == null) {
	    // No interested listeners
	    return;
	}
	final int num = listeners.size();
	for (int ii = 0; ii < num; ii++) {
	    ListenerStruct struct = (ListenerStruct)listeners.elementAt(ii);
	    struct.listener.renderedObjectChanged(
		itemSelector, 
		createRenderedObject(removed ? null : item, struct.context));
	}
    } 
    
    /**
     * A method to generate the rendered object
     * @param item The Item to base the object on.
     * @param context The context to use to render the object.
     */
    abstract protected Object createRenderedObject(Item item,
						   Object context);
    
    private class ListenerStruct {
	public RenderedObjectListener listener;
	public Object context;
	public ListenerStruct(RenderedObjectListener listener,
			      Object context) {
	    this.listener =  listener;
	    this.context = context;
	}
    }

    private class CatListener extends CategoryAdapter {
	
	public void itemAdded(Item item) {
	    String selector = item.getSelector();
	    _items.put(selector, item);
	    notifyListeners(item, false);
	}
	
	public void itemChanged(Item oldItem, Item newItem) {
	    itemAdded(newItem);
	}
	
	public void itemRemoved(Item item) {
	    String selector = item.getSelector();
	    _items.remove(selector);
	    notifyListeners(item, true);
	}
    }

    /**
     *  @return The renderer's ResourceStack.
     */
    protected ResourceStack getResourceStack() {
	return _rs;
    }

    /**
     *  @return The full name of the Category for which items are being
     *           rendered.
     */
    protected String getFullCategoryName() {
	return _name;
    }


}
