//
// Guide.java
//
//	Base class for the Guide interface of a Task.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;
import java.util.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.taskData.*;

/**
 * Guide is the base class for the Guide interface of a Task.  In
 * typical usage, a subclass implements registerPages(), in which the
 * initial GuidePages of the Guide are created and registered with the
 * Guide base class using Guide.appendPage().  Subclasses and
 * TaskDataVerifiers can also use Guide.removePage() and
 * Guide.insertPage() at any time to change the list of GuidePages
 * based on User input.
 * <P>
 * The Guide base class is responsible for maintaining the list of GuidePages,
 * and responding to Guide navigation events by calling the appropriate
 * TaskDataVerifiers and changing the GuidePage that is currently displayed.
 * <P>
 * For more information about writing Tasks in Rhino, see the
 * <A HREF="../tutorials/HowToWriteATask.html">How to write a Task</A>
 * tutorial.
 *
 * @see com.sgi.sysadm.ui.Task
 * @see com.sgi.sysadm.ui.taskData.TaskDataVerifier
 */
public abstract class Guide extends JPanel {
    private final TaskContext _taskContext;
    private final ResourceStack _rs;
    private final String _shortName;
    private final String _serverName;
    private final String _errorCancelStr;
    private final String _errorContinueStr;
    private final TaskControlPanel _controlPanel;
    private final Vector _pageList = new Vector();
    private int _numPages;
    private final CardLayout _layout = new CardLayout();
    private int _currentPageIndex;
    
    /**
     * The property <i>Guide.titleFormat</i> is a format string that includes
     * three arguments: {0} for the task title string (see the
     * property Task.titleFormat), {1} for the page number of the
     * currently-displayed GuidePage, and {2} for the total number of
     * GuidePages currently in the Guide.  The format string is used
     * by Guide.showGuide() to format the Task title associated with the Guide
     * interface each time a new GuidePage is displayed.
     * <P>
     * A default value for this property is provided in TaskContextP.properties
     * and it may be overridden in either {package}.PackageP.properties or
     * {package}.{taskname}P.properties.
     */
    public static final String TITLE_FORMAT = "Guide.titleFormat";
    
    /**
     * The property <i>Guide.icon</i> describes the
     * <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH relative</A> 
     * name of the <A HREF="glossary.html#IconImageFile">icon image
     * file</A> to be used on any GuidePage that does not have its own
     * icon (specified via the property <I>GuidePage.icon</I>).  The
     * property is used by GuidePage.createUI() to load the GuidePage icon.
     * <P>
     * A default value for this property is provided in TaskContextP.properties
     * and it may be overridden in either {package}.PackageP.properties or
     * {package}.{taskname}P.properties.
     */
    public static final String ICON = "Guide.icon";
    
    /**
     * The resource <i>Guide.dynamicSize</i> is a Boolean that
     * specifies whether or not the Form should determine its height
     * dynamically based on Task.TASK_WIDTH_IN_POINTS.  If false, the
     * height is based on Task.TASK_HEIGHT_IN_POINTS if present, or
     * the golden ratio.
     */
    public static final String DYNAMIC_SIZE = "Guide.dynamicSize";
    
    /**
     * The resource <i>Guide.nextPageError.cancel</i> is a String that
     * is displayed as the label for the button on the dialog that
     * appears when a user attemps to go to the next page, but the
     * verifiers fail.  This label is for the button that returns them
     * to the page they were on.
     */
    public static final String NEXT_PAGE_CANCEL = 
        "Guide.nextPageError.cancel";

    /**
     * The resource <i>Guide.nextPageError.continue</i> is a String that
     * is displayed as the label for the button on the dialog that
     * appears when a user attemps to go to the next page, but the
     * verifiers fail.  This label is for the button that ignores the
     * error and turns the page anyway.
     */
    public static final String NEXT_PAGE_CONTINUE = 
        "Guide.nextPageError.continue";
    

    /**
     * Constructor.
     *
     * @param taskContext Context of the task.
     */
    public Guide(TaskContext taskContext) {
	_taskContext = taskContext;
	_rs = _taskContext.getResourceStack();
	_shortName = _rs.getString(Task.SHORT_NAME);
	_serverName = _taskContext.getServerName();
	_controlPanel = _taskContext.getTaskControlPanel();
	_errorCancelStr = _rs.getString(NEXT_PAGE_CANCEL);
	_errorContinueStr = _rs.getString(NEXT_PAGE_CONTINUE);
	setLayout(_layout);
	
	// Respond to Guide navigation events.  When moving forward from the
	// current page, first confirm that the TaskData attributes set so
	// far are valid and consistent.  No verification is needed when
	// moving to previous pages.
	_controlPanel.addControlListener(new TaskControlAdapter() {
	    public void taskGoToPage(GuideNavigationEvent event) {
		final GuidePage page =
		    (GuidePage)_pageList.elementAt(_currentPageIndex);
		switch (event.getPageCode()) {
		case GuideNavigationEvent.FIRST_PAGE:
		    _currentPageIndex = 0;
		    showGuide();
		    break;
		    
		case GuideNavigationEvent.LAST_PAGE:
		    _taskContext.busy();
		    page.okToAdvance(new ResultListener() {
			public void succeeded(ResultEvent event) {
			    _taskContext.notBusy();
			    _currentPageIndex = _numPages - 1;
			    showGuide();
			}
			
			public void failed(ResultEvent event) {
			    _taskContext.notBusy();
			    // If the ResultEvent contains a reason for the
			    // failure, post that in an error dialog.
			    String reason = event.getReason();
			    if (reason != null) {
				_taskContext.postError(reason);
			    }
			}
		    });
		    break;
		    
		case GuideNavigationEvent.NEXT_PAGE:
		    Log.assert(_currentPageIndex < _numPages-1,
			       "already at last page.");
		    _taskContext.busy();
		    page.okToAdvance(new ResultListener() {
			public void succeeded(ResultEvent event) {
			    _taskContext.notBusy();
			    // In case a page was removed in the verifier,
			    // check before advancing _currentPageIndex.
			    if (_currentPageIndex < _numPages -1) {
				_currentPageIndex++;
			    }
			    showGuide();
			}
			
			public void failed(ResultEvent event) {
			    _taskContext.notBusy();
			    // If the ResultEvent contains a reason for the
			    // failure, post that in an error dialog.
			    String reason = event.getReason();
			    // The reason will be null if the verifier
			    // already alerted the user to the error
			    // in another way (e.g. by posting its own
			    // dialog), so don't bother posting another.
			    if (reason == null) {
				return;
			    }
			    
			    if (page.getAllowTurnPageOnError()) {
				_taskContext.postQuestion( 
				    reason, 
				    new ActionListener() {
				        public void actionPerformed(
					    ActionEvent e) {
					}
			            },
                                    new ActionListener() {
					public void actionPerformed(
					    ActionEvent e) {
					    if (_currentPageIndex < 
						_numPages -1) {
						_currentPageIndex++;
					    }
					    showGuide();
					}
				    },
                                    _errorContinueStr, 
				    _errorCancelStr, 
				    UIContext.ERROR_ICON);
			    } else {
				_taskContext.postError(
				    reason);
			    }
			}
		    });
		    break;
		    
		case GuideNavigationEvent.PREV_PAGE:
		    Log.assert(_currentPageIndex > 0,
			       "already at first page.");
		    _currentPageIndex--;
		    showGuide();
		    break;
		    
		case GuideNavigationEvent.SPECIFIC_PAGE:
		    // XXX not yet implemented. (not needed for version 1.0)
		    break;
		}
	    }
	});
    }
    
    /**
     * Tell the Guide to show its current page.  The first time this is called,
     * the subclass will be asked to register the initial set of GuidePages.
     * However, the interface components for each GuidePage will not
     * be created until its showPage() method is called.
     * <P>
     * The TaskControlPanel will be updated based on which page of the Guide is
     * shown.  The Task window title will be generated using the
     * property Guide.TITLE_FORMAT and TaskContext.getTitleString().
     *
     * @see com.sgi.sysadm.ui.TaskControlPanel
     */
    public void showGuide() {
	if (_numPages == 0) {
	    // Tell the subclass to register its initial set of GuidePages.
	    registerPages();
	    Log.assert(_numPages > 0, "No pages have been added to Guide");
	}
	
	final GuidePage currentPage =
	    (GuidePage)_pageList.elementAt(_currentPageIndex);
	
	// Tell the page it is about to be shown, allowing it to create its
	// UI elements if necessary.
	currentPage.showPage();
	
	// Show the page.
	_layout.show(this, currentPage.getPageName());

	// Update the TaskControlPanel based on which page is shown.
	_controlPanel.setPrevEnabled(_currentPageIndex > 0);
	_controlPanel.setNextEnabled(_currentPageIndex+1 < _numPages);
	_controlPanel.setOKEnabled(_currentPageIndex+1 == _numPages);
	
	// Set the Task title.
	_taskContext.setTaskTitle(
	    MessageFormat.format(_rs.getString(Guide.TITLE_FORMAT),
			    new Object[] { _taskContext.getTitleString(),
					   new Integer(_currentPageIndex+1),
					   new Integer(_numPages) }));

	// Fix keyboard focus.  If our Window already has a legitimate
	// focus owner, don't mess with it.  If there is no focus
	// owner, or if the focus owner is disabled or invisible, try
	// to find a new one.
	// 
	// This has to be done in an invokeLater() because of
	// interactions with UIContext.busy().
	SwingUtilities.invokeLater(new Runnable() {
	    public void run() {
		boolean hasFocus = true;
		Container topLevel = getTopLevelAncestor();
		try {
		    Component comp = ((Window)topLevel).getFocusOwner();
		    hasFocus = comp != null && comp.isShowing()
			&& comp.isEnabled();
		} catch (ClassCastException ex) {
		}		    
		if (!hasFocus) {
		    // Try to set focus on the current guide page.
		    currentPage.requestFocus();

		    // Check to see if our attempt to set focus succeeded.
		    hasFocus = true;
		    try {
			Component comp = ((Window)topLevel).getFocusOwner();
			hasFocus = comp != null && comp.isShowing()
			    && comp.isEnabled();
		    } catch (ClassCastException exc) {
		    }		    
		    if (!hasFocus) {
			// We tried setting focus on the GuidePage, but it
			// didn't work, so try again at the top level.
			Component comp =
			    UIContext.findFocusTraversable(topLevel);
			if (comp != null) {
			    comp.requestFocus();
			}
		    }
		}
	    }
	});
    }
    
    /**
     * Called by the base class to request that the subclass create and
     * register the initial GuidePages via appendPage().
     */
    public abstract void registerPages();
    
    /**
     * Append a GuidePage to the ordered list of GuidePages.
     *
     * @param page The GuidePage to append to Guide.
     */
    public void appendPage(GuidePage page) {
	page.setupSizing(_rs, DYNAMIC_SIZE, Task.WIDTH_IN_POINTS,
			 Task.HEIGHT_IN_POINTS);
	_pageList.addElement(page);
	_numPages++;
	add(page, page.getPageName());
    }
    
    /**
     * Insert a GuidePage into the ordered list of GuidePages.
     *
     * @param page The GuidePage to insert into the Guide.
     * @param previousPage The GuidePage which should preceed <TT>page</TT>
     *                     in the ordered list of GuidePages.
     *                     <TT>null</TT> if <TT>page</TT> should be
     *                     the first page.
     */
    public void insertPage(GuidePage page, GuidePage previousPage) {
	int beforeIndex = _pageList.indexOf(previousPage);
	Log.assert(beforeIndex >= 0,
		   "Attempt to insert before nonexistent Guide page");
	
	page.setupSizing(_rs, DYNAMIC_SIZE, Task.WIDTH_IN_POINTS,
			 Task.HEIGHT_IN_POINTS);
	
	// CardLayout has no insert.  Remove all of the pages from previousPage
	// to the end of the layout.  Then add the new page and re-add
	// the pages that were just removed.
	int index;
	for (index = beforeIndex; index < _numPages; index++) {
	    remove((GuidePage)_pageList.elementAt(index));
	}
	_pageList.insertElementAt(page, beforeIndex);
	_numPages++;
	for (index = beforeIndex; index < _numPages; index++) {
	    GuidePage insertPage = (GuidePage)_pageList.elementAt(index);
	    add(insertPage, insertPage.getPageName());
	}

	// If the page inserted is before the current page displayed, make
	// sure to increment _currentPageIndex.
	if (beforeIndex < _currentPageIndex) {
	    _currentPageIndex++;
	}
    }
    
    /**
     * Remove a GuidePage from the ordered list of GuidePages.
     *
     * @param page The GuidePage to be removed from the Guide.
     */
    public void removePage(GuidePage page) {
	Log.assert(_numPages > 0, "No pages left to remove");
	
	int removePageIndex = _pageList.indexOf(page);
	Log.assert(removePageIndex >= 0,
		    "Attempt to remove non-existent page");

	remove(page);
	_pageList.removeElement(page);
	_numPages--;
	
	// Decrement _currentPageIndex if a page before _currentPageIndex is
	// being removed or if _currentPageIndex is the last page.
	if (_currentPageIndex > 0 && (removePageIndex < _currentPageIndex ||
			              _currentPageIndex == _numPages)) {
	    _currentPageIndex--;
	}
    }

    /**
     * Get the current page of the Guide.
     *
     * @return The current page of the Guide.
     */
    public GuidePage getCurrentPage() {
	return (GuidePage)_pageList.elementAt(_currentPageIndex);
    }
}
