//
// GuidePage.java
//
//	A single page of the Guide interface to a Task.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.util.*;
import java.awt.*;
import java.net.URL;
import java.util.*;

/**
 * GuidePage is a TaskPage that is added to the Guide 
 * interface of a Task.  Each GuidePage instance has a name, which is
 * used to find the properties for the page icon, page title, and
 * optional introductory text.
 * <P>
 * For more information about writing Tasks in Rhino, see the
 * <A HREF="../tutorials/HowToWriteATask.html">How to write a Task</A>
 * tutorial.
 *
 * @see com.sgi.sysadm.ui.Task
 */
public class GuidePage extends TaskPage {
    private String _pageName;
    private TaskContext _taskContext;
    private TaskDataVerifier _verifier;
    private boolean _createdUI;
    private ResourceStack _rs;
    private boolean _canTurnPageOnError = true;
    
    /**
     * The property <I>{GuidePage name}.title</I> is a String that is
     * used as the title of the GuidePage.  This property must be defined in
     * either {package}.PackageP.properties or
     * {package}.{taskname}P.properties if the subclass calls super.createUI()
     * to set up the page.
     */
    public static final String TITLE = "title";
    
    /**
     * The property <I>{GuidePage name}.icon</I> describes the
     * <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH relative</A> name
     * of the <A HREF="glossary.html#IconImageFile">icon image file</A> to be
     * used on the named GuidePage.  If this property is not defined,
     * then the property Guide.ICON is used.
     */
    public static final String ICON = "icon";
    
    /**
     * The property <I>{GuidePage name}.introText</I> is a String containing
     * the introductory text for the named GuidePage.  If this property exists,
     * the text will be displayed in a RichTextArea at the top of the
     * GuidePage.
     * <P>
     * If the introductory text on your GuidePage needs to
     * be dynamic (in other words, it can change based on User input or
     * system state), then you should not define this property and you
     * should add a RichTextArea component to your GuidePage.
     *
     * @see com.sgi.sysadm.ui.richText.RichTextArea
     */
    public static final String INTRO_TEXT = "introText";
    
    /**
     * The property <I>{GuidePage name}.introText.traversable</I> is a
     * Boolean that controls whether the RichTextArea created to
     * display the <B>INTRO_TEXT</B> is focus traversable.  This
     * property defaults to false if not present.
     * <P>
     * Setting <B>INTRO_TEXT_TRAVERSAL</B> to true is useful for a
     * GuidePage that contains no other Components.  It prevents
     * keyboard focus from going to the Guide Buttons, which would
     * disable keyboard navigation of the Guide.
     */
    public static final String INTRO_TEXT_TRAVERSABLE
               = "introText.traversable";
    
    /**
     * The resource <i>GuidePage.numColumns</i>
     * is an int that tells how to position the labels and components.
     * This is the default that is used if  NUM_COLUMNS is not found.
     * The options are: <dl COMPACT> 
     * <dt> 1
     * <dd> aligns the labels to the top of their components
     * <dt> 2
     * <dd> aligns the labels to the left of their components
     * </dl>
     */
    public static final String DEFAULT_NUM_COLUMNS =
        "GuidePage.numColumns";

    /**
     * Constructor.
     *
     * @param taskContext The TaskContext associated with this page.
     * @param pageName Name for this page that is used to retrieve
     *                 properties such as the page title and page icon.
     */
    public GuidePage(TaskContext taskContext, String pageName) {
	super(taskContext.getResourceStack());
	_taskContext = taskContext;
	_rs = taskContext.getResourceStack();
	_pageName = pageName;
    }
    
    /**
     * Called to control the behavior of the Task when the user hits
     * the Next button but a verifier fails.  If <tt>true</tt> is
     * passed, then the error dialog will have a button that allows
     * them to browse to the next page anyway.  If <tt>true</tt> is
     * passed, then the error dialog will only have an OK button and
     * the user will not be able to turn the page until the error is
     * corrected.
     * <p>
     * The default for GuidePages is to allow turining pages on
     * errors.
     *
     * @param okToTurnPage Pass <tt>true</tt> if it's okay to turn
     * the page if a verifier fails, else <tt>false</tt>
     *
     */
    public void setAllowTurnPageOnError(boolean okToTurnPage) {
	_canTurnPageOnError = okToTurnPage;
    }
    
    /**
     *  If this method returns true, then the infrastructure will
     * allow the user to see the next page of a Guide even if the
     * first page's verifiers failed.  Use <tt>setAllowTurnPageOnError</tt>
     * to set this behavior
     *
     * @return A boolean indicating if browsing to the next page is
     * allowed
     * @see setAllowTurnPageOnError(boolean)
     */
    public boolean getAllowTurnPageOnError() {
	return _canTurnPageOnError;
    }
    
    /**
     * Called by the base class the first time this page is made visible.
     * Subclasses should override this method to create their specific
     * GuidePage Components.  To have the base class automatically set the
     * page title, page icon, and intro text, subclass.createUI() should
     * call super.createUI() <B>before</B> creating any GuidePage-specific
     * Components.
     */
    public void createUI() {
	
	// Set the title for this page.
	setTitle(_rs.getString(_pageName + "." + TITLE));
	
	// Set the icon for this page.  First look for a specific icon
	// for this page under the name {_pageName}.{GuidePage}.icon.
	// If not found, fallback to the default Guide.ICON.
	URL iconURL = null;
	String iconName = null;
	try {
	    setIcon(_pageName + "." + GuidePage.ICON);
	} catch (MissingResourceException exception) {
	    setIcon(Guide.ICON);
	}
	
	// Set the introductory text area for this page, if specified
	// in the properties file.
	String introString = null;
	try {
	    introString = _rs.getString(_pageName + "." + INTRO_TEXT);
	} catch (MissingResourceException exception) {
	    // Do nothing, the introductory text is optional.
	}
	
	if (introString != null) {
	    RichTextArea introTextArea = new RichTextArea("intro", _rs);
	    introTextArea.setFocusTraversable(
		_rs.getBoolean(_pageName + "." +
			       INTRO_TEXT_TRAVERSABLE,
			       false));
	    introTextArea.setText(introString);
	    setIntroText(introTextArea);
	}
    }
    
    /**
     * Tell this page it is being shown.  If this is the first time the
     * page is shown, it will create its user interface components.
     */
    public void showPage() {
	if (!_createdUI) {
	    // Tell the subclass to create its components.
	    createUI();
	    _createdUI = true;
	}
    }

    /**
     * Called when this GuidePage should get keyboard focus.
     * GuidePage implementation uses UIContext.findFocusTraversable()
     * to find a showing traversable Component.
     */
    public void requestFocus() {
	Component comp = UIContext.findFocusTraversable(this);
	if (comp != null) {
	    comp.requestFocus();
	}
    }

    /**
     * Set (or replace) the verifier that will be called by the Guide base
     * class when the user wants to advance to the next (or subsequent) page.
     * The purpose of the verifier is to determine if the data on this
     * page is valid and consistent with other data entered into the Guide.
     *
     * @param verifier TaskDataVerifier to call before moving forward
     *                 from this page, <TT>null</TT> to remove the verifier.
     */
    public void setVerifier(TaskDataVerifier verifier) {
	_verifier = verifier;
    }
    
    /**
     * Get the name of this page.
     *
     * @return The name of this page.
     */
    public String getPageName() {
	return _pageName;
    }
    
    /**
     * Call the verifier for this page to determine if it is OK to
     * advance to a subsequent page in the Guide.  The verifier is
     * responsible for posting any error or warning dialogs.
     *
     * @param listener A ResultListener to be notified when the verification
     *                 succeeds or fails.
     */
    public void okToAdvance(ResultListener listener) {
	if (_verifier == null) {
	    listener.succeeded(new ResultEvent(this));
	    return;
	}
	
	// We used to pass MAY_BE_EMPTY here, but we're
	// deprecating that feature.  If we like the change, we'll
	// remove MAY_BE_EMPTY for good.
	_verifier.dataOK(TaskDataVerifier.MUST_BE_SET, null, listener);
    }
}

