//
// HostContext.java
//
//	Access to sysadm services for a particular host.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import java.util.*;
import java.net.*;
import java.awt.*;
import java.awt.event.*;
import java.applet.*;
import java.text.*;
import java.io.*;
import javax.help.*;

/**
 * HostContext provides the following services:
 * <ul>
 * <li>Accessor methods for system administration services:
 *   <a href=#getAssociation>getAssociation</a>,
 *   <a href=#getCategory>getCategory</a>,
 *   <a href=#getHostName>getHostName</a>,
 *   <a href=#getPrivBroker>getPrivBroker</a>,
 *   <a href=#getTaskRegistry>getTaskRegistry</a>.</li>
 * <li>Keeping track of client windows, for the purposes of blocking
 *   input to all clients when an application-modal dialog is needed
 *   and for determining when the application should exit:
 *   <a href=#blockAllClients>blockAllClients</a>,
 *   <a href=#registerClient>registerClient</a>,
 *   <a href=#unregisterClient>unregisterClient</a>,
 *   <a href=#unblockAllClients>unblockAllClients</a>.</li>
 * <li>Exit handling: <a href=#abort>abort</a>, <a href=#exit>exit</a>,
 *   <a href=#setExitHandler>setExitHandler</a>.</li>
 * <li>Providing help to the user: <a href=#help>help</a>
 * <li>Getting and setting product attributes:
 *   <a href=#getProductAttribute>getProductAttribute</a>,
 *   <a href=#getProductAttributeFromCache>getProductAttributeFromCache</a>,
 *   <a href=#getProductAttributes>getProductAttributes</a>,
 *   <a href=#monitorProductAttributes>monitorProductAttributes</a>,
 *   <a href=#setProductAttribute>setProductAttribute</a>,
 *   <a href=#unmonitorProductAttributes>unmonitorProductAttributes</a>,
 *   <a href=#unsetProductAttributes>unsetProductAttributes</a>.</li>
 * <li>Shared renderers: <a href=#getIconRenderer>getIconRenderer</a>,
 *   <a href=#getNameRenderer>getNameRenderer</a>.
 * </ul>
 * <p>
 * <tt>registerClient</tt>, <tt>getClients</tt>, and
 * <tt>unregisterClient</tt> may be used to implement run-once
 * behavior.  When the user requests that a Component be launched,
 * clients of HostContext can use <tt>getClients</tt> to determine
 * whether an appropriate Component already exists and raise it to the
 * front of the window stacking order (using <tt>RFrame.toFront</tt>).
 * If no clients are running yet, the Component is created and
 * registered.
 * <p>
 * Product attributes are key/value pairs shared by components within
 * a product that share a HostContext.  When a client calls
 * <tt>getProductAttribute</tt> or <tt>getProductAttributes</tt> and
 * the Attributes need to be set, HostContext will create an instance
 * of ProductAttributeSetter and call its setProductAttributes
 * method.  The ProductAttributeSetter can display a user interface to
 * prompt the user for information necessary to determine product
 * Attributes, if necessary.
 * @see ProductAttributeSetter
 * @see RFrame
 */
public abstract class HostContext {

    /**
     * A resource <i>&lt;package-qualified Category
     * name&gt;.iconRenderer</i> is a resource that specifies the
     * IconRenderer subclass to use for the Category
     * &lt;package-qualified Category name&gt;.
     */
    public static final String ICON_RENDERER = ".iconRenderer";

    /**
     * A resource <i>&lt;package-qualified Category
     * name&gt;.nameRenderer</i> is a resource that specifies the
     * NameRenderer subclass to use for the Category
     * &lt;package-qualified Category name&gt;.
     */
    public static final String NAME_RENDERER = ".nameRenderer";

    /**
     * A resource <i>ProductAttributeSetter</i> is a resource that
     * can be in the PackageP resource file corresponding to the
     * product that is passed to getProductAttribute or
     * getProductAttributes.  If the resource exists, it is used as
     * the name of a class that implements the ProductAttributeSetter
     * interface.
     */
    public static final String SETTER = "ProductAttributeSetter";

    /**
     * Passed to getProductAttribute[s] to specify that the product
     * attributes should be refreshed by reinvoking the SETTER.  If
     * the SETTER fails, then the old values will be retained.
     */
    public static final boolean REFRESH_ATTRIBUTES = true;

    /**
     * Passed to getProductAttribute[s] to specify that the product
     * attributes should be used as is, unless they havn't been
     * retrieved yet, in which case the SETTER will be invoked
     */
    public static final boolean DONT_REFRESH_ATTRIBUTES = false;

    /**
     * <i>HostContext.loadingHelp</i> is a String that is displayed in
     * a dialog box if loading the help will take a while.
     */
    public static final String LOADING_HELP = "HostContext.loadingHelp";


    /**
     * <i>HelpSet</i> is a String that points to a JavaHelp Help Set
     * (.hs) file.  It should be of the format
     * <CODE>/com/sgi/simgr/help/simgr.hs</CODE>.  Typically, this
     * constant is used to look up the Help Set in a Resource Stack
     * and pass it to the <tt>help</tt> method.
     */
    public static final String HELP_SET = "HelpSet";

    private static final String CLASS_NAME = "HostContext";

    private static Vector _contexts = new Vector();

    private static Window _topWindow = null;

    private Hashtable _products = new Hashtable();

    // This hashtable stores (as keys) the name of the products whose attributes
    // have been loaded from the server.  The value should be ignored.
    private Hashtable _areProductAttrsLoadedFromServer = new Hashtable();
    private Hashtable _areProductAttrsLoadedFromSetter = new Hashtable();
    private Hashtable _savedProductMonitors = new Hashtable();
    private Hashtable _setters = new Hashtable();

    private ExitHandler _exitHandler;
    private int _clientsBlockedLevel = 0;
    private boolean _exited = false;

    // The set of clients sharing this HostContext.  When the last
    // client calls clientExit(), HostContext will shut down the jvm.
    private Hashtable _clients = new Hashtable();
    private Hashtable _clientKeys = new Hashtable();

    // The renderers that this object knows about.
    private Hashtable _iconRenderers = new Hashtable();
    private Hashtable _nameRenderers = new Hashtable();

    /**
     * An array of system properties that are used to initialize product
     * Attributes. The format of the value should be:
     * <CODE>HostContext.productAttr&lt;n&gt;=&lt;product&gt;:&lt;attribute 
     * key&gt;:&lt;attribute value&gt;</CODE>
     */
    public static final String STATIC_PRODUCT_ATTR = "HostContext.productAttr";
    
    private static Hashtable _helpBrokerHash = new Hashtable(2);
    private static String _busyString;
    private static ResourceStack _defaultStack;

    static {
        _defaultStack = new ResourceStack();
	_defaultStack.pushBundle("com.sgi.sysadm.ui.SysadmUI" +
                                 ResourceStack.BUNDLE_SUFFIX);
    }
    
    /**
     * Provide help to the user, using JavaHelp
     *
     * @param helpSetPath The path to the HelpSet (.hs) file to use.
     * @param key help tag of help content.
     */
    public static void help(final String helpSetPath, final String key) {
        Log.debug(CLASS_NAME, "Called help(" + helpSetPath + ", " +
                  key + ")");
        if (_busyString == null) {
            _busyString = _defaultStack.getString(LOADING_HELP);
        }
        createHelpBroker(helpSetPath, new ResultListener() {
                public void succeeded(ResultEvent e) {
                    showHelp(helpSetPath, key);
                }
                public void failed(ResultEvent e) {
                    // Won't get here.
                }
            });
    }
    
    /** 
     * Loads the HelpBroker for a particular helpSet.  
     * Maintains a cache of HelpBrokers and returns immediately if
     * it's already loaded.  Instead of returning the helpBroker,
     * it puts in in the _helpBrokerHash.  This is because this
     * method is asynchronous.  It will return right away, but the
     * HelpBroker won't be available in the _helpBrokerHash until
     * the ResultListener is called.
     */
    private static void createHelpBroker(final String helpSetPath,
                                  final ResultListener listener) {
        if (_helpBrokerHash.containsKey(helpSetPath)) {
            listener.succeeded(null);
            return;
        }
        final UIContext uic = new UIContext();
        uic.busy(_busyString, null);
        
        // We want to do this in a seperate thread because many of
        // the methods called below will block for several
        // seconds, and we don't want the GUI blocking for that
        // long.
        // We're careful not to call anything in here that would
        // actually display anything on the screen, as Swing is
        // not thread safe, and we would then have two threads
        // involved.
        new Thread(new Runnable() {
                public void run() {
                    HelpSet hs = null;
                    try {
                        URL hsURL =
                        getClass().getResource(helpSetPath);
                        if (hsURL == null) {
                            Log.fatal("Couldn't load help set: " + 
                                      helpSetPath + " (1)");
                            return;
                        }
                        hs = new HelpSet(null, hsURL);
                    } catch (Exception ee) {
                        Log.fatal("Couldn't load help set:  " + 
                                  helpSetPath + " (2)");
                        return;
                    }
                    HelpBroker hb = hs.createHelpBroker();
                    
                    // Initialize the presentation while we have the
                    // busy dialog up.  This makes the window appear
                    // faster the first time we try to display it.
                    hb.initPresentation();
                    
                    try {
                        hb.setLocation(new Point(50,50));
                    } catch (UnsupportedOperationException e) {
                        // Oh well ...
                    }
                    _helpBrokerHash.put(helpSetPath, hb);
                    uic.notBusy();
                    
                    // We want to make sure that the rest of the
                    // help stuff happens on the UI thread
                    SwingUtilities.invokeLater(new Runnable() {
                            public void run() {
                                listener.succeeded(null);
                            }
                        });
                }
            }).start();
    }
    
    /**
     * Displays the help screen
     * 
     * @param helpSetPath The package-qualified path to a helpSet
     *                    (.hs) file containing the help information
     *
     * @param key The key identifing the help page to show.
     */
    private static void showHelp(String helpSetPath, String key) {
        try {
            HelpBroker hb = (HelpBroker)_helpBrokerHash.get(helpSetPath);
            hb.setCurrentID(key);
            hb.setDisplayed(true);
        } catch (Exception ee) {
            Log.error(CLASS_NAME, "Couldn't show help for: " + key
                      + "\n" + SysUtil.stackTraceToString(ee));
            return;
        }
    }    

    /**
     * ExitHandler is used to handle requests to exit.
     */
    public static interface ExitHandler {
	/**
	 * Exit the current application.
	 *
	 * @param status exit status.
	 */
	public void exit(int status);
    }

    /**
     * AppExitHandler is used to handle exit requests when running as
     * a Java application.
     */
    public static class AppExitHandler implements ExitHandler {
	/**
	 * Exit the current application.  Calls System.exit().
	 *
	 * @param status exit status.
	 */
	public void exit(int status) {
	    System.exit(status);
	}
    }

    /**
     * Client is used to encapsulate both a Frame and UIContext for
     * registry with HostContext to allow all of the clients in a
     * HostContext to be blocked by host-wide dialogs.
     */
    public static class Client {
	Frame _frame;
	UIContext _uic;
	int _blockedLevel = 0;

	/**
	 * Constructor.
	 *
	 * @param frame The Frame associated with this client.
	 * @param uic The UIContext associated with this client.
	 */
	public Client(Frame frame, UIContext uic) {
	    _frame = frame;
	    _uic = uic;
	}

	/**
	 * Get the Frame associated with this client.
	 *
	 * @return The Frame associated with this client.
	 */
	public Frame getFrame() {
	    return _frame;
	}

	/**
	 * Get the UIContext associated with this client.
	 *
	 * @return The UIContext associated with this client.
	 */
	public UIContext getUIContext() {
	    return _uic;
	}

	/**
	 * Block or unblock the client.  If the client is not blocked,
	 * a request to unblock will be ignored.
	 *
	 * @param block True if the client should be blocked, False
	 *              otherwise.
	 */
	private void block(boolean block) {
	    if (block) {
		_uic.blockInput(true, _topWindow);
		_blockedLevel++;
	    } else {
		if (_blockedLevel > 0) {
		    _uic.blockInput(false, _topWindow);
		    _blockedLevel--;
		}
	    }
	}
    }

    /**
     * Construct a HostContext.
     */
    public HostContext() {
	_contexts.addElement(this);
	String arg;
	char sep = ':';
	for (int ii = 0; 
	     (arg = SysUtil.getSystemProperty(
		 STATIC_PRODUCT_ATTR + ii)) != null; 
	     ii++) {
	    Log.assert(arg.indexOf(sep) > 0 && 
		       arg.indexOf(sep) != arg.lastIndexOf(sep), 
		       "Bad format for: " + arg);
	    String product = arg.substring(0, arg.indexOf(sep));
	    String key  = arg.substring(1 + arg.indexOf(sep),
					arg.lastIndexOf(sep));
	    String value = arg.substring(1 + arg.lastIndexOf(sep));
	    Log.info(CLASS_NAME, STATIC_PRODUCT_ATTR + " flag is set:" +
			"\n\tproduct: " + product +
			"\n\tkey:     " + key +
			"\n\tvalue:   " + value);
	    setProductAttribute(product, new Attribute(key, value));
	}
    }

    /**
     * Gets a product attribute.
     *
     * @param product The product that contains the desired Attribute
     * @param key The key of the desired attribute
     * @param uic The UIContext that will be passed to the
     *            ProductAttributeSetter
     * @param refresh Should be either REFRESH_ATTRIBUTES or
     *                DONT_REFRESH_ATTRIBUTES.
     * @param context An object that will be sent to the
     *                ProductAttributeSetter
     * @param listener The listener to return the Attribute to.  The
     *                 result sent to the listener should be cast to
     *                 an Attribute.  If the host context is unable to
     *                 get the Attribute requested, the failed method of the
     *                 listener will be called.
     */
    public void getProductAttribute(final String product,
				    final String key,
				    final UIContext uic,
				    final boolean refresh,
				    final Object context,
				    final ResultListener listener) {
	final UIContext uiContext = new UIContext();
	uiContext.setDialogParent(uic.getDialogParent(),
				  new ResultListener() {
	    public void failed(ResultEvent event) {
		// Won't get here
	    }

	    public void succeeded(ResultEvent event) {
		getAttrs(product, uiContext, refresh, context,
			 new ResultListener() {
		    public void succeeded(ResultEvent event) {
			if (listener != null) {
			    listener.succeeded(new ResultEvent(
				HostContext.this,

				((AttrBundle)_products.get(product)).
				getAttr(key)));}
		    }

		    public void failed(ResultEvent event) {
			if (listener != null) {
			    listener.failed(event);
			}
		    }
		});
	    }
	});
    }

    /**
     * Gets product attributes from the cache.  This will not force a
     * load of the product if the product or attribute is not found.
     * It will just return null.  Useful for loaders that need to know
     * about the previous state of the attributes.
     *
     * @param product The product that contains the Attribute
     * @param attribute The Attribute to get
     */
    public Attribute getProductAttributeFromCache(String product,
						  String attribute) {
	AttrBundle bundle =  (AttrBundle)_products.get(product);
	if (bundle == null) {
	    return null;
	} else {
	    return bundle.getAttr(attribute);
	}
    }

    /**
     * Sets a product Attribute
     *
     * @param product The product that contains the Attribute
     * @param attribute The Attribute to set
     */
    public void setProductAttribute(String product, Attribute attribute) {
	AttrBundle attrBundle = (AttrBundle)_products.get(product);
	if (attrBundle == null) {
	    attrBundle = new AttrBundle("");

	    // Check if there are any saved product monitors for this
	    // product.
	    Vector productListeners =
		(Vector)_savedProductMonitors.get(product);
	    if (productListeners != null) {
		// Add the saved product monitors to the new AttrBundle
		Enumeration listeners = productListeners.elements();
		while (listeners.hasMoreElements()) {
		    attrBundle.addAttrListener(
			(AttrListener)listeners.nextElement());
		}
	    }

	    _products.put(product, attrBundle);
	}
	attrBundle.setAttr(attribute);
    }

    /**
     * Unsets all the product attributes.  Listeners are notified
     * appropriately.
     * @param product The product to unset
     */
    public void unsetProductAttributes(String product) {
	((AttrBundle)_products.get(product)).removeAllAttrs();
	_products.remove(product);
    }

    /**
     * Unsets the specified product attribute.  Listeners are notified
     * appropriately.
     * @param product The product of the Attribute to unset
     * @param key The key of the Attribute to unset.
     */
    public void unsetProductAttribute(String product, String key) {
	((AttrBundle)_products.get(product)).removeAttr(key);
    }
	
    /**
     * Fills the AttributeBundle passed to the method with all known
     * product attributes for the requested product.
     *
     * @param product The product that contains the desired Attribute
     * @param bundle The bundle to fill with the Attributes
     * @param uic The UIContext that will be passed to the
     *            ProductAttributeSetter
     * @param refresh Should be either REFRESH_ATTRIBUTES or
     *                DONT_REFRESH_ATTRIBUTES.
     * @param context An object that will be sent to the
     *                ProductAttributeSetter
     * @param listener The listener to notify when the bundle is
     *                 filled and ready to be read.
     */
    public void getProductAttributes(final String product,
				     final AttrBundle bundle,
				     final UIContext uic,
				     final boolean refresh,
				     final Object context,
				     final ResultListener listener) {
	final UIContext uiContext = new UIContext();
	uiContext.setDialogParent(uic.getDialogParent(),
				  new ResultListener() {
	    public void failed(ResultEvent event) {
		// Won't get here
	    }

	    public void succeeded(ResultEvent event) {
		getAttrs(product, uiContext, refresh, context,
			 new ResultListener() {
		    public void succeeded(ResultEvent event) {
			ResultEvent newEvent =
			    new ResultEvent(HostContext.this);
			AttrBundle localBundle =
			    (AttrBundle)_products.get(product);
			if (localBundle == null) {
			    // Didn't find any productAttributes
			    if (listener != null) {
				listener.succeeded(newEvent);
			    }
			} else {
			    Enumeration enum = localBundle.getAttrEnum();
			    while (enum.hasMoreElements()) {
				bundle.setAttr((Attribute)enum.nextElement());
			    }
			    if (listener != null) {
				listener.succeeded(newEvent);
			    }
			}
		    }
		    public void failed(ResultEvent event) {
			if (listener != null) {
			    listener.failed(event);
			}
		    }
		});
	    }

	});
    }

    /**
     * Monitor a set of product attributes for changes.
     *
     * @param product The product to monitor.
     * @param listener The AttrListener that will be added to the
     *                 product AttrBundle.
     */
    public void monitorProductAttributes(String product,
					 AttrListener listener) {
	Vector productMonitors =
	    (Vector)_savedProductMonitors.get(product);
	if (productMonitors == null) {
	    // This is the first monitor for this product
	    productMonitors = new Vector();
	    _savedProductMonitors.put(product, productMonitors);
	}
	productMonitors.addElement(listener);
	AttrBundle productBundle = (AttrBundle)_products.get(product);
	if (productBundle == null) {
	    // Save this request until the product is loaded
	} else {
	    productBundle.addAttrListener(listener);
	    // Notify the listeners of exiting attributes
	    Enumeration enum = productBundle.getAttrEnum();
	    while (enum.hasMoreElements()) {
		listener.attrAdded(
		    new AttrEvent(productBundle,
				  (Attribute)enum.nextElement()));
	    }
	}
    }

    /**
     * Stop monitoring a set of product attributes.
     *
     * @param product The product to stop monitoring.
     * @param listener The AttrListener to remove from the product
     *                 AttrBundle.
     */
    public void unmonitorProductAttributes(String product,
					   AttrListener listener) {
	// Remove the saved request.
	Vector productMonitors =
	    (Vector)_savedProductMonitors.get(product);
	productMonitors.removeElement(listener);
	AttrBundle productBundle = (AttrBundle)_products.get(product);
	if (productBundle != null) {
	    productBundle.removeAttrListener(listener);
	}
    }

    private void getAttrs(final String product,
			  final UIContext uic,
			  final boolean refresh,
			  final Object context,
			  final ResultListener listener) {
	// ignore the refresh parameter here, as the server's product
	// attrs won't be changing while the app is running.
	if (_areProductAttrsLoadedFromServer.get(product) == null) {
	    loadProductAttrsFromServer(product, new ResultListener() {
		public void succeeded(ResultEvent event) {
		    _areProductAttrsLoadedFromServer.put(product, product);
		    getAttrs2(product, uic, refresh, context, listener);
		}
		public void failed(ResultEvent event) {
		    // Can't get here.
		}
	    });
	} else {
	    getAttrs2(product, uic, refresh, context, listener);
	}
    }
    
    private void getAttrs2(final String product,
			   final UIContext uic,
			   final boolean refresh,
			   final Object context,
			   final ResultListener listener) {
	ResultEvent event = new ResultEvent(this);
	if (!refresh && 
	    (_areProductAttrsLoadedFromSetter.get(product) != null)) {
	    listener.succeeded(event);
	    return;
	}
	ProductAttributeSetter setter =
	    (ProductAttributeSetter)_setters.get(product);
	if (setter == null) {
	    setter = loadSetter(product, uic);
	    if (setter != null) {
		_setters.put(product, setter);
	    }
	}
	if (setter == null) {
	    // There's no setter, so just report success
	    _areProductAttrsLoadedFromSetter.put(product, product);
	    listener.succeeded(event);
	} else {
	    setter.setProductAttributes(context, new ResultListener()
               {
		   public void succeeded(ResultEvent event) {
		       _areProductAttrsLoadedFromSetter.put(product,
							    product);
		       listener.succeeded(event);
		   }
		   
		   public void failed(ResultEvent event) {
		       listener.failed(event);
		   }
	       });
	}
    }

    private ProductAttributeSetter loadSetter(String product,
					      UIContext uic) {
	ResourceStack rs = (ResourceStack)_defaultStack.clone();
	rs.pushBundle(product + ".Package" + ResourceStack.BUNDLE_SUFFIX);
	String className = null;
	try {
	    className = rs.getString(SETTER);
	    ProductAttributeSetter setter =
		(ProductAttributeSetter)SysUtil.createObject(
		    className, ProductAttributeSetter.class, null, null);
	    uic.getResourceStack().pushBundle(
		product +
		".Package" +
		ResourceStack.BUNDLE_SUFFIX);
	    setter.initializeProductAttributeSetter(this, uic);
	    return setter;
	}
	catch (Exception e) {
	    Log.error(CLASS_NAME, "Could not load " + className +
		      " for use as a ProductAttributeSetter");
	    return null;
	}
    }

    /**
     * Register a client that is sharing this HostContext.  When the
     * last client sharing this HostContext calls
     * <a href=#unregisterClient>unregisterClient</a>,
     * <a href=#exit>exit</a> will be called.  Otherwise the JVM will
     * continue running.
     *
     * @param client Object that is sharing this HostContext.
     */
    public void registerClient(Object client) {
	registerClient(client, "");
    }

    /**
     * Register a client that is sharing this HostContext, and
     * associate it with a key.  When the
     * last client sharing this HostContext calls
     * <a href=#unregisterClient>unregisterClient</a>,
     * <a href=#exit>exit</a> will be called.  Otherwise the JVM will
     * continue running.  This version of <tt>registerClient</tt>
     * associates a key with the client, so that the client can later
     * be retrieved using the <tt>getClients</tt> method.  This
     * mechanism can be used to implement run-once behavior.
     * If this client has already been registered, the new key replaces
     * any previous key associated with this client.
     *
     * @param client Object that is sharing this HostContext.
     * @param key Key to be used for subsequent retrieval of client.
     */
    public void registerClient(Object client, String key) {

	// first look up client to see if it was previously registered
	// using another key
	//
	String oldKey = (String)_clientKeys.remove(client);
	if (oldKey != null) {
	    Vector vec = (Vector)_clients.get(oldKey);
	    if (vec != null) {
		vec.removeElement(client);
		if (vec.isEmpty()) {
		    _clients.remove(oldKey);
		}
	    }
	}

	// now add new key for client
	//
	Vector vec = (Vector)_clients.get(key);
	if (vec == null) {
	    vec = new Vector();
	    _clients.put(key, vec);
	}
	vec.addElement(client);
	_clientKeys.put(client, key);

	if (_clientsBlockedLevel > 0) {
	    // A host-wide dialog is posted, so don't let input get to
	    // this new client yet.
	    try {
		HostContext.Client hcClient = (HostContext.Client)client;
		hcClient.block(true);
	    } catch (ClassCastException exception) {
		// Can't block input on this client so ignore it.
	    }
	}

    }

    /**
     * Get a Vector of clients that were registered with
     * <tt>key</tt>.  The returned Vector contains the set of Objects
     * that have been registered by <tt>registerClient</tt> using
     * <tt>key</tt>.
     *
     * @param key Specifies which clients we're interested in.
     *
     * @return Vector of clients registered with <tt>key</tt>.
     */
    public Vector getClients(String key) {
	Vector vec = (Vector)_clients.get(key);
	return vec != null ? (Vector)vec.clone() : new Vector();
    }

    /**
     * Unregister a client that is sharing this HostContext.  If this
     * is the only client left, <a href=#exit>exit</a>, otherwise keep the
     * JVM running.
     *
     * @param client Object that will no longer be sharing this HostContext.
     */
    public void unregisterClient(Object client) {
	String key = (String)_clientKeys.remove(client);
	// We forcibly remove clients when we exit, so let them
	// unregister themselves even though they're gone.
	if (_exited && key == null) {
	    return;
	}
	Log.assert(_exited || key != null,
		   "Attempt to unregister a client that was not registered");
	Vector vec = (Vector)_clients.get(key);
	vec.removeElement(client);
	if (vec.isEmpty()) {
	    _clients.remove(key);
	}
	if (_clients.isEmpty() && !_exited) {
	    Log.debug("HostContext",
		      "HostContext.unregisterClient() calling exit");
	    exit(0);
	}
    }

    /**
     * Iterate through all of the clients and block input to them by
     * raising their glass pane.  This will only work for clients that
     * are JComponents or RootPaneContainers; other client types will
     * be ignored.
     *
     * @param topWindow The Window which should be brought to the
     *                  front if the user attempts to click on a
     *                  blocked client.  May be null.
     */
    public void blockAllClients(Window topWindow) {
	Log.assert(_topWindow == null, "all clients already blocked");
	_topWindow = topWindow;
	_clientsBlockedLevel++;
	Enumeration clientVectors = _clients.elements();
	while (clientVectors.hasMoreElements()) {
	    Enumeration clients =
		((Vector)clientVectors.nextElement()).elements();
	    while (clients.hasMoreElements()) {
		HostContext.Client client = null;
		try {
		    client = (HostContext.Client)clients.nextElement();
		    client.block(true);
		} catch (ClassCastException exception) {
		    // Ignore this client; its input can't be blocked.
		}
	    }
	}
    }

    /**
     * Iterate through all of the clients and unblock input to them by
     * dropping their glass pane.  This will only work for clients
     * that are JComponents or RootPaneContainers; other client types
     * will be ignored.
     */
    public void unblockAllClients() {
	Log.assert(_clientsBlockedLevel > 0, "Clients not blocked.");
	Enumeration clientVectors = _clients.elements();
	while (clientVectors.hasMoreElements()) {
	    Enumeration clients =
		((Vector)clientVectors.nextElement()).elements();
	    while (clients.hasMoreElements()) {
		HostContext.Client client = null;
		try {
		    client = (HostContext.Client)clients.nextElement();
		    client.block(false);
		} catch (ClassCastException exception) {
		    // Ignore this client; its input can't be blocked.
		}
	    }
	}
	_clientsBlockedLevel--;
	_topWindow = null;
    }

   
    /**
     * Exit the application or applet.
     *
     * @param status exit status.
     */
    public void exit(int status) {
	_exited = true;
	Enumeration enum = _clients.elements();
	while (enum.hasMoreElements()) {
	    Object elem = enum.nextElement();
	    if (elem instanceof Window) {
		Window win = (Window)elem;
		win.setVisible(false);
		win.dispose();
	    } else if (elem instanceof Vector) {
		Enumeration clients =
		    ((Vector)elem).elements();
		while (clients.hasMoreElements()) {
		    try {
			Client client = (Client)clients.nextElement();
			client._frame.setVisible(false);
			client._frame.dispose();
		    } catch (ClassCastException exception) {
			// Ignore this client.
		    }
		}
	    }
	}
	_contexts.removeElement(this);
	if (_exitHandler == null) {
	    _exitHandler = new AppExitHandler();
	}
	_exitHandler.exit(status);
    }

    /**
     * Set the ExitHandler which will handle requests to exit.
     *
     * @param exitHandler Handles requests to exit.
     */
    public void setExitHandler(ExitHandler exitHandler) {
	_exitHandler = exitHandler;
    }

    /**
     * Exit all HostContexts in this JVM.
     */
    public static void abort() {
	Vector contexts = (Vector)_contexts.clone();
	Enumeration enum = contexts.elements();
	while (enum.hasMoreElements()) {
	    HostContext hc = (HostContext)enum.nextElement();
	    hc.exit(1);
	}
    }

    /**
     * Get a Category for a host.
     *
     * @param name Name of Category to get.
     *
     * @return Category
     */
    public abstract Category getCategory(String name);

    /**
     * Get an Association for a host.
     *
     * @param parentCategory The type of the parent Item.
     * @param parentSelector The selector of the parent Item.
     * @param childCategory The type of the related Item(s).
     *
     * @return Association
     */
    public abstract Association getAssociation(String parentCategory,
					       String parentSelector,
					       String childCategory);

    /**
     * Get the PrivBroker instance for a host.
     *
     * @return PrivBroker instance.
     */
    public abstract PrivBroker getPrivBroker();

    /**
     * Get the TaskRegistry instance for a host.
     *
     * @return TaskRegistry instance.
     */
    public abstract TaskRegistry getTaskRegistry();

    /**
     * Get the ProductInfo instance for a host.
     *
     * @return ProductInfo instance.
     */
    public abstract ProductInfo getProductInfo();

    /**
     * Get the name of the host this HostContext represents.
     *
     * @return name of the host this HostContext represents.
     */
    public abstract String getHostName();

    /**
     * Gets a renderer from the hashtable or creates a new one.
     * @param category The category the renderer is for.
     * @param hash The hashtable containing renderers already created
     * @param suffix The string to append to the category as the name
     *               of the resource that gives the renderer.
     * @param rs The resource stack that is used to find the renderer
     *           and is passed to a new renderer.
     * @param defaultObj The object that is returned if the resource
     *                   lookup fails.
     */
    private GenericItemRenderer getRenderer(String category,
					    Hashtable hash,
					    String suffix,
					    boolean iconRenderer) {
	GenericItemRenderer renderer =
	    (GenericItemRenderer)hash.get(category);
	if (renderer != null) {
	    return renderer;
	} 
	
	ResourceStack rs = (ResourceStack)_defaultStack.clone();
	try {
	    rs.pushPackageBundles(category);
	    rs.pushBundle(category + ResourceStack.BUNDLE_SUFFIX);
	} catch (MissingResourceException mre) {
	}
	
	GenericItemRenderer rendererObj = null;
	String className = null;
	try {
	    className = rs.getString(category+suffix);
	    rendererObj = (GenericItemRenderer)
		SysUtil.createObject(className,
				     GenericItemRenderer.class, 
				     new Class[] {
					 String.class,
					 ResourceStack.class,
					 HostContext.class
				     } , new Object[] {
					 category, rs, this
				     });
	    
	} catch (MissingResourceException mre) {
	    // do nothing.  The Renderer is optional.
	} catch (ClassNotFoundException cnfe) {
	    Log.fatal("Failed to load class " +
		      className +
		      ".  Error was: " + cnfe.getMessage());
	}
	catch (SysUtil.ClassLoadException exception) {
	    Log.fatal("Failed to load class " +
		      exception.getClassName() +
		      ".  Error was: " + exception.getMessage());
	}
	
	if (rendererObj == null) {
	    if (iconRenderer) {
		rendererObj = new ResourceBasedIconRenderer(category, rs,
							this);
	    } else {
		rendererObj = new ResourceBasedNameRenderer(category, rs,
							this);
	    }
	}
	hash.put(category, rendererObj);
	return rendererObj;
    }
    
    /**
     * Returns an IconRenderer.  IconRenderers are shared.
     *
     * @param category The category of Items to be rendered.
     */
    public IconRenderer getIconRenderer(String category) { 
	return (IconRenderer)getRenderer(category,
					 _iconRenderers,
					 ICON_RENDERER,
					 true);
    }

    /**
     * Returns an NameRenderer.  NameRenderers are shared.
     *
     * @param category The category of Items to be rendered.
     */
    public NameRenderer getNameRenderer(String category) {
	return (NameRenderer)getRenderer(category,
					 _nameRenderers,
					 NAME_RENDERER,
					 false);
    }
    
    private void loadProductAttrsFromServer(final String productName,
					    final ResultListener listener) {
	getProductInfo().getProductAttrs(productName, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		ResourceBundle bundle =
		    (ResourceBundle)event.getResult();
		Enumeration enum = bundle.getKeys();
		String key;
		while (enum.hasMoreElements()) {
		    key = (String)enum.nextElement();
		    Attribute attr = 
			new Attribute(key, bundle.getString(key));
		    setProductAttribute(productName, attr);
		    Log.debug(CLASS_NAME, 
			      "Setting a product Attribute for: "
			      + productName + " to: " + attr.getKey() +
			      ", " + attr.stringValue());
		}
		listener.succeeded(event);
	    }
	    public void failed(ResultEvent event) {
		// Won't get here.
	    }
	});
    }
}
