//
// ItemFinder.java
//
//	A subclass of JComboBox that displays the Items in a Category
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import javax.swing.*;
import com.sgi.sysadm.category.*;
import java.text.Collator;
import java.util.*;
import com.sgi.sysadm.util.*;
import java.awt.event.*;
import com.sgi.sysadm.ui.event.*;
import java.awt.*;
import javax.swing.text.*;
import javax.swing.event.*;

/** 
 * A subclass of JComboBox that displays the Items in a Category.
 * <p> 
 * The ItemFinder displays a drop-down menu of the names of Items in a
 * particular Category.  The ItemFinder also allows the user to type
 * in the name of an Item instead of choosing it from the menu.  The
 * ItemFinder does not prevent the user from typing in the name of an
 * Item that does not exist, but allows client code to detect that
 * state via the ItemFinderState.
 * <p>
 * The ItemFinder can use one of two ways to determine the name which
 * will be displayed for each Item in the drop-down menu.  The easy way
 * is to pass it a String which will be used as an Attribute key; the
 * name displayed for each Item will be its value of that Attribute.
 * For more control over the names displayed, you can pass your own
 * implementation of ItemDisplayNameRenderer which determines the name
 * to display for each Item.
 * <p>
 * The Items can be filtered in two ways.  The first is to pass an
 * Association that contains only the Items to display to the
 * <tt>setCategory</tt> method.  The other possibility is to pass an
 * ItemTester to the ItemFinder constructor.  Only Items that pass the
 * ItemTester will be displayed in the ItemFinder.
 * <p>
 * In some cases, it may be desireable to display one or more strings
 * in the ItemFinder that do not correspond directly to Items.  One
 * example would be an "All Items" string that the user could choose
 * to specify that they wished to choose all the available Items.
 * These strings can be passed to the constructor as the <tt>headers</tt>
 * parameter, or set via the <tt>setHeaders</tt> method.
 * These headers will always be displayed at the top of the menu.
 * <p>
 * ItemFinder uses a monospace font defined in a properties
 * file. Using a monospace font in a text field assures that the
 * number of input characters displayed is equal to the column width
 * of the text field.  The font resource is shared with RPasswordField and
 * RTextField.
 * 
 * @see javax.swing.JComboBox
 * @see com.sgi.sysadm.ui.RPasswordField
 * @see com.sgi.sysadm.ui.RTextField
 */
public class ItemFinder extends JComboBox implements ExtraCleanup {

    /** 
     * A resource <i>Field.font</i> specifies
     * the font to be used in the text area.  The resource is defined in
     * <TT>com.sgi.sysadm.ui.SysadmUIP.properties</TT>, and  may be
     * overridden in {package}/PackageP.properties, but please note
     * that this resource is shared with RPasswordField and
     * RTextField.
     */
    public static final String FONT = "Field.font";

    /** 
     * Pass this value to <tt>setClearBehavior</tt> to specify that the
     * ItemFinder should clear the text when the Category is set via
     * the <tt>setCategory</tt> method.
     */
    public static final int ALWAYS_CLEAR = 1;

    /**
     * Pass this value to <tt>setClearBehavior</tt> to specify that
     * the ItemFinder should <B>not</B> clear the text when the
     * Category it set via the <tt>setCategory</tt> method.
     */
    public static final int NEVER_CLEAR = 2;

    /**
     * Pass this value to <tt>setClearBehavior</tt> to specify that
     * the ItemFinder should clear the text field only
     * if it used to be displaying a Category, and <tt>setCategory</tt> was called
     * again.
     */
    public static final int CLEAR_IF_SWITCHING = 3;

    private class Item_Name {
	public Item item;
	public String name;
	public Item_Name(Item item, String name) {
	    this.item = item;
	    this.name = name;
	}
    }

    //  Perhaps this should be moved into its own separate public class?
    private static class ItemAttrDisplayNameRenderer
                         implements ItemDisplayNameRenderer {
	private String _attrKey;
	public ItemAttrDisplayNameRenderer(String attrKey) {
	    _attrKey = attrKey;
	}
        public String getDisplayName(Item item) {
            return item.getValueString(_attrKey);
        }
    }

    private ItemFinderState _state = new ItemFinderState();
    private Vector _displayedNames = new Vector();
    private Hashtable _filteredItems = new Hashtable();
    private Hashtable _unfilteredItems = new Hashtable();
    private Vector _listeners = new Vector();
    private int _changeBlocks = 0;
    private static Collator collator = Collator.getInstance();
    private ItemTester _itemTester;
    private String[] _headers;
    private ItemDisplayNameRenderer _renderer;
    private boolean _endExistsCalled = true;
    private String _currentString;
    private Category _category;
    private CategoryListener _categoryListener;
    private int _clearBehavior = CLEAR_IF_SWITCHING;

    private static Font font;

    static {
	ResourceStack rs = new ResourceStack();
	rs.pushBundle("com.sgi.sysadm.ui.SysadmUI" +
		      ResourceStack.BUNDLE_SUFFIX);
	font = rs.getFont(FONT);
    }
    

    /**
     * Construct an ItemFinder with a particular width.
     * 
     * @param itemTester An ItemTester that is used to decide which
     *                   Items to display.  Pass null to use no
     *                   ItemTester, which will result in all the
     *                   Items in the Category passed to setCategory
     *                   being displayed.
     *
     * @param itemDisplayNameRenderer the Object to use to convert the
     *                                Items into "user friendly" names
     *                                for display in this ItemFinder.
     *                                Pass null to use the selectors
     *                                as display strings.  This
     *                                renderer must provide unique
     *                                names for the Items in the
     *                                ItemFinder to avoid duplicate
     *                                menu items.
     *
     * @param headers An array of Strings that will be displayed, in
     *                order, at the top of the ItemFinder regardless
     *                of any other Items that are displayed.  Pass
     *                null to display no header strings.
     *
     * @param width  The width (in pixels) of the ItemFinder.
     *               Must be &gt; 0, or a default size will be used.
     */
    public ItemFinder(ItemTester itemTester,
		      ItemDisplayNameRenderer itemDisplayNameRenderer, 
		      String[] headers,
		      int width) {
	if (width > 0) {
	    setPreferredSize(new Dimension(width,
					   getPreferredSize().height));
	}
	_headers = (headers == null )? new String[0] : headers;
	_itemTester = itemTester;
	_renderer = itemDisplayNameRenderer == null ?
	    new ItemDisplayNameRenderer() {
		public String getDisplayName(Item item) {
		    return item.getSelector();
		}
	    }
	    : itemDisplayNameRenderer;

        setFont(font);

	setEditable(true);
	Component editorComponent = getEditor().getEditorComponent();
	if (editorComponent instanceof JTextComponent) {
	    // If the editorComponent is a JTextComponent, then we're
	    // lucky - we can attach a listener to the Document and
	    // get notification whenever a user types a key or
	    // selects an Item from the ComboBox.
	    ((JTextComponent)editorComponent).getDocument().
		addDocumentListener(new DocumentListener() {

		    public void changedUpdate(DocumentEvent e) {
			selectionChanged((String)getEditor().getItem());
		    }
		    
		    public void insertUpdate(DocumentEvent e) {
			selectionChanged((String)getEditor().getItem());
			
		    }
		    
		    public void removeUpdate(DocumentEvent e) {
			selectionChanged((String)getEditor().getItem());
		    }
		});
	} else {
	    // If the editorComponent is not a JTextComponent, then
	    // we're not so lucky.  All we can do is attach an
	    // ActionListener to the ComboBox.   We'll get immediate
	    // notification if the user selects something from the
	    // popup menu, but won't get notification about keypresses
	    // until focus leaves the ComboBox.
	    addActionListener(new ActionListener() {
		public void actionPerformed(ActionEvent e) {
		    if (e.getActionCommand().equals("comboBoxChanged")) {
			selectionChanged((String)getSelectedItem());
		    } else {
			selectionChanged(e.getActionCommand());
		    }
		}
	    });
	}
    }
    

    /**
     * Construct an ItemFinder with a default width.
     * 
     * @param itemTester An ItemTester that is used to decide which
     *                   Items to display.  Pass null to use no
     *                   ItemTester, which will result in all the
     *                   Items in the Category passed to setCategory
     *                   being displayed.
     *
     * @param itemDisplayNameRenderer the Object to use to convert the
     *                                Items into "user friendly" names
     *                                for display in this ItemFinder.
     *                                Pass null to use the selectors
     *                                as display strings.   This
     *                                renderer must provide unique
     *                                names for the Items in the
     *                                ItemFinder to avoid duplicate
     *                                menu items.
     *
     * @param headers An array of Strings that will be displayed, in
     *                order, at the top of the ItemFinder regardless
     *                of any other Items that are displayed.  Pass
     *                null to display no header strings.
     */
    public ItemFinder(ItemTester itemTester, 
		      ItemDisplayNameRenderer itemDisplayNameRenderer, 
		      String[] headers) {
	this(itemTester, itemDisplayNameRenderer, headers, 0);
    }	


    /**
     * Construct an ItemFinder which will display the given Attribute's
     * value as each Item's name.  It will be created with a default width.
     * For backwards compatibility, this constructor doesn't take a
     * String[] of headers (it would have broken delivered code which
     * was calling <CODE>new ItemTester(null, null, null)</CODE>); if
     * necessary, you can call <CODE>setHeaders(String[])</CODE> after
     * creating the ItemTester (or use one of the other constructors).
     *
     * @param itemTester An ItemTester that is used to decide which
     *                   Items to display.  Pass null to use no
     *                   ItemTester, which will result in all the
     *                   Items in the Category passed to setCategory
     *                   being displayed.
     *
     * @param itemDisplayNameAttrKey The key of the Item Attribute whose
     *                               value will be displayed as each Item's
     *                               name in this ItemFinder.  The values
     *                               of this Attribute must be unique among
     *                               the Items in this ItemFinder to avoid
     *                               duplicate menu items.
     */
    public ItemFinder(ItemTester itemTester,
		      String itemDisplayNameAttrKey) {
	this(itemTester,
             new ItemAttrDisplayNameRenderer(itemDisplayNameAttrKey),
             null,
             0);
    }


    /**
     * Construct an ItemFinder which will display the given Attribute's
     * value as each Item's name.  It will be created with the given width.
     * For backwards compatibility, this constructor doesn't take a
     * String[] of headers (it would have broken delivered code which
     * was calling <CODE>new ItemTester(null, null, null, width)</CODE>); if
     * necessary, you can call <CODE>setHeaders(String[])</CODE> after
     * creating the ItemTester (or use one of the other constructors).
     * 
     * @param itemTester An ItemTester that is used to decide which
     *                   Items to display.  Pass null to use no
     *                   ItemTester, which will result in all the
     *                   Items in the Category passed to setCategory
     *                   being displayed.
     *
     * @param itemDisplayNameAttrKey The key of the Item Attribute whose
     *                               value will be displayed as each Item's
     *                               name in this ItemFinder.  The values
     *                               of this Attribute must be unique among
     *                               the Items in this ItemFinder to avoid
     *                               duplicate menu items.
     *
     * @param width  The width (in pixels) of the ItemFinder.
     *               Must be &gt; 0, or a default size will be used.
     */
    public ItemFinder(ItemTester itemTester,
		      String itemDisplayNameAttrKey, 
		      int width) {
	this(itemTester,
             new ItemAttrDisplayNameRenderer(itemDisplayNameAttrKey),
             null,
             width);
    }


    /**
     * Sets the way that the ItemFinder reacts when setCategory is
     * called.  ItemFinder defaults to CLEAR_IF_SWITCHING.
     *
     * @param clearSelection Pass either ALWAYS_CLEAR, NEVER_CLEAR, or
     *                       CLEAR_IF_SWITCHING to control what
     *                       happens to the text that is displayed in
     *                       the ItemFinder text area when
     *                       <tt>setCategory</tt> is called.
     */
    public void setClearBehavior(int clearBehavior) {
	_clearBehavior = clearBehavior;
    }

    /** 
     * Add an ItemFinderListener to this ItemFinder
     * 
     * @param listener The listener that will be notified when the
     *                 state of the ItemFinder changes.
     */
    public void addItemFinderListener(ItemFinderListener listener) {
	_listeners.addElement(listener);
	listener.itemFinderStateChanged(new ItemFinderEvent(this,_state));
    }
    
    /**
     * Removes an ItemFinderListener from the list of listeners
     * 
     * @param listener The listener to remove
     */     
    public void removeItemFinderListener(ItemFinderListener listener) {
	_listeners.removeElement(listener);
    }

    /**
     * Sets the headers to be displayed at the top of the list of
     * Items in the ItemFinder.
     *
     * @param headers An array of Strings to use as headers.  You can
     *                use null to clear any headers that are currently
     *                set.  Any headers previously set are discarded.
     */
    public void setHeaders(String[] headers) {
	_headers = (headers == null )? new String[0] : headers;
	refresh();
    }

    /**
     * Sets the ItemTester that will be used to filter the Items.
     * Pass <tt>null</tt> to show all the Items.
     * 
     * @param tester The ItemTester
     */
    public void setItemTester(ItemTester tester) {
	_itemTester = tester;
	
	// retest all of the Items
	_filteredItems.clear();
	_displayedNames.removeAllElements();
	Enumeration enum = _unfilteredItems.elements();
	Item_Name item_name;
	while (enum.hasMoreElements()) {
	    item_name = (Item_Name)enum.nextElement();
	    if (_itemTester == null ||
		_itemTester.testItem(item_name.item).passed()) {
		_filteredItems.put(item_name.item.getSelector(),item_name);
		_displayedNames.addElement(item_name.name);
	    }
	}
	// The state of the ItemFinder, specifically whether the item
	// matched the filter, may have changed as a result of
	// changing the ItemTester.  selectionChanged() will take care
	// of getting the state up to date and sending events.
	selectionChanged(_state.getText());
	sort();
	refresh();
    }

    /**
     * Sets the text that is shown in the ItemFinder.  If the text
     * matches the name of an Item (as returned by the getDisplayName
     * method of the ItemDisplayNameRenderer object passed to the
     * constructor - or the Item's selector if
     * the ItemDisplayNameRenderer is null) that is currently
     * displayed in the ItemFinder,
     * then that Item is selected.  If the name doesn't match any
     * Item, then the text is simply displayed in the text field.
     * ItemFinderListeners are notified just as if a user had typed
     * <tt>text</tt> into the ItemFinder.
     * 
     * @param text The text to display
     */
    public void setTextByName(String text) {
	getEditor().setItem(text);
	setSelectedItem(text);
    }

    /** 
     * Sets the Item being displayed in the ItemFinder.  Generates an
     * assertion if the Item represented by the selector doesn't
     * exist, or if that Item has been rejected by the ItemFilter.   
     * ItemFinderListeners are notified just as if a user had
     * selected the Item.
     *
     * @param selector The selector of the Item.
     */
    public void setTextBySelector(String selector) {
	Item_Name item_name = (Item_Name)_filteredItems.get(selector);
	if (item_name == null) {
	    setTextByName("");
	} else {
	    setTextByName(item_name.name);
	}
    }
	
    /**
     * Returns the current state of the ItemFinder
     *
     * @return An ItemFinderState object that describes the current
     *         state.
     */
    public ItemFinderState getItemFinderState() {
	return _state;
    }
    
    /** 
     * Sets internal state and sends events to all of the current
     * ItemFinderListeners when the selection changes.
     *
     * @param newSelection The new selection in the ItemFinder
     */
    private void selectionChanged(String newSelection) {
	// check if the newSelection matches one of the Item's
	// we're displaying
	if (!_endExistsCalled) {
	    // It's pointless to try to set the state of the
	    // ItemFinder if we haven't gotten the endExists yet, so
	    // just return.  We'll call selectionChanged from
	    // endExists to take care of notifications.
	    _currentString = newSelection;
	    return;
	}
	
	_state = null;
	if (newSelection.equals("")) {
	    _state = new ItemFinderState();
	} else {
	    // Check to see if the selection is one of the headers.
	    int size = _headers.length;
	    for (int ii = 0; ii < size; ii++) {
		if (newSelection.equals(_headers[ii])) {
		    _state = new ItemFinderState(newSelection,
						 ItemFinderState.HEADER);
		    break;
		}
	    }
	    Enumeration enum;
	    Item_Name item_name;
	    if (_state == null) {
		// Check to see if the new selection is the name of one of
		// the filtered Items.
		enum = _filteredItems.elements();
		while (enum.hasMoreElements()) {
		    item_name = (Item_Name)enum.nextElement();
		    if (item_name.name.equals(newSelection)) {
			_state = new ItemFinderState(newSelection, 
						     item_name.item);
			break;
		    }
		}
	    }
	    if (_state == null) {
		// check to see if the Item exists at all
		enum = _unfilteredItems.elements();
		Vector reasons = new Vector();
		while (enum.hasMoreElements()) {
		    item_name = (Item_Name)enum.nextElement();
		    if (item_name.name.equals(newSelection)) {
			reasons.addElement(
			    _itemTester.testItem(item_name.item).reason());
		    }
		}
		if (reasons.size() != 0) {
		    Object[] reasonArray = new Object[reasons.size()];
		    reasons.copyInto(reasonArray);
		    _state = new ItemFinderState(
			newSelection,reasonArray);
		}
	    }
	    if (_state == null) {
		_state = new ItemFinderState(newSelection);
		
	    }
	}
	ItemFinderEvent event = new ItemFinderEvent(this,_state);
	int size = _listeners.size();
	for (int ii = 0; ii < size; ii++) {
	    ((ItemFinderListener)_listeners.elementAt(ii)).
		itemFinderStateChanged(event);
	}
    }
    
    /**
     * Refreshes the UI
     */
    private void refresh() {
	if (_changeBlocks == 0 && _endExistsCalled) {
	    removeAllItems();
	    int size = _headers.length;
	    for (int ii = 0; ii < size; ii++) {
		addItem(_headers[ii]);
	    }
	    size = _displayedNames.size();
	    for (int ii = 0; ii < size; ii++) {
		addItem(_displayedNames.elementAt(ii));
	    } 
	    //  It might be the case the name in the text area now
	    //  corresponds to an Item (or now doesn't correspond to
	    //  an Item), so we need to recalculate our state.
            selectionChanged(_state.getText());
	}
    }
    
    /**
     * Overridden method that catches and ignores the
     * IllegalArgumentException so that the ItemFinder doesn't die if
     * a refresh() occurs when the popup is showing.  This works
     * around a bug in Java's Swing code. (4135319)
     *
     */
    public void setSelectedIndex(int anIndex) {
	try {
	    super.setSelectedIndex(anIndex);
	} catch (IllegalArgumentException e) {
            String stack = SysUtil.stackTraceToString(e);
            // It's possible that super.setSelectedIndex will throw an
            // IllegalArgementException that's not the result of the
            // bug.  For example, when task data is bound to the
            // ItemFinder, all of the code in the binders gets run as
            // a result of the setSelectedIndex call, and that code
            // might throw an exception.  So we only will eat the
            // exception if we went through refresh() to get here.
            if (stack.indexOf("com.sgi.sysadm.ui.ItemFinder.refresh") < 0) {
                throw(e);
            }
	}
    }

    /**
     * Called by RFrame when a TaskFrame is closed.  Clean up so that
     * garbage collection can occur.
     */
    public void extraCleanup() {
	// Remove our CategoryListener so that we can be garbage-collected.
	if (_category != null) {
	    _category.removeCategoryListener(_categoryListener);
	}
    }

    /**
     * A method that sorts the data in _displayedNames
     */
    private void sort() {
	if (_changeBlocks == 0 && _endExistsCalled) {
	    int size = _displayedNames.size();
	    Object[] array = new Object[size];
	    _displayedNames.copyInto(array);
	    Sort.stable_sort(array, 0, size, new BinaryPredicate() { 
		public boolean apply(Object x, Object y) {
		    return collator.compare((String)x, (String)y) < 0;
		}
	    }); 
	    for (int ii = 0; ii < size; ii++) {
		_displayedNames.setElementAt(array[ii],ii);
	    }
	}
    }
    
    /** 
     * Sets the Category that this ItemFinder is listening to.
     *
     * @param category The category to use.  You can pass <tt>null</tt> to not
     *                 use any Category and clear the ItemFinder
     */
    public void setCategory(Category category) {
	if (_category != null) {
	    _category.removeCategoryListener(_categoryListener);
	    _displayedNames.removeAllElements();
	    _filteredItems.clear();
	    _unfilteredItems.clear();
	    refresh();
	} 
	if (_clearBehavior == ALWAYS_CLEAR || 
	    (_clearBehavior == CLEAR_IF_SWITCHING && _category != null)) {
	    setTextByName("");
	    _currentString = null;
	} else {
	    // Save off the current selection in _currentString.  That
	    // way, we can get it back after endExists.
	    _currentString = _state.getText();
	}
	_category = category;
	if (_category == null) {
	    return;
	}
	_endExistsCalled = false;
	
	_categoryListener = new CategoryAdapter() { 
	    public void itemAdded(Item item) {
		String itemName = _renderer.getDisplayName(item);
		String itemSelector = item.getSelector();
		Item_Name item_name =  new Item_Name(item,itemName);
	
		_unfilteredItems.put(itemSelector, item_name);
		if (_itemTester == null || _itemTester.testItem(item).passed()) {
		    _displayedNames.addElement(itemName);
		    _filteredItems.put(itemSelector,item_name);
		    sort();
		    refresh();
		}
	    }
  
	    public void itemChanged(Item oldItem, Item newItem) {
		String oldName = ((Item_Name)_unfilteredItems.get(
		    oldItem.getSelector())).name;
		String newName = _renderer.getDisplayName(newItem);
		String oldSelector = oldItem.getSelector();
		String newSelector = newItem.getSelector();
		int size = _displayedNames.size();
		boolean oldItemDisplayed = 
		    _filteredItems.containsKey(oldSelector);
		boolean newItemDisplayed = _itemTester == null ?
		    true :_itemTester.testItem(newItem).passed();
		boolean nameChanged = !oldName.equals(newName);
		Item_Name newItem_Name = 
		    new Item_Name(newItem, newName);

		_unfilteredItems.put(newSelector,newItem_Name);

		// If we need to remove the item from the list, or if the name
		// changed, then remove the item from both lists.
		if (oldItemDisplayed && !newItemDisplayed || 
		    oldItemDisplayed && newItemDisplayed && nameChanged) {
		    _filteredItems.remove(oldSelector);
		    for (int ii = 0; ii < size; ii++) {
			if (((String)_displayedNames.elementAt(ii)).
			    equals(oldName)) {
			    _displayedNames.removeElementAt(ii);
			    break;
			}
		    }
		    if (!( oldItemDisplayed && newItemDisplayed && nameChanged)) {
			// If this is not a replace, then refresh the
			// view.  We'll refresh the replace case when we
			// add the element back.
			refresh();
		    }
		}
		// If we need to add the item to the list, or if the name
		// changed, the add the item to both lists
		if (!oldItemDisplayed && newItemDisplayed ||
		    oldItemDisplayed && newItemDisplayed && nameChanged) {
		    _filteredItems.put(newSelector,newItem_Name);
		    _displayedNames.addElement(newName);
		    sort();
		    refresh();
		}
		// If the item changed but we're still displaying it and the
		// name didn't change, then just set the item to the new item
		// in the filtered list.
		if (oldItemDisplayed && newItemDisplayed && !nameChanged) {
		    _filteredItems.put(newSelector,newItem_Name);
		}

		// Don't do anything more if we weren't displaying the item and we
		// still aren't.
	    }

	    public void itemRemoved(Item item) {
		String itemName =
		    ((Item_Name)_unfilteredItems.get(item.getSelector())).name;
		int size = _displayedNames.size();
		for (int ii = 0; ii < size; ii++) {
		    if (((String)_displayedNames.elementAt(ii)).
			equals(itemName)) {
			_displayedNames.removeElementAt(ii);
			break;
		    }
		}
		_unfilteredItems.remove(item.getSelector());
		_filteredItems.remove(item.getSelector());
		// no need to sort in this case.
		refresh();
	    }
   
	    public void beginBlockChanges() {
		_changeBlocks++;
	    }

	    public void endBlockChanges() {
		_changeBlocks--;
		sort();
		refresh();
	    }
	    
	    public void endExists(){ 
		_endExistsCalled = true;
		if (_currentString != null) {
		    selectionChanged(_currentString);
		}
		sort();
		refresh();
	    }
	};
	_category.addCategoryListener(_categoryListener,
				      NotificationFilter.ALL_ITEMS);
    }

    /** 
     * This is a copy of the JComboBox method of the same name, with
     * minor tweaks for better performance
     *
     * We avoid sending a DESELECTED event because it's not
     * necessary and it results in more notificiations than
     * necessary. 
     */
    protected void selectedItemChanged() {

/* --- Removed from JComboBox.selectedItemChanged

        if ( selectedItemReminder != null ) {

            fireItemStateChanged(new ItemEvent(this,ItemEvent.ITEM_STATE_CHANGED,
                                               selectedItemReminder,
                                               ItemEvent.DESELECTED));
        }
*/ 
	
        selectedItemReminder = getModel().getSelectedItem();
	
        if ( selectedItemReminder != null )
            fireItemStateChanged(new ItemEvent(this,ItemEvent.ITEM_STATE_CHANGED,
                                               selectedItemReminder,
                                               ItemEvent.SELECTED));
        fireActionEvent();
    }
}
