//
// ItemFinderState.java
//
//	A class that represents the state of an ItemFinder
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui; 

import com.sgi.sysadm.category.*;
import com.sgi.sysadm.util.*;

/**
 * A class that represents the state of an ItemFinder
 */
public class ItemFinderState {
     
    /**
     * This is a reason that will be returned by <tt>getReasons</tt>
     * if the user has typed in a name of an object that the
     * ItemFinder doesn't know about.
     */
    public static final Object NO_SUCH_ITEM = "No such Item";

    /** 
     * This is a reason that will be returned by
     * <tt>getReasons</tt>if the text field of the ItemFinder is blank.
     */
    public static final Object NO_ITEM_CHOSEN = "No Item chosen";

    /**
     * This is the reason that will be returned  returned by
     * <tt>getReasons</tt> if the name selected is a header.
     */
    public static final Object HEADER = "This entry is a header";
    
    private Item _item;
    private String _text;
    private Object[] _reasons;

     /**
     * A constructor.  Use this constructor if the ItemFinder has no
     * current selection
     */
    ItemFinderState() {
	_text = "";
	_reasons = new Object[] {NO_ITEM_CHOSEN};
	_item = null;
    }
    
    /**
     * A constructor.  Use this constructor if the ItemFinder rejected
     * the Item.
     * 
     * @param text The String that is currently selected in the
     *             ItemFinder
     * @param reasons The reasons that the ItemTester rejected the Item.
     */
    ItemFinderState(String text, Object[] reasons) {
	_text = text;
	_reasons = reasons;
	_item = null;
    }
    
    /**
     * A constructor.  Use this constructor if the ItemFinder rejected
     * the Item.
     * 
     * @param text The String that is currently selected in the
     *             ItemFinder
     * @param reason The reason that the ItemTester rejected the Item.
     */
    ItemFinderState(String text, Object reason) {
	_text = text;
	_reasons = new Object[] {reason};
	_item = null;
    }
    
    
    /**
     * A constructor.  Use this constructor if the ItemFinder accepted
     * the Item.
     * 
     * @param text The String that is currently selected in the
     *             ItemFinder
     * @param item The Item that is currently selected in the
     *             ItemFinder.
     */
    ItemFinderState(String text, Item item) {
	_text = text;
	_reasons = null;
	_item = item;	
    }

    /** 
     * A constuctor.  Use this constructor if the Item is not known.
     *
     * @param text The String that is currently selected in the
     *             ItemFinder.  Will be the empty string if nothing is
     *             currently selected.
     */
    ItemFinderState(String text) {
	_text = text;
	_reasons = new Object[] {NO_SUCH_ITEM};
	_item = null;
    }
    
    /**
     * Returns the Item that is currently selected.  Can be null if
     * the user has typed text that does not identify a known Item
     * or if the ItemFinder's ItemFilter has rejected the Item whose
     * name the user typed.
     *
     * @return The Item that is selected.
     */
    public Item getItem() {
	return _item;
    }
    
    /**
     * Returns the text that is currently displayed in the ItemFinder.
     * 
     * @return A String that contains the current text in the text area.
     */
    public String getText() {
	return _text;
    }
    
    /**
     * Returns the reason that <tt>getItem</tt> returned null.  It can
     * be one of the constants <tt>NO_SUCH_ITEM</tt>,
     * <tt>NO_ITEM_CHOSEN</tt>, or <tt>HEADER</tt>, which means that the
     * ItemFinder does not know of any Item that matches the current
     * selection, or it can be an Object that the ItemTester has
     * returned that tells why the ItemTester rejected the Item.
     * <p>
     * An assertion will be thrown if this method is called when
     * <tt>getItem</tt> would have returned a non-null Item.
     * 
     * @return An array of Objects representing the reason that
     * <tt>getItem</tt> returned null.  To understand why there can be
     * multiple reasons, consider the following situation - There
     * could be several Items in the Category with the same display
     * name, each of which is being filtered out for a different
     * reason.  If the user types that name
     * into the ItemFinder, then for each of the Items, a reason will
     * be returned.  The reasons returned are specific to the
     * ItemTester, and there should be agreement between the client of
     * <tt>getReasons</tt> and the ItemTester about the meaning of the reasons.
     */
    public Object[] getReasons() {
	Log.assert(_item == null, 
		   "Can't call getReasons if the item was not null");
	return _reasons;
    }
}
