//
// ItemTable.java
//
//	A Class that encapsulates an ItemTable
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.category.*;
import java.util.*;
import java.text.*;

/** 
 * A class that encapsulates an ItemTable.
 * An ItemTable is a table where each row represents one Item in a
 * Category.  The columns of the table represent information about the
 * state or configuration of the Item.  ItemTables do not necessarily
 * show all known information about an Item - just the "first level"
 * information that a user might be interested in.  The
 * infrastructure will attempt to obtain an icon representing the Item
 * by using an IconRenderer.  If it sucessfully finds an icon, the
 * icon will be placed in the first column of the table.  Typically,
 * the second column of the ItemTable will be the name of the Item,
 * followed by some number of additional columns with more infomation.
 * <p>
 * To create an ItemTable, you should first call the static
 * <tt>createItemTable</tt> to invoke the ItemTableFactory and to
 * instantiate an ItemTable.  It is then necessary to call <tt>setCategory</tt>
 * on the ItemTable to tell the ItemTable which Category or Association
 * to display. 
 * <p>
 * To learn more about how to use resource files to define an
 * ItemTable, see the ItemTableController documentation.
 * To learn how to control the overall look of the ItemTable by
 * setting various properties, look at the ItemTableProperties
 * documentation.
 * 
 * @see ItemTableFactory
 * @see ItemTableProperties
 * @see ItemTableController
 * @see Category
 * @see Association
 */
public class ItemTable implements ItemSelectable, ItemTableProperties {
     
    private static final String CLASS_NAME = "ItemTable";
    private ItemTableController _itController;
    private ItemTableContext _itContext;
    private String _categoryName;
    private RenderedObjectListener _nameL;
    private Vector _titleListeners = new Vector(2);
    private NameRenderer _nameRenderer; 
    private String _currentTitle;
    Vector _itemListeners = new Vector();

    /** 
     * The constructor
     * 
     * @param itController the ItemTableController
     * @param ivContext the ItemTableContext
     */
    ItemTable(ItemTableController itController, 
	      ItemTableContext itContext,
	      String name) {
	_itContext = itContext;
	_itController = itController;
	_categoryName = name;
	_itController.getItemTablePanel().addListSelectionListener(
	    new RowSelectionListener());
    }
    
    /**
     * Sets the Cateogory that contains the Items the ItemTable should
     * display.  You can pass a Category or an Association to this
     * method.  The Category or the child category of the Association
     * should be the same as the name of the Category that you passed
     * to the <tt>createItemTable</tt> method.
     *
     * @param category The Category that contains the Items to
     * display.
     */
    public void setCategory(Category category) {
	_itController.setCategory(category);
    }
     
    /**
     * Return the name passed into the constructor
     * @return The category name
     */
    public String getCategoryName() {
	return _categoryName;
    } 

    /**
     * Destroys an ItemTable by removing listeners on categories.
     */
    public void destroy() {
	_itController.destroy();
	if (_nameL != null) {
	    _nameRenderer.removeNameListener(NameRenderer.CATEGORY_ONLY, _nameL);
	}
	// Null-out the member variables for better garbage collection
	// and so we'll throw an exception if someone calls any
	// methods after calling this one.
	_itController = null;
	_itContext = null;
	_nameL = null;
    }
    
    /**
     * Gets the ItemTablePanel that contains the ItemTable.
     * 
     * @return an ItemTablePanel
     */
    public ItemTablePanel getItemTablePanel() {
	return _itController.getItemTablePanel();
    }

    /**
     * Gets the ItemTableContext being used by the ItemTable
     *
     * @return the ItemTableContext
     */
    public ItemTableContext getItemTableContext() {
	return _itContext;
    }

    /**
     * Adds an ItemViewLaunchRequestListener that will be notified if
     * a user requests that an ItemView be launched (such as by
     * clicking on a hyperlink in the ItemTable) 
     *
     * @param listener the ItemViewLaunchRequestListener to add
     */
    public void addItemViewLaunchRequestListener(
	ItemViewLaunchRequestListener listener) {
	_itController.addItemViewLaunchRequestListener(listener);
    }

    /**
     * Removes an ItemViewLaunchRequestListener
     *
     * @param listener the ItemViewLaunchRequestListener to remove
     */
    public void removeItemViewLaunchRequestListener( 
	ItemViewLaunchRequestListener listener){
	_itController.removeItemViewLaunchRequestListener(listener);
    }

    /**
     * Adds a TitleListener to the ItemTable.  The TitleListener will
     * be notified whenever the title of the ItemTable changes.  The
     * ItemTable never displays its title directly.  It's up to the
     * ItemTableFrame (or any other frame) to add itself as a
     * TitleListener and set the Title appropriately.
     *
     * @param TitleListener The title listener to notify.
     */
    public void addTitleListener(TitleListener listener) {
	if (_nameRenderer == null) {
	    _nameRenderer = 
		_itContext.getHostContext().getNameRenderer(_categoryName);
	    _nameRenderer.addNameListener(
		NameRenderer.CATEGORY_ONLY,
		NameRenderer.ITEM_TABLE,
		_nameL = new RenderedObjectListener() {
		    public void renderedObjectChanged(String selector, 
						      Object object) {
                        _currentTitle = (String)object;
			notifyTitleListeners();
                    }		  
		}
		);
	}
	_titleListeners.addElement(listener);
	if (_currentTitle != null) {
	    listener.titleChanged(new TitleEvent(getItemTablePanel(), 
						 _currentTitle));
	}
    }
    
    private void notifyTitleListeners() {
	final int num = _titleListeners.size();
	for (int ii = 0; ii < num; ii++) {
	    ((TitleListener)_titleListeners.elementAt(ii)).titleChanged(
		new TitleEvent(getItemTablePanel(), _currentTitle));
	}
    }
    
    /** 
     * Creates an ItemTable by invoking the ItemTableFactory
     * 
     * @param hostContext the HostContext to place in the
     *                    ItemTableContext that will be created for
     *                    the ItemTable.
     * @param categoryName the Category name to use to find property
     *                     files and to prepend to the ItemTable
     *                     resources.
     * @exception ItemTableException if the ItemTable can't be created.
     */
    public static ItemTable createItemTable(HostContext hostContext, 
					    String categoryName)
	throws ItemTableException {
	return createItemTable(hostContext, categoryName,
			      (String)null, -1);
    }
    
    /** 
     * Creates an ItemTable by invoking the ItemTableFactory
     *
     * @param hostContext the HostContext to place in the
     *                    ItemTableContext that will be created for
     *                    the ItemTable.
     * @param categoryName the Category name to use to find property
     *                     files and to prepend to the ItemTable
     *                     resources.
     * @param resourceStack Additional ResourceBundles to push onto
     *                      the ItemTable's ResourceStack
     *
     * @exception ItemTableException if the ItemTable can't be created.
     */
    public static ItemTable createItemTable(HostContext hostContext, 
					    String categoryName,
					    ResourceStack resourceStack) 
	throws ItemTableException {
	return createItemTable(hostContext, categoryName,
			       resourceStack, -1);
    }
      
    /** 
     * Creates an ItemTable by invoking the ItemTableFactory
     * 
     * @param hostContext the HostContext to place in the
     *                    ItemTableContext that will be created for
     *                    the ItemTable.
     * @param categoryName the Category name to use to find property
     *                     files and to prepend to the ItemTable
     *                     resources.
     * @param resourceBundle Additional ResourceBundle to push onto
     *                       the ItemTable's ResourceStack
     *
     * @exception ItemTableException if the ItemTable can't be
     * created.
     */
    public static ItemTable createItemTable(HostContext hostContext, 
					    String categoryName,
					    String resourceBundleName)
	throws ItemTableException{
	return createItemTable(hostContext, categoryName,
			       resourceBundleName, -1);
    }
    
    /** 
     * Creates an ItemTable by invoking the ItemTableFactory
     * 
     * @param hostContext the HostContext to place in the
     *                    ItemTableContext that will be created for
     *                    the ItemTable.
     * @param categoryName the Category name to use to find property
     *                     files and to prepend to the ItemTable
     *                     resources.
     * @param numRows How many rows the table should have initially
     *
     * @exception ItemTableException if the ItemTable can't be created.
     */
    public static ItemTable createItemTable(HostContext hostContext, 
					    String categoryName,
					    int numRows)
	throws ItemTableException {
	return createItemTable(hostContext, categoryName, (String)null,
			       numRows);
    }
    
    /** 
     * Creates an ItemTable by invoking the ItemTableFactory
     * 
     * @param hostContext the HostContext to place in the
     *                    ItemTableContext that will be created for
     *                    the ItemTable.
     * @param categoryName the Category name to use to find property
     *                     files and to prepend to the ItemTable
     *                     resources.
     * @param resourceBundle Additional ResourceBundle to push onto
     *                       the ItemTable's ResourceStack
     * @param numRows How many rows the table should have initially
     *
     * @exception ItemTableException if the ItemTable can't be created.
     */
    public static ItemTable createItemTable(HostContext hostContext, 
					    String categoryName,
					    String resourceBundleName,
					    int numRows) 
	throws ItemTableException {
	ItemTableContext itc = new 
	    ItemTableContext(categoryName, hostContext);
	ResourceStack rs = itc.getResourceStack();
	if (resourceBundleName != null) {
	    rs.pushBundle(resourceBundleName);
	}
	return createItemTable(hostContext, categoryName, rs, itc, numRows);
    }
      
    /** 
     * Creates an ItemTable by invoking the ItemTableFactory
     * 
     * @param hostContext the HostContext to place in the
     *                    ItemTableContext that will be created for
     *                    the ItemTable.
     * @param categoryName the Category name to use to find property
     *                     files and to prepend to the ItemTable
     *                     resources.
     * @param resourceStack Additional ResourceBundles to push onto
     *                       the ItemTable's ResourceStack
     * @param numRows How many rows the table should have initially
     *
     * @exception ItemTableException if the ItemTable can't be created.
     */
    public static ItemTable createItemTable(HostContext hostContext, 
					    String categoryName,
					    ResourceStack resourceStack,
					    int numRows)
	throws ItemTableException{
	

	ItemTableContext itc = new 
	    ItemTableContext(categoryName, hostContext);
	ResourceStack rs = itc.getResourceStack();
	if (resourceStack != null) {
	    rs.pushStack(resourceStack);
	}
	return createItemTable(hostContext, categoryName, rs, itc, numRows);
	
    }
    
    private static ItemTable createItemTable(HostContext hostContext, 
					     String categoryName,
					     ResourceStack rs,
					     ItemTableContext itc,
					     int numRows)
	throws ItemTableException{ 
	
	if (hostContext == null) {
	    throw new IllegalArgumentException(
		"Can't call createItemTable with hostContext null");
	}
	if (categoryName == null) {
	    throw new IllegalArgumentException(
		"Can't call createItemTable with categoryName null");
	}
	 
	ItemTableFactory factory = null;
	try {
	    String factoryName = rs.getString(ITEM_TABLE_FACTORY);
	    try {
		factory = (ItemTableFactory)SysUtil.createObject(
		    factoryName, ItemTableFactory.class, null, null);
	    } catch (Exception e) {
		Log.error(CLASS_NAME, "Failed to load a class named \""
			  + factoryName + "\".  The error was: " +
			  e);
		throw new ItemTableException(
		    MessageFormat.format(
			rs.getString(CANT_LOAD_CLASS),
			new Object[] { factoryName, 
					   ItemTableFactory.class.getName()}));
	    }
	} catch (MissingResourceException exception) {
	    // Use the default ItemTableFactory
	}
	if (factory == null) {
	    factory = new ItemTableFactory(categoryName, itc);
	}
	ItemTable table = factory.createItemTable();
	if (numRows >= 0) {
	    table.getItemTablePanel().setNumRows(numRows);
	}
	return table;
    }

    // Need to support launching ItemTables with a particular Category or
    // Association.  But ItemTable URL's aren't used now so just
    // comment out.
    
//    private static final String URL_BEGIN = "ItemTable://category=";
//      /**
//       * Creates a URL that can be used to launch an ItemTable
//       * 
//       * @param category The name of category that contains the item to view
//       * @param item The name of the item to view.
//       * @return A URL that contains <TT>category</TT> and <TT>item</TT>
//       */
//      public static String createURLToLaunch(String category, String item) {
//  	return URL_BEGIN + category;
//      }
    
//      /**
//       * Gets the category out of a URL that was created with
//       * <TT>createURLToLaunch</TT>
//       *
//       * @param url The string to extract the category from
//       * @return the name of the category
//       * @see com.sgi.sysadm.ui.ItemTable#createURLToLaunch
//       */
//      public static String getCategoryFromURL(String url) {
	
//  	Log.assert(url.startsWith(URL_BEGIN),"Malformed ItemTable URL");
//  	return url.substring(URL_BEGIN.length());
//      }

    // -------------------- ItemSelectable Methods ---------------

    /**
     * Add a ItemListener to receive ItemEvents when the Items are selected
     * in the ItemTable, or when the selection changes.
     *
     * @param listener the listener to recieve events 
     * @see java.awt.ItemSelectable#addItemListener(ItemListener)
     */
    public void addItemListener(ItemListener listener) {
	_itemListeners.addElement(listener);
    }
      
    /**
     * Remove an ItemListener
     *
     * @param listener the listener being removed
     * @see java.awt.ItemSelectable#removeItemListener(ItemListener)
     */
    public void removeItemListener(ItemListener listener) {
	_itemListeners.removeElement(listener);
    }
    
    /**
     * Gets the list of Items that are selected in the ItemTable
     * 
     * @return The list of selected Items
     */
    public Object[] getSelectedObjects() {
	return _itController.getSelectedItems();
    }  

    /** 
     * A class that handles notifying listeners when the selection
     * changes
     */
    private class RowSelectionListener implements ListSelectionListener {

	private Vector _items = new Vector(); // Holds last selected items.

	public void valueChanged(ListSelectionEvent e) {
	    
	    // There's no point in doing anything if there are no
	    // listeners.
	    if (_itemListeners.size() == 0) {
		return;
	    }
	    
	    // Get currently selected items.
	    Item[] items = _itController.getSelectedItems();

	    for (int ii = 0; ii < items.length; ii++) {
		// Handle new selections
		if (!_items.contains(items[ii])) {
		    ItemEvent event = new ItemEvent(
			ItemTable.this,
			ItemEvent.ITEM_STATE_CHANGED,
			items[ii],
			ItemEvent.SELECTED);
		    notifyListeners(event);
		}
	    }
	    loop: for (int ii = 0; ii < _items.size(); ii++) {
		// Handle deselections
		for (int jj = 0; jj < items.length; jj++) {
		    // Look though items to see if the item is in there.
		    if (_items.elementAt(ii) == items[jj]) {
			continue loop;
		    }
		}
		// If we're here, then the Item wasn't found in
		// items.  It must have been deselected.
		ItemEvent event = new ItemEvent(
		    ItemTable.this,
		    ItemEvent.ITEM_STATE_CHANGED,
		    _items.elementAt(ii),
		    ItemEvent.DESELECTED);
		notifyListeners(event);
	    }
	    _items.removeAllElements();
	    for (int ii = 0; ii < items.length; ii++) {
		_items.addElement(items[ii]);
	    }
	}
	
	private void notifyListeners(ItemEvent event) {
	    for (int ii = 0; ii < _itemListeners.size(); ii++) {
		((ItemListener)_itemListeners.elementAt(ii)).
		    itemStateChanged(event);
	    }
	}
	
    }
}

	
