//
// ItemTableColumnRenderer
//
//	An interface that describes how to render a particular column
//	of an ItemTable.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.category.*;
import javax.swing.JComponent;

/**
 * An interface that describes how to render a particular column of an
 * ItemTable and how to sort Items based on that column.
 * The ItemTableColumnRenderer is used by the ItemTableController to
 * render cells in ways that are not automatically handled by the
 * ItemTableController.  To
 * use a ItemTableColumnRenderer, the programmer should set the METHOD
 * resource to <tt>componentRenderer</tt>, <tt>stringRenderer</tt>, or
 * <tt>richTextRenderer</tt> in the Category's property file and set the
 * COLUMN_RENDERER property to a class that implements this interface.
 * See the ItemTable documentation for more information
 *
 * @see ItemTableController
 * @see ItemTable
 * @see ItemTableProperties#COLUMN_RENDERER
 * @see ItemTableProperties#METHOD
 */
public interface ItemTableColumnRenderer {
    /** 
     * Returns a string that represents the value of a particular
     * Item in a particular column.
     * The string will be displayed in a cell of an ItemTable.
     * 
     * @param item The Item being rendered
     * @param columnName The column containing the cell that needs to
     *                   be rendered.
     * @param context An ItemTableContext for use by the renderer
     * @return A String representing the state of Item
     *         <tt>item</tt> that should be displayed in column
     *         <tt>columnName</tt>.
     */
    public String getStringForCellOfItemTable(Item item,
					      String columnName,
					      ItemTableContext context);
    
    /**
     * Returns a JComponent that should be displayed for a particular
     * Item in a particular column.  The component will be displayed
     * in a cell of an ItemTable.
     * 
     * @param item The Item being rendered
     * @param columnName The column containing the cell that needs to
     *                   be rendered.
     * @param context An ItemTableContext for use by the renderer
     * @return A JComponent representing the state of Item
     *         <tt>item</tt> that should be displayed in column
     *         <tt>columnName</tt>.
     */
    public JComponent getComponentForCellOfItemTable(Item item, 
						     String columnName,
						     ItemTableContext context);
    
    /**
     * Returns a String that should be displayed in a
     * RichTextComponent in the table's cell for a particular
     * Item in a particular column.   The string will be used to create a
     * RichTextComponent that will be displayed in an ItemTable.
     * If desired, the string that is returned can contain HTML markup
     * that describes a link to an ItemView, as created by ItemView's
     * <tt>createURLToLaunch</tt>.  In this case, the
     * ItemTableController will automatically handle the user's click
     * on the link.
     * 
     * @param item The Item being rendered
     * @param columnName The column containing the cell that needs to
     *                   be rendered.
     * @param context An ItemTableContext for use by the renderer
     * @return A String representing the state of Item
     *         <tt>item</tt> that should be displayed in column
     *         <tt>columnName</tt>.
     * @see ItemView#createURLToLaunch
     */
    public String getRichTextForCellOfItemTable(Item item, 
						String columnName,
						ItemTableContext context);
    
    /**
     * Compare two items based on a particular column.  Returns 
     * <ul> 
     * <li> &lt; 0 if item1 &lt; item2
     * <li> 0 if item1 == item2
     * <li> &gt; 0 if item1 &gt; item2
     * </ul>
     *
     * @param item1 The first Item
     * @param item2 The second Item
     * @param columnName The column of the ItemTable to compare on.
     * @return An integer
     */
    public int compareItemsForItemTable(Item item1,  
					Item item2, 
					String columnName);
}
