//
// ItemTableController.java
//
//	A class that controls an ItemTablePanel by looking up information
//	on how to display Items in a resource file.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.category.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.richText.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.table.*;
import javax.swing.border.*;
import java.util.*;
import java.awt.*;
import java.awt.Component;
import java.awt.event.*;
import java.text.Collator;

/**
 * A class that controls an ItemTablePanel by looking up information
 * on how to display Items in a resource file.
 * <p> 
 * The resources are described here, and defined in
 * ItemTableProperties.
 * <p>
 * ItemTableController has a default mode that it uses if it cannot
 * find any properties relating to the Category it is displaying.  In
 * that case, there is a column in the table for each of the
 * Attributes in the Items of the Category. 
 * <p>
 * To control how the ItemTable shows Items, you add resources to the
 * property file of the Category.  The most important resource is the
 * COLUMNS resource, which names the columns that the ItemTable will
 * use.  
 * 
 * For example, an ItemTable that is
 * displaying user accounts might have a properties file that
 * contains:
 * <PRE>
 * com.sgi.product.UserCategory.ItemTable.column0 = name
 * com.sgi.product.UserCategory.ItemTable.column1 = gecos
 * com.sgi.product.UserCategory.ItemTable.column2 = uid
 * com.sgi.product.UserCategory.ItemTable.column3 = shell
 * </PRE>
 * The names of the columns will frequently correspond to the names
 * of the Attributes of the Item that determine the value of that
 * column.  All of the methods of rendering cells except for the "renderer"
 * methods assume that this is the case.  If this is not the case,
 * then you must use the
 * BASED_ON attribute to specify which Attribute the column should be
 * based on.  For example, to display
 * the "state" Attribute twice, once as an icon and once as a
 * string, the properties file should contain:
 * <PRE>
 *  com.sgi.product.UserCategory.ItemTable.column0 = state_icon
 *  com.sgi.product.UserCategory.ItemTable.column1 = state_string
 * 
 *  com.sgi.product.UserCategory.ItemTable.basedOn.state_icon = state
 *  com.sgi.product.UserCategory.ItemTable.basedOn.state_string = state
 * </PRE> 
 * There are 7 different methods to choose from to display a column.
 * To set the method that a column uses, use the METHOD resource.
 * The methods are as follows:
 * <dl>
 * <dt><tt>toString</tt>
 * <dd>This is the most basic method of conversion.  The value of
 * the attribute is turned into a String with the toString method
 * and displayed.  No additional resources are needed for this
 * method.
 * 
 * <dt><tt>componentRenderer</tt>
 * <dd>This specifies that the
 * <tt>getComponentForCellOfItemTable</tt> method will be called on the
 * ItemTableColumnRenderer object that was passed into the
 * constructor.  The renderer can return any JComponent, and that
 * component will be displayed in the table. No additional
 * resources are needed for this method.
 *
 * <dt><tt>stringRenderer</tt>
 * <dd> This specifies that the
 * <tt>getStringForCellOfItemTable</tt> method will be called on the
 * ItemTableColumnRenderer object that was passed into the
 * constructor.  The renderer can return any String, and that
 * String will be displayed in the table.  No additional resources
 * are needed for this method.
 * 
 * <dt><tt>lookup</tt>
 * <dd>This specifies that the Attribute's value should be turned
 * into a String with the toString method, and then used to create
 * the resource name LOOKUP which will be used as the string to
 * display.  If that resource is not found, the toString method will
 * be used.  This method is good for cases when the value of the
 * Attributes comes from a limited set of possible values, and you
 * wish to provide a mapping from the Attribute to some more
 * easily understandable String.  For example, if the Attribute
 * was a boolean attribute, the following could be used in the
 * propery file:
 * <pre>
 * com.sgi.product.UserCategory.ItemTable.lookup.turnedOn.true = True
 * com.sgi.product.UserCategory.ItemTable.lookup.turnedOn.false = False
 * </pre>
 * The values displayed for true and false can now be localized
 * or modified easily.
 * 
 * <dt>richText
 * <dd>This is similar to the toString method, except that the
 * value of the Attribute will be turned into a link that can be
 * clicked to launch an ItemView.  The Attribute's value will be
 * used as the displayed text of the link. The value of the
 * resource SELECTOR
 * will be interpreted as the key of an Attribute that contains
 * the selector of the Item to view.  The value of the resource
 * CATEGORY will be used as the package-qualified name of the Category
 * to view.
 *
 * <dt>richTextRenderer
 * <dd>Similar to the stringRenderer method, except that the
 * string returned by the <tt>getRichTextForCellOfItemTable</tt>
 * method of the ItemTableColumnRenderer is used to construct a RichText
 * component.  The ItemTableController will handle catching user
 * clicks on the RichText and launching the appropriate ItemView.
 * 
 * <dt>icon
 * <dd>This specifies that the Attribute's value should be used to
 * determine an icon to display.  The attribute's value is turned
 * into a string with the toString method, and then is used to
 * lookup the resource ICON.  This resource should be the pathname of an icon
 * to show in the table.  If the resource is not found, the
 * resource DEFAULT_ICON is used as a default.
 * </dl>
 * <p>
 * For each column, you can provide a resource LABEL that defines
 * the label of the column.
 *
 * <p> 
 * For each column, you can provide a resource WIDTH that defines
 * the width (in points) of the column.  If the WIDTH resource is not
 * found for a particular column, the DEFAULT_WIDTH resource is used.
 *
 * <p>
 * For each column, you can provide a resource SORT that defines the
 * way that the Items will be sorted if the user requests that the
 * Table be sorted on a particular column.  The choices are: <dl>
 *
 * <dt>lexical
 * <dd>Specifies that a lexical sort should be used.  Can be used
 * if the method for this column is <tt>toString</tt>, <tt>stringRenderer</tt>,
 * <tt>lookup</tt>, <tt>richText</tt>, or <tt>richTextRenderer</tt>
 * 
 * <dt>numeric
 * <dd>Specifies that a numeric sort should be used.  Can be used
 * if the method for this column is <tt>toString</tt>, <tt>stringRenderer</tt>,
 * <tt>lookup</tt>, <tt>richText</tt>, or <tt>richTextRenderer</tt>
 *
 * <dt>none
 * <dd>Specifies that there is no sort order for the column.  Can
 * be used with any method.
 * 
 * <dt>sorter
 * <dd>Specifies that any two items can be sorted by passing the
 * two items and the attribute's name to sort on to the
 * compareItemsForItemTable method of the ItemTableAttributeRenderer
 * object passed to the constructor.  Can be used with any method.
 * </dl> 
 * <p>
 * For each column, you can provide a resource ALIGNMENT that defines the
 * way that the Strings will be alligned in the column.  The choices
 * are: <ul>
 * <li><tt>left</tt>
 * <li><tt>center</tt>
 * <li><tt>right</tt>
 * </ul>
 * Alignment is only available on columns using the <tt>toString</tt>,
 * <tt>lookup</tt>, or <tt>stringRenderer</tt> methods. 
 * 
 * <p> 
 * For each column, you can provide a resource MISSING that defines a
 * string that will be displayed in the cell if the Attribute that the
 * column is based on is not present in the Item.  The MISSING
 * resource works for columns using the <tt>toString</tt>,
 * <tt>lookup</tt>, or <tt>richText</tt> methods.
 *
 * @see ItemTableProperties#COLUMNS
 * @see ItemTableProperties#LABEL
 * @see ItemTableProperties#METHOD
 * @see ItemTableProperties#LOOKUP
 * @see ItemTableProperties#BASED_ON
 * @see ItemTableProperties#ICON
 * @see ItemTableProperties#DEFAULT_ICON
 * @see ItemTableProperties#WIDTH
 * @see ItemTableProperties#DEFAULT_WIDTH
 * @see ItemTableProperties#CATEGORY
 * @see ItemTableProperties#SELECTOR
 * @see ItemTableProperties#SORT
 * @see ItemTableProperties#ALIGNMENT
 * @see ItemTableProperties#MISSING
 */
public class ItemTableController implements ItemTableProperties {
     
    private static final String CLASS_NAME= "ItemTableController";

    private static final String TO_STRING_METHOD_STR= 
        "toString";
    private static final String COMPONENT_RENDERER_METHOD_STR= 
        "componentRenderer";
    private static final String STRING_RENDERER_METHOD_STR= 
        "stringRenderer";
    private static final String LOOKUP_METHOD_STR= 
        "lookup";
    private static final String RICH_TEXT_METHOD_STR=
        "richText";
    private static final String RICH_TEXT_RENDERER_METHOD_STR= 
        "richTextRenderer";
    private static final String ICON_METHOD_STR=
        "icon";

    private static final byte TO_STRING_METHOD = 0;
    private static final byte COMPONENT_RENDERER_METHOD = 1;
    private static final byte STRING_RENDERER_METHOD = 2;
    private static final byte LOOKUP_METHOD = 3;
    private static final byte RICH_TEXT_METHOD = 4;
    private static final byte RICH_TEXT_RENDERER_METHOD = 5;
    private static final byte ICON_METHOD = 6;

    private static final String LEXICAL_SORT_STR = "lexical";
    private static final String NUMERIC_SORT_STR = "numeric";
    private static final String NO_SORT_STR = "none";
    private static final String SORTER_SORT_STR = "sorter";
    
    private static final byte  LEXICAL_SORT = 0;
    private static final byte  NUMERIC_SORT = 1;
    private static final byte  NO_SORT = 2;
    private static final byte  SORTER_SORT = 3;

    private static final String LEFT_ALIGNMENT_STR = "left";
    private static final String CENTER_ALIGNMENT_STR = "center";
    private static final String RIGHT_ALIGNMENT_STR = "right";

  

    private static final String NO_CODE_SELECTOR= "Selector";
    
    private class Item_data {
	public Item item;
	public Vector data;
	public Item_data(Item item, Vector data) {
	    this.item = item;
	    this.data = data;
	}
    }
    
    private int _itemCount;
    private int _columnCount;
    private Vector _item_DataList = new Vector();
    private Hashtable _item_DataHash = new Hashtable();
    private Vector _data = new Vector();
    private ItemTablePanel _itp;
    private ResourceStack _rs;
    private ItemTableColumnRenderer _itcr;
    private Collator collator = Collator.getInstance();
    private boolean _usingResources;
    private String[] _columns;
    private byte[] _methods;
    private String[] _basedOns;
    private byte[] _sorts;
    private String[] _categories;
    private String[] _selectors;
    private ItemTableColumn[] _itemTableColumns;
    private boolean _inChangeBlock;
    private int _sortColumn = 1;
    private boolean _ascending = true;
    private Vector _itemViewLaunchRequestListeners = new Vector();
    private HostContext _hc;
    private Category _category;
    private String _name;
    private ItemTableContext _itc;
    private Icon[] _icons; 
    private IconRenderer _iconRenderer;
    private int _iconWidth;
    private int _iconHeight;
    private CategoryListener _catListener = new CatListener();

    /**
     * Construct an ItemTableController
     * @param itp The ItemTablePanel to control
     * @param name The string to use to name this ItemTable
     * @param rs A ResourceStack to use to locate resources
     * @param hostContext The HostContext
     * @param itcr An ItemTableColumnRenderer to use for rendering
     *             attributes.  Can be null.
     */
    public ItemTableController(ItemTablePanel itp,
			       String name,
			       ItemTableContext itc,
			       ItemTableColumnRenderer itcr) {
	_name = name;
	_itc = itc;
	_hc = itc.getHostContext();
	_itp = itp;
	_rs = itc.getResourceStack();
	_itcr = itcr;
	_itp.setData(_data);
	_iconRenderer = itc.getHostContext().getIconRenderer(name);
	_iconWidth = _rs.getPixels(ItemTablePanel.ROW_HEIGHT);
	_iconHeight = _iconWidth;
	String lookup = _name + COLUMNS;
	try {
	    _columns = _rs.getStringArray(lookup);
	    _usingResources = true;
	    setupTable();
	}
	catch (MissingResourceException mre) {
	    Log.warning("ItemTableController", 
			"Didn't find a resource with the name: \""+
			lookup + "\".  " + 
			"Defaulting to no-resource ItemTable "+
			"methods.\n" + 
			"The resource stack was: " + _rs);
	}

	// Setup the sorting
	_itp.addTableSortRequestListener(
	    new TableSortRequestListener() {
	      public void tableSortRequested(TableSortRequestEvent event) {
		  _sortColumn = event.getColumn();
		  _ascending = !event.isShiftDown();
		  sort();
		  _itp.fireTableDataChanged();
	      }
	});

	// Setup for releasing resources when our Frame goes away.
	_itp.addDestroyListener(new ItemTablePanel.DestroyListener() {
	    public void itemTableDestroyed(ItemTablePanel panel) {
		destroy();
	    }
	});
    }
    
    
    /**
     * Removes any CategoryListeners that have been installed so that
     * the object can be garbage collected.
     */
    public void destroy() {
	if (_category != null) {
	    setCategory(null);
	}   
    }
     
    /** 
     * Returns the ItemTablePanel that was passed to the constructor
     * @return an ItemTablePanel
     */
    public ItemTablePanel getItemTablePanel() {
	return _itp;
    }
    
    /**
     * Sets the Category that this ItemTableController is listening
     * to.  This category should be either the Category whose name was
     * passed to the constructor, or an Association with that Category
     * as the type of its members.
     *
     * @param category The category to use
     */
    public void setCategory(Category category) {
	Log.debug(CLASS_NAME,
		  "Set category called with category: " + category);
	if (_category != null) {
	    int numOldItems = _itemCount;
	    for (int ii = 0; ii < _itemCount; ii++) {
		_iconRenderer.removeIconListener(
		    ((Item_data)_item_DataList.elementAt(ii))
		    .item.getSelector(), _iconL);
	    }    
	    _itemCount = 0;
	    _item_DataList.removeAllElements();
	    _item_DataHash.clear();
	    setData();
            if (numOldItems > 0) {
                _itp.fireTableRowsDeleted(0, numOldItems-1);
            }
            
	    _category.removeCategoryListener(_catListener);
	}
	_category = category;
	
	if (_category != null) {
	    _category.addCategoryListener(_catListener, 
					  NotificationFilter.ALL_ITEMS);
	}
    }

    /** 
     * Setup the table according to the _columns
     */
    private void setupTable() {
	_columnCount  = _columns.length + 1;
	String[] newDisplayOrder = new String[_columnCount];
	for (int ii = 1; ii < _columnCount; ii++) {
	    newDisplayOrder[ii] = _columns[ii-1];
	}
	newDisplayOrder[0] = "ITEM_ICON";
	_columns = newDisplayOrder;
	_methods = new byte[_columnCount];
	_basedOns = new String[_columnCount];
	_sorts = new byte[_columnCount];
	_itemTableColumns = new ItemTableColumn[_columnCount];
	_categories = new String[_columnCount];
	_selectors = new String[_columnCount];
 
	_methods[0] = ICON_METHOD;
	_basedOns[0] = null;
	_sorts[0] = NO_SORT;
	_itemTableColumns[0] = new ItemTableColumn(
	    "", 
	    _iconWidth, 
	    new DefaultTableCellRenderer() {
		{
		    setBorder(new EmptyBorder(0, 0, 0, 0));
		}
		// Override getTableCellRendererComponent() to prevent
		// our base class from setting a border.  If the base
		// class sets a border, our icon gets clipped.
		public Component getTableCellRendererComponent(
		    JTable table, Object value, boolean isSelected,
		    boolean hasFocus, int row, int column) {
		    if (isSelected) {
			super.setForeground(table.getSelectionForeground());
			super.setBackground(table.getSelectionBackground());
		    } else {
			super.setForeground(table.getForeground());
			super.setBackground(table.getBackground());
		    }			
		    setValue(value);
		    return this;
		}
		public void setValue(Object value) {
		    if (value == null) {
			setIcon(null);
		    } else {		    
			setIcon(((JLabel)value).getIcon());
		    }
		}
	    }
		, null);
	_categories[0] = null;
	_selectors[0] = null;

	for (int ii = 1 ; ii < _columnCount; ii++) {
	    String key = _columns[ii];
		
	    // We look up the label in the resource file.  If it's not
	    // there, we'll just use the key as is and issue a
	    // warning.
	    String label;
	    String lookup = _name + LABEL + key;
	    try {
		label = _rs.getString(lookup);
	    }
	    catch (MissingResourceException mre) {
		// Default to using the key as the label.
		label = key;
		if (_usingResources) {
		    Log.warning("ItemTableController", 
				"Didn't find \"" + lookup + 
				"\" that was expected because \"" +
				key + "\" is in the \"" +
				COLUMNS + "\" string.");
		}
	    }
		
	    // Look up the method to use in the resource file
	    String methodStr; 
	    byte method;
	    lookup = _name + METHOD + key;
	    try {
		methodStr = _rs.getString(lookup);
		method = ResourceStack.mapStringToByte(
		    methodStr, new String[] {
			TO_STRING_METHOD_STR,
			COMPONENT_RENDERER_METHOD_STR,
			STRING_RENDERER_METHOD_STR,
			LOOKUP_METHOD_STR,
			RICH_TEXT_METHOD_STR,
			RICH_TEXT_RENDERER_METHOD_STR,
			ICON_METHOD_STR
		    }, new byte[] {
			TO_STRING_METHOD,
			COMPONENT_RENDERER_METHOD,
			STRING_RENDERER_METHOD,
			LOOKUP_METHOD,
			RICH_TEXT_METHOD,
			RICH_TEXT_RENDERER_METHOD,
			ICON_METHOD
		    });
	    }
	    catch (MissingResourceException mre) {
		if (key == NO_CODE_SELECTOR) {
		    method = RICH_TEXT_METHOD;
		} else {
		    // default to using TO_STRING_METHOD
		    method = TO_STRING_METHOD;
		}
		if (_usingResources) {
		    Log.warning(CLASS_NAME, 
				"Didn't find \"" + lookup +
				"\" that was expected because \"" +
				key + "\" is in the \"" +
				COLUMNS + "\" string.  " +
				"Defaulting to \"toString\"");
		}
	    }
	   
	    // Look up the alignment string in the resource file
	    String alignmentStr;
	    lookup = _name + ALIGNMENT + key;
	    try {
		alignmentStr = _rs.getString(lookup);
	    }
	    catch (MissingResourceException mre) {
		// default to LEFT_ALIGNMENT
		alignmentStr = LEFT_ALIGNMENT_STR;
		if (_usingResources) {
		    Log.warning(CLASS_NAME,
				"Didn't find \"" + lookup +
				"\" that was expected.  Defaulting " +
				"to \"" + LEFT_ALIGNMENT_STR + "\".");
		}
	    }
	    int alignment = JLabel.LEFT;;
	    if (alignmentStr.equals(LEFT_ALIGNMENT_STR)) {
		alignment = JLabel.LEFT;
	    } else if (alignmentStr.equals(CENTER_ALIGNMENT_STR)) {
		alignment = JLabel.CENTER;
	    } else if (alignmentStr.equals(RIGHT_ALIGNMENT_STR)) { 
		alignment = JLabel.RIGHT;
	    } else if (_usingResources) {
		Log.warning(CLASS_NAME,
			    "Resource \"" + lookup +
			    "\" wasn't one of the legal values: " +
			    "\"" + LEFT_ALIGNMENT_STR + "\", " +
			    "\"" + CENTER_ALIGNMENT_STR + "\", or" +
			    "\"" + LEFT_ALIGNMENT_STR + "\".  " +
			    "Defaulting to \"" + LEFT_ALIGNMENT_STR +
			    "\".");
	    }
		
	    // Setup the render based on the method
	    TableCellRenderer renderer;
	    if (method == TO_STRING_METHOD ||
		method == LOOKUP_METHOD ||
		method == STRING_RENDERER_METHOD) {
		renderer = new DefaultTableCellRenderer();
		((DefaultTableCellRenderer)renderer).
		    setVerticalTextPosition(JLabel.TOP);
		((DefaultTableCellRenderer)renderer).
		    setVerticalAlignment(JLabel.TOP);
		((DefaultTableCellRenderer)renderer).
		    setHorizontalAlignment(alignment);
	    } else if (method == COMPONENT_RENDERER_METHOD ||
		       method == RICH_TEXT_RENDERER_METHOD ||
		       method == RICH_TEXT_METHOD) {
		renderer = new TableCellComponentRenderer();
	    } else if (method == ICON_METHOD) {
		renderer = new DefaultTableCellRenderer() {
		    public void setValue(Object value) { 
			setIcon((Icon)value); 
		    };
		};
	    } else {
		// It's a good time to warn about invalid methods.
		if  (_usingResources) {
		    Log.warning(CLASS_NAME,"The " + _name + METHOD + key +
				" string was not one of the expected " +
				"values of: \"" +
				TO_STRING_METHOD_STR + "\", \"" +
				LOOKUP_METHOD_STR + "\", \"" +
				STRING_RENDERER_METHOD_STR + "\", \"" +
				COMPONENT_RENDERER_METHOD_STR + "\", \"" +
				RICH_TEXT_RENDERER_METHOD_STR + "\", \"" +
				ICON_METHOD_STR + "\", or \"" +
				RICH_TEXT_METHOD_STR + "\".  Defaulting to \"" +
				TO_STRING_METHOD_STR + "\".");
		    method = TO_STRING_METHOD;
		}
		renderer = new DefaultTableCellRenderer();
		((DefaultTableCellRenderer)renderer).
		    setVerticalTextPosition(JLabel.TOP);
		((DefaultTableCellRenderer)renderer).
		    setVerticalAlignment(JLabel.TOP);
	    }
	    if (method == RICH_TEXT_METHOD) {
		lookup = _name + CATEGORY + key;
		try {
		    _categories[ii] = _rs.getString(lookup);
		} catch (MissingResourceException mre) {
		    if (key != NO_CODE_SELECTOR) {
			Log.fatal("\"" + _name + METHOD + key  + 
				 "\" was set to \"" + 
				 RICH_TEXT_METHOD_STR + "\" " +
				 "but \"" + lookup +  
				 "\" was not set.");
		    }
		    _categories[ii] = null;
		} 
		lookup = _name + SELECTOR + key;
		String itemSelector = null;
		try {
		    _selectors[ii] = _rs.getString(lookup);
		} catch (MissingResourceException mre) {
		    if (key != NO_CODE_SELECTOR) { 
			Log.fatal("\"" + _name + METHOD + key   
				 + "\" was set to \"" + 
				 RICH_TEXT_METHOD_STR + "\" " +
				 "but \"" + lookup +  
				 "\" was not set.");
		    }
		    _selectors[ii] = null;
		    }
	    }
	    _methods[ii] = method;
		
	    String basedOn;
	    lookup = _name + BASED_ON + key;
	    basedOn = _rs.getString(lookup, key);
	    _basedOns[ii] = basedOn;
		
	    int width;
	    width = _rs.getPixels(new String[] {_name + WIDTH + key,
						DEFAULT_WIDTH});
	    byte sort;
	    lookup = _name + SORT + key;
	    try {
		sort = ResourceStack.mapStringToByte(
		    _rs.getString(lookup), new String[] { LEXICAL_SORT_STR,
							  NUMERIC_SORT_STR,
							  NO_SORT_STR,
							  SORTER_SORT_STR},
		    new byte[] {
			LEXICAL_SORT,
			NUMERIC_SORT ,
			NO_SORT ,
			SORTER_SORT}
		    );
	    }
	    catch (MissingResourceException mre) {
		String error = "The resource " + lookup + 
				" was not found.  Defaulting to ";
		if (method == ICON_METHOD 
		    || method == COMPONENT_RENDERER_METHOD) {
		    if (_usingResources) {
			Log.warning(CLASS_NAME, error + NO_SORT_STR);
		    }
		    sort = NO_SORT;
		} else {
		    if (_usingResources) {
			Log.warning(CLASS_NAME, error + LEXICAL_SORT_STR);
		    }
		    sort = LEXICAL_SORT;
		}
	    }

	    if ((sort == LEXICAL_SORT ||
		 sort == NUMERIC_SORT) &&
		(method == ICON_METHOD 
		 || method == COMPONENT_RENDERER_METHOD)) {
		String methodStr2 = method == ICON_METHOD ?
		    ICON_METHOD_STR : COMPONENT_RENDERER_METHOD_STR;
		Log.warning(CLASS_NAME, "\"" + _name + METHOD + key + 
			    "\" was \"" + methodStr2 + " and \"" +
			    lookup + "\" was \"" + 
			    sort + "\".  These are incompatible.  " +
			    "You must use \"" + NO_SORT_STR + "\" or \"" +
			    SORTER_SORT_STR + "\" on columns with that " +
			    "method.");
		sort = NO_SORT;
	    }
	    _sorts[ii] = sort;
	    _itemTableColumns[ii] = 
		new ItemTableColumn(label, width, renderer, null);
	}
	_itp.setupColumns(_itemTableColumns);
    }

    /**
     * Returns an object representing the information to display in a
     * particular column for a particular item.
     * @param item The item to use
     * @param column The column of interest
     * @return An Object that will be used by the Table to display the
     *         information.
     */
    private Object getCellForItem(Item item, int column) {
	if (column == 0) {
	    // handle the icon seperately
	    return null;
	}
	
	// Determine what method to use:
	String display = _columns[column];
	byte method = _methods[column];
	String basedOn = _basedOns[column];
	Attribute attr = item.getAttr(basedOn);

	String missing = null;
	if (attr == null && 
	    (method == LOOKUP_METHOD || method == TO_STRING_METHOD ||
	     method == RICH_TEXT_METHOD)) {
	    String lookup = _name + MISSING + display;
	    try {
		missing = _rs.getString(lookup);
	    } catch (MissingResourceException mre) {
		if (_usingResources) {
		    Log.warning(CLASS_NAME,"There is no attribute: \"" +
				basedOn + 
				"\" in the Item with selector: \"" +
				item.getSelector() + 
				"\".  Will use blanks/defaults " +
				"to represent this attribute, since \"" + 
				lookup + "\" was not set");
		}
		missing = "";
	    }
	}
	String methodLookup = _name + METHOD + display;

	if (method == LOOKUP_METHOD) {
	    if (attr != null) {
		String val = attr.getValueString();
		// If we can't find a particular value, fall
		// back to just using the attr's value.
		// Don't issue a warning here because
		// LOOKUP_METHOD could be used to only provide
		// string for some possible values, such as
		// "Zero" for 0.
		return _rs.getString(_name + LOOKUP + display + "." + val, val);
	    } else {
		return missing;
	    }
	} else if (method == STRING_RENDERER_METHOD) {
	    Log.assert(_itcr != null,
		       "Column \"" + display + "\" is using the " +
		       STRING_RENDERER_METHOD_STR +
		       " method, but no ItemTableColumnRenderer has been set.");
	    return _itcr.getStringForCellOfItemTable(item,display,_itc);

	} else if (method == COMPONENT_RENDERER_METHOD) {
	    Log.assert(_itcr != null,
		       "Column \"" + display + "\" is using the " + 
		       COMPONENT_RENDERER_METHOD_STR +
		       " method, but no ItemTableColumnRenderer has been set.");
	    return _itcr.getComponentForCellOfItemTable(item,display, _itc);
	} else if (method == ICON_METHOD) {
	    String iconLookup = _name + ICON + display + "." + attr.getValueString();
	    String defaultLookup = _name + DEFAULT_ICON + display;
	    
	    if (attr != null) {
		try {
		    return _rs.getIcon(
			new String[] {iconLookup, defaultLookup});
		} catch (MissingResourceException mre) {
		    String icon;
		    if ((icon = _rs.getString(iconLookup, null)) != null) {
			Log.fatal("The resource \"" + iconLookup +
				  "\" is set to: \"" + icon + "\", but " +
				  "that icon couldn't be loaded.");
		    } else if ((icon = _rs.getString(defaultLookup, null)) != null) {
			Log.fatal("The resource \"" + defaultLookup +
				  "\" is set to: \"" + icon + "\", but " +
				  "that icon couldn't be loaded.");
		    } else {
			Log.fatal("\"" + methodLookup + "\" was set to \"" +
				  ICON_METHOD_STR + "\", but " + "neither \"" + 
				  iconLookup + "\" nor \"" +
				  defaultLookup + "\" were set.");
		    }
		}
	    } else {
		try {
		    return _rs.getIcon(defaultLookup);
		} catch (MissingResourceException mre) {
		    Log.fatal("\"" + methodLookup + "\" " +
			     "was set to \"" + ICON_METHOD_STR + "\", and " +
			     "the attribute \"" + basedOn +
			     "\" was not found, and " + defaultLookup 
			     + "\" was not set.");
		}
	    }
	} else if (method == RICH_TEXT_RENDERER_METHOD || 
		   method == RICH_TEXT_METHOD) {
	    String rt; // The rich Text to display
	    if (method == RICH_TEXT_RENDERER_METHOD) {  
		Log.assert(_itcr != null,
			   "Column \"" + display + "\" is using the " +
			   RICH_TEXT_RENDERER_METHOD_STR +
			   " method, but no ItemTableColumnRenderer has been set.");
		rt = _itcr.getRichTextForCellOfItemTable(item, display, _itc);
	    } else {
		if (attr != null) {
		    String itemSelector = _selectors[column];
		    String category = _categories[column];
		    if (itemSelector.equals("ITEM_SELECTOR")) {
			itemSelector = item.getSelector();
		    } else {
			itemSelector = item.getString(itemSelector);
		    }
		    String attrString =  attr.getValueString();
		    rt = "<A HREF=" + RichTextComponent.quoteString(
			ItemView.createURLToLaunch(
			    category, 
			    itemSelector)) +  
			">" + RichTextComponent.quoteString(attrString) + 
			"</A>";
		} else {
		    if (display == NO_CODE_SELECTOR) {
			String category = item.getType();
			String selector = item.getSelector();
			rt = "<A HREF=" + RichTextComponent.quoteString(
			    ItemView.createURLToLaunch(
				category,
				selector)) +
			    ">" + RichTextComponent.quoteString(selector) + 
			    "</A>";
		    } else {
			rt = missing;
		    }
		}
	    }
	    RichTextComponent rtc =  new RichTextComponent(display, _rs, rt);
	    // Make the richText component have no margins so it
	    // matches the way the DefaultCellRenderer looks.
	    rtc.setMargins(new Insets(0,0,0,0));
	    rtc.setOpaque(true);
	    rtc.setAutoWrap(false);
	    rtc.addLinkListener(new LinkListener() {
		public void linkActivated(LinkEvent event) {
		    launchItemView(event.getTarget());
		}
	    });
	    return rtc;
	}
	
	if (method != TO_STRING_METHOD) {
	    Log.warning(CLASS_NAME,
			"Didn't recognize method \"" + 
			method + "\".  "+
			"Defaulting to toString.");
	}
	if (attr != null) {
	    return attr.getValueString();
	} else {
	    return missing;
	}
    } 
 
    private void launchItemView(String url) {
	ItemViewLaunchRequestEvent launchEvent = new
	    ItemViewLaunchRequestEvent(
		this,
		ItemView.getCategoryFromURL(url),
		ItemView.getItemFromURL(url));
	int numListeners =  _itemViewLaunchRequestListeners.size();
	if (numListeners == 0) {
	    UIContext uic = new UIContext();
	    uic.setHostContext(_hc);
	    ItemViewFrame.launchItemViewFrame(launchEvent,uic);
	} else {
	    for (int ii = 0 ; ii < numListeners ; ii++) {
		((ItemViewLaunchRequestListener)
		 _itemViewLaunchRequestListeners.elementAt(ii)).
		    itemViewLaunchRequested(launchEvent);
	    }
	}
    }
    

    /** 
     * Compares two items.  Used by algorithms that sort items.
     * @param item1 The first item
     * @param item2 The second item
     * @param column The column of the table to base the sort on.
     * @return < 0 if item1 < item 2, 0 if item1 == item2, > 0 if item1 > item2
     */
    private int compareItems(Item item1, Item item2, int column){
	byte method = _methods[column];
	byte sort = _sorts[column];
	if (sort == NO_SORT) {
	    return 0;
	}
	if (sort == SORTER_SORT) {
	    return 
		_itcr.compareItemsForItemTable(item1, item2, _columns[column]);
	    }
	if (method == TO_STRING_METHOD ||
	    method == LOOKUP_METHOD ||
	    method == RICH_TEXT_METHOD ||
	    method == RICH_TEXT_RENDERER_METHOD ||
	    method == STRING_RENDERER_METHOD) {
	    String one, two;
	    Object obj1, obj2;
	    obj1 = ((Vector)_item_DataHash.get(item1.getSelector())).
		elementAt(column);
	    obj2 = ((Vector)_item_DataHash.get(item2.getSelector())).
		elementAt(column);
	    if (method == RICH_TEXT_METHOD ||
		method == RICH_TEXT_RENDERER_METHOD) {
		one = RichTextComponent.removeTags(
		    ((RichTextComponent)obj1).getText());
		two = RichTextComponent.removeTags(
		    ((RichTextComponent)obj2).getText());
	    } else {
		one = (String)obj1;
		two = (String)obj2;
	    }   
	    if (sort == NUMERIC_SORT) {
		try {
		    int int1 = Integer.parseInt(one);
		    int int2 = Integer.parseInt(two);
		    if (int1 < int2) {
			return -1;
		    } else if (int1 > int2) {
			return 1;
		    } else {
			return 0;
		    }
		}
		catch (NumberFormatException nfe) {
		    Log.warning(CLASS_NAME,
				"The sort method for " +
				_columns[column] + "was \"" +
				NUMERIC_SORT + "\" but " +
				one + " and " + two + 
				" couldn't be sorted numerically.");
		    return 0;
		}
	    }
	    return collator.compare(one,two);
	}
	// We should never get here
	Log.fatal("Logic botch in compareItem.  Method was " + method
		 + "and sort was " + sort + ".");
	return 0;  // make the compiler happy
    }
   
    /**
     * A convenience method for getting an item out of the Vector
     * @return the Item
     * @param item_dataList a Vector of Item_data objects
     * @param index the index of the Vector to look at
     */
    private static Item getItem(Vector item_dataList, int index) {
	return ((Item_data)item_dataList.elementAt(index)).item;
    }

    /**
     * A convenience method for extracting the current data from the
     * item_DataList array and pushing it into the _data array so that the
     * TableModel can use it.
     */
    private void setData() {
	_data.setSize(_item_DataList.size());
	for (int ii = 0 ; ii < _itemCount ; ii++) {
	    _data.setElementAt(((Item_data)_item_DataList.elementAt(ii)).data,
			       ii);
	}
    }

    /** 
     * Adds an ItemViewLaunchRequestListener.  The listener will be
     * called when a user clicks on a URL whose action is to launch an
     * ItemView.  If no listeners are added, the default behavior is
     * to launch the ItemView in a new ItemViewFrame.
     *
     * @param listener the listener to add
     */
    public void addItemViewLaunchRequestListener(ItemViewLaunchRequestListener
						 listener) {
	_itemViewLaunchRequestListeners.addElement(listener);
    }
    
    /** Removes an ItemViewLaunchRequestListener
     * @param listener the listener to remove
     */
    public void removeItemViewLaunchRequestListener(
	ItemViewLaunchRequestListener listener) {
	_itemViewLaunchRequestListeners.removeElement(listener);
    }
    

    

//-------------Category Listener Methods ---------------

     
  
    private IconL _iconL = new IconL();
    private class IconL implements RenderedObjectListener {
	public void renderedObjectChanged(String iconSelector, Object object) {
	    final Vector row = (Vector)_item_DataHash.get(iconSelector);
	    Log.assert(row != null, "Row wasn't found.  " +
		       "selector was: " + iconSelector);
	    JLabel iconLabel = new JLabel((Icon)object);
	    iconLabel.addMouseListener(new MouseAdapter() {
		int _currentLinkState;
		public RichTextComponent getRTC(MouseEvent e) {
		    Object col1 = row.elementAt(1);
		    if (col1 instanceof RichTextComponent) {
			RichTextComponent rtc = ((RichTextComponent)col1);
			if (rtc.getNumLinks() != 0) {
			    return rtc;
			}
		    }
		    return null;
		}
		
		public void mousePressed(MouseEvent e) {
		    RichTextComponent rtc = getRTC(e);
		    if (rtc != null) {
			rtc.setLinkState(0, 
					 RichTextComponent.LINK_STATE_ACTIVE);
		    }
		}

		public void mouseReleased(MouseEvent e) {
		    RichTextComponent rtc = getRTC(e);
		    if (rtc != null) {
			rtc.setLinkState(0, _currentLinkState);
			for (int ii = 0 ; ii < _itemCount; ii++) {
			    if (((Item_data)_item_DataList.elementAt(ii)).data == row) {
				_itp.fireTableCellChanged(ii,1);
				break;
			    }
			}
		    }
		}
		
		public void mouseClicked(MouseEvent e) {
		    RichTextComponent rtc = getRTC(e);
		    if (rtc != null) {
			_currentLinkState =
			    RichTextComponent.LINK_STATE_VISITED;
			rtc.setLinkState(0, 
					 RichTextComponent.LINK_STATE_VISITED);
			launchItemView(rtc.getLinkTarget(0));
		    }
		}
		
		public void mouseEntered(MouseEvent e) {
		    RichTextComponent rtc = getRTC(e);
		    if (rtc != null) {
			_currentLinkState = rtc.getLinkState(0);
		    }
		}
		
		
		public void mouseExited(MouseEvent e) {
		    RichTextComponent rtc = getRTC(e);
		    if (rtc != null) {
			rtc.setLinkState(0, _currentLinkState);
			for (int ii = 0 ; ii < _itemCount; ii++) {
			    if (((Item_data)_item_DataList.elementAt(ii)).data == row) {
				_itp.fireTableCellChanged(ii,1);
				break;
			    }
			}
		    }
		}
	    });
	    row.setElementAt(iconLabel, 0);
	    if (!_inChangeBlock) {
		// figure out which row changed
		for (int ii = 0 ; ii < _itemCount; ii++) {
		    if (((Item_data)_item_DataList.elementAt(ii)).data == row) {
			_itp.fireTableCellChanged(ii,0);
			break;
		    }
		}
	    }
	}
    } 
    
    private class CatListener extends CategoryAdapter {
	/**
     * Called by Category after an Item has been added or 
     * detected at startup to the Category.
     * 
     * @param item The item that was added.
     */
	public void itemAdded(Item item) {
	    Log.debug(CLASS_NAME,
		      "itemAdded: " + item.getSelector());
	    if (_columns == null) {
		// This means that we're a no-code item table and this is
		// the first item.  Use its attributes to setup the
		// table.
		_columns = new String[item.size()+1];
		_columns[0] = NO_CODE_SELECTOR;
		Enumeration enum = item.getAttrEnum();
		int counter = 1;
		while (enum.hasMoreElements()) {
		    _columns[counter++] =
			((Attribute)enum.nextElement()).getKey();
		}
		setupTable();
	    }
	    _itemCount++;
	    Vector row = new Vector(_columnCount);
	    for (int ii = 0 ; ii < _columnCount ; ii++) {
		row.addElement(getCellForItem(item,ii));
	    }
	    _item_DataList.addElement(new Item_data(item,row));
	    _item_DataHash.put(item.getSelector(), row);
	    if (!_inChangeBlock) {
		sort();
		for (int ii = 0 ; ii < _itemCount; ii++) {
		    if (getItem(_item_DataList,ii) == item) {
			_itp.fireTableRowInserted(ii);
			break;
		    }
		}
	    }   
	    _iconRenderer.addIconListener(item.getSelector(),
					  null,
					  _iconWidth,
					  _iconHeight,
					  _iconL);
	
	}

	/**
	 * Called by Category when an Item changes. 
	 * 
	 * @param oldItem The old item.
	 * @param newItem The new item.
	 */
	public void itemChanged(Item oldItem, Item newItem){
	     Log.debug(CLASS_NAME,
		      "itemChanged: " + newItem.getSelector());
	    Vector row = new Vector(_columnCount);
	    Vector oldRow = (Vector)_item_DataHash.get(oldItem.getSelector());
	    row.addElement(oldRow.elementAt(0));
	    for (int ii = 1; ii < _columnCount ; ii++) {
		row.addElement(getCellForItem(newItem,ii));
	    }
	    boolean found = false;
	    for (int ii = 0 ; ii < _itemCount ; ii++) {
		Item_data item_data =(Item_data)_item_DataList.elementAt(ii);
		if (item_data.item == oldItem) {
		    item_data.item = newItem;
		    item_data.data = row;
		    found = true;
		    break;
		}
	    }
	    _item_DataHash.put(newItem.getSelector(), row);
	    
	    Log.assert(found, "item wasn't found in itemChanged");
	    if (!_inChangeBlock) {
		sort();
		for (int ii = 0 ; ii < _itemCount; ii++) {
		    if (getItem(_item_DataList,ii) == newItem) {
			_itp.fireTableRowUpdated(ii);
			break;
		    }
		}
	    }
	}
	
	/**
	 * Called by Category when an Item is removed.
	 * 
	 * @param item The item that was removed from the list of items
	 */
	public void itemRemoved(Item item) {
	    Log.debug(CLASS_NAME,
		      "itemRemoved: " + item.getSelector());
	    _iconRenderer.removeIconListener(item.getSelector(),
					     _iconL);
	    int position = -1;
	    for (int ii = 0 ; ii < _itemCount ; ii++) {
		if (getItem(_item_DataList, ii) == item) {
		    position = ii;
		    break;
		}
	    }
	    _item_DataHash.remove(item.getSelector());
	    Log.assert(position >= 0, 
		       "itemRemoved couldn't find item in list!");
	    _itemCount--;
	    _item_DataList.removeElementAt(position);
	    if (!_inChangeBlock) {
		// It's necessary to notify the table that a row has been
		// deleted before we actually delete it.
		_itp.fireTableRowDeleted(position);
		setData();
	    }
	}
	
	/**
	 * Called by Category prior to a block of
	 * of changes.  
	 */
	public void beginBlockChanges(){
	    _inChangeBlock = true;
	}   
	
	/**
	 * Called by Category after a block of changes.
	 */
	public void endBlockChanges() {
	    _inChangeBlock = false;
	    sort();
	    _itp.fireTableDataChanged();
	}
    }
    
    /**
     * Returns the Items that are selected.  There could be zero, one,
     * or multiple Items selected.
     *
     * @return An Item[] of the selected Items.  If no Items are
     * selected, an empty Item[] will be returned.
     */
    public Item[] getSelectedItems() {
	int[] rows = _itp.getSelectedRows();
	Item[] returnVal = new Item[rows.length];
	for (int ii = 0; ii < rows.length; ii++) {
	    returnVal[ii] =
		((Item_data)_item_DataList.elementAt(rows[ii])).item;
	}
	return returnVal;
    }
  
    
//--------------------sorting code------------------


    /**
     * A method that sorts the data in _itemDataList on the column
     * _sortColumn and influenced by _ascending
     */
    private void sort() {
	Object[] array = new Object[_itemCount];
	_item_DataList.copyInto(array);
	Sort.stable_sort(array, 0, _itemCount, new BinaryPredicate() { 
	    public boolean apply(Object x, Object y) {
		int result = compareItems(((Item_data)x).item,
					  ((Item_data)y).item,
					  _sortColumn);
		if (_ascending) {
		    return result < 0;
		} else {
		    return result > 0;
		}
	    }
	}); 
	for (int ii = 0 ; ii < _itemCount ; ii++) {
	    _item_DataList.setElementAt(array[ii],ii);
	}
	setData();
    }
}
