//
// ItemTableFactory
//
//	A class that knows how to construct an ItemTable
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import com.sgi.sysadm.category.*;
import com.sgi.sysadm.util.*;
import java.util.*;
import java.text.*;

/**
 * A class that knows how to construct an ItemTable.  If you don't want
 * to use the default ItemTablePanel or ItemTableController, you must
 * subclass this class.  You <B>do not</B>
 * need to subclass this class to get an ItemTable that uses an
 * ItemTableColumnRenderer.
 * See the description of getItemTableColumnRenderer() for more information.
 */
public class ItemTableFactory implements ItemTableProperties {

    /** 
     * The ItemTableContext that was passed to the constructor
     */
    private ItemTableContext _itemTableContext;
 
    /**
     * The fully qualified name of the Category this ItemTable is
     * displaying 
     */
    private String _fullCategoryName;

    /**
     * The name of the category that this ItemTable is displaying
     */
    private String _categoryName;

    /**
     * The HostContext that was passed in the ItemTableContext to the
     * constructor.
     */
    private HostContext _hostContext;


    private static final String CLASS_NAME = "ItemTableFactory";

    /**
     * Constructor
     *
     * @param categoryName The fully-qualified name of the category
     *                     that contains the item you wish to view.
     * @param ItemTableContext The ItemTableContext to use
     */
    public ItemTableFactory(String categoryName, 
			    ItemTableContext ItemTableContext) {
	_fullCategoryName = categoryName;
	if (categoryName == null) {
	    throw new IllegalArgumentException(
		"Can't call ItemTableFactory constructor with categoryName null");
	}
	if (ItemTableContext == null) {
	    throw new IllegalArgumentException(
		"Can't call ItemTableFactory constructor with ItemTableContext null");
	}
	_hostContext = ItemTableContext.getHostContext();
	_categoryName = ResourceStack.getClassName(categoryName);
	_itemTableContext = ItemTableContext;
    } 
	
    /** 
     * Creates an ItemTable using the results of the various get()
     * methods and the values passed into the constructor.
     *
     * @return An ItemTable
     * @exception ItemTableException if there is an error while
     *            creating the ItemTable.
     */
    public ItemTable createItemTable() throws ItemTableException{
	ItemTableController controller = getItemTableController();
	ItemTablePanel ivp = controller.getItemTablePanel();
	_itemTableContext.setDialogParent(ivp);
	return new ItemTable(controller,  _itemTableContext,
			     _fullCategoryName);
    }


    /**
     * Returns the ItemTableController to use in the ItemTable.  By
     * default, this method returns the ItemTableController base class
     * that is constructed by passing <tt>categoryName</tt> and
     * <tt>ItemTableContext</tt> (as passed to the ItemTableFactory
     * constructor) to the  constructor of ItemTableController.
     * Override this method to use  your own ItemTableController.
     *
     * @exception ItemTableException if there is an error while
     *            creating the ItemTable.
     */
    protected ItemTableController getItemTableController() 
	throws ItemTableException{
	return new ItemTableController(getItemTablePanel(),
				       _fullCategoryName,
				       _itemTableContext,
				       getItemTableColumnRenderer());
    }
    
    /** 
     * Returns the ItemTablePanel to use for the ItemTable.  By
     * default, this method returns the ItemTablePanel base class that
     * is constructed by passing <tt>ItemTableContext</tt> (as passed to the
     * ItemTableFactory constructor) to the constructor of
     * ItemTablePanel.  Override this method to use your own
     * ItemTablePanel subclass.
     *
     * @exception ItemTableException if there is an error while
     *            creating the ItemTable.
     */
    protected ItemTablePanel getItemTablePanel()
	throws ItemTableException {
	return new ItemTablePanel(_itemTableContext);
    }
     
    /**
     * Loads a class by first checking the resource stack for the
     * class' name, then by appending the appendStr to the
     * _fullCategoryName.
     * 
     * @param resourceLookupStr The key to lookup in the resourceStack
     *                          for the class name.
     * @param appendString The string to append to the
     *                     _fullCategoryName to create a class name.
     * @param returnType The type of the object to return.  If the
     *                   loaded class is not an instance of this type,
     *                   an exception will be thrown.
     *
     * @exception ItemTableException Thrown if the class specified by
     *            the resource can't be located, or if a class is
     *            located but can't be instantiated for some reason.
     */
    private Object loadRenderer(String resourceLookupStr,
				String appendStr,
				Class returnType)  
	throws ItemTableException { 
	ResourceStack rs = _itemTableContext.getResourceStack();
	String rendererClassName = rs.getString(resourceLookupStr,
						_fullCategoryName + appendStr);
	try {
	    return SysUtil.createObject(rendererClassName,
					returnType, null, null);
	} catch (ClassNotFoundException cnfe) {
	    if (!rendererClassName.equals(_fullCategoryName + appendStr)) {
		// if we found a class in the resource stack but it
		// wasn't there
		Log.warning(CLASS_NAME, "The ResourceStack specified that \"" +
			    rendererClassName + "\" was to be used as a " +
			    "renderer, but that class was not found.");
	    }
	    return null;
	}   
	catch (SysUtil.ClassLoadException cle) {
	    throw new ItemTableException(MessageFormat.format(
		rs.getString(ItemView.CANT_LOAD_CLASS),
		new Object[] {rendererClassName, returnType.getName()}));
	}
    }
    
    /**
     * Returns the ItemTableColumnRenderer to use with this
     * ItemTable.  By default, this method first checks to see if there
     * is a resource specifying the class to use
     * (COLUMN_RENDERER).  If this resource is not found, this
     * method attempts to load a class with the name
     * <i>{categoryName}CategoryRenderer</i>  If that class is not
     * found, this method returns null.
     *
     * @exception ItemTableException Thrown if the class specified in
     *            the resource file does not exist, or if the class
     *            cannot be loaded.
     */
    protected ItemTableColumnRenderer getItemTableColumnRenderer()
	throws ItemTableException{
	return (ItemTableColumnRenderer)
	    loadRenderer(_fullCategoryName + COLUMN_RENDERER, 
			 "ColumnRenderer",
			 ItemTableColumnRenderer.class);
    }
}
