//
// ItemTableFrame
//
//	A class that holds an ItemTablePanel, buttons, and menu in a JFrame
//
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.event.*;

/**
 * A subclass of RFrame that holds an ItemTable.  The ItemTable will
 * occupy the majority of the Frame, with an OK button at the bottom.
 * The ItemTableFrame adds a TitleListener to
 * the ItemTable and keeps the frame's title up to date.
 */
public class ItemTableFrame extends RFrame implements ItemTableProperties {

    private ItemTableContext _itc;
    private ItemTable _it;

    /**
     * A constructor that is used when you need the frame, but don't
     * yet know the ItemTable.  You can call setItemTable later to set
     * the ItemTable that gets displayed.
     */
    public ItemTableFrame() {
	setDefaultCloseOperation(DISPOSE_ON_CLOSE);
	addWindowListener(new WindowAdapter() {
	    public void windowClosed(WindowEvent e) {
		if (_it != null) {
		    _it.destroy();
		    _it = null;
		}
	    }
	});
    }

    /**
     * A constructor that is used when you have the ItemTable to
     * display
     *
     * @param itemTable The ItemTable to display in the frame.
     */
    public ItemTableFrame(ItemTable itemTable) {
	this();
	setItemTable(itemTable);
    }

    /**
     * Sets the ItemTable that is displayed in this frame.  It is okay
     * to call this method multiple times with different ItemTables.
     *
     * @param itemTable The ItemTable to display in the frame.
     */
    public void setItemTable(ItemTable itemTable) {
	if (_it != null) {
	    _it.destroy();
	}
	_it = itemTable;
	_itc = itemTable.getItemTableContext();
	ResourceStack rs = _itc.getResourceStack();
	int vgap = rs.getPixels(VGAP);
	getContentPane().setLayout(new BorderLayout(0,vgap));
	JPanel buttonPanel = new JPanel();
	JButton button = new JButton(rs.getString(OK));
	buttonPanel.add(button);
	button.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		setVisible(false);
		dispose();
	    }
	});
	getContentPane().add(buttonPanel, BorderLayout.SOUTH);
	getContentPane().add(itemTable.getItemTablePanel(),BorderLayout.CENTER);
	itemTable.addTitleListener(new TitleListener() {
	    public void titleChanged(TitleEvent event) {
		setTitle(event.getTitle());
	    }
	});
    }

    /**
     * A static method to launch an ItemTable in a new Frame.
     *
     * @param event An ItemTableLaunchRequestEvent containing
     * information about the ItemTable to show.
     * @param uic The UIContext over which to display the busy cursor
     *            while the ItemTableFrame is being launched.
     */
    public static void launchItemTableFrame(
	final ItemTableLaunchRequestEvent event,
	final UIContext uic) {
	uic.busy(new ResultListener() {
	    public void succeeded(ResultEvent ev) {
		try {
		    String categoryName = event.getCategorySelector();
		    String key;
		    HostContext hc = uic.getHostContext();
		    if ((key = runOnce(categoryName, hc)) != null) {
			ItemTable it =
			    ItemTable.createItemTable(hc,
						      categoryName);
			launchIT(it, hc, categoryName, key);
		    }
		} catch (ItemTableException ite) {
		    uic.postError(ite.getMessage());
		}
		uic.notBusy();
	    }
	    public void failed(ResultEvent event) {
	    }
	});
    }

    /**
     * A static method to launch an ItemTable in a new frame.
     * @param iv The itemTable to show
     * @param uic The UIContext over which to display the busy cursor
     *            while the ItemTableFrame is being launched.
     */
    public static void launchItemTableFrame(final ItemTable it,
					    final UIContext uic) {
	final String key;
	final String catName = it.getCategoryName();
	final HostContext hc = uic.getHostContext();
	if ((key = runOnce(catName, hc)) != null) {
	    uic.busy(new ResultListener() {
		public void succeeded(ResultEvent event) {
		    launchIT(it, hc, catName, key);
		    uic.notBusy();
		}
		public void failed(ResultEvent event) {
		    uic.notBusy();
		}
	    });
	}
    }

    private static void launchIT(ItemTable it, final HostContext hc,
				 String categoryName, String key) {
	ItemTableFrame frame = new ItemTableFrame(it);
	final HostContext.Client itemTableClient =
	    new HostContext.Client(frame, it.getItemTableContext());
	hc.registerClient(itemTableClient, key);
	frame.setDefaultCloseOperation(
	    WindowConstants.DISPOSE_ON_CLOSE);
	frame.addWindowListener(new WindowAdapter() {
	    public void windowClosed(WindowEvent event) {
		hc.unregisterClient(itemTableClient);
	    }
	});
	frame.pack();
	frame.setVisible(true);
    }

    /**
     * Pops the IT to the front and returns null if an IT was
     * already running.  Otherwise, returns the run once key.
     */
    private static String runOnce(String categoryName, HostContext hc) {

	// Run once logic: Use category name; if HostContext already
	// knows about a corresponding ItemTable, use it.
	Vector vec = hc.getClients(categoryName);
	if (vec.size() > 0) {
		((HostContext.Client)vec.elementAt(0)).getFrame().toFront();
	    return null;
	}
	return categoryName;
    }
}
