//
// ItemTablePanel.java
//
//	A JPanel that displays an ItemTable
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import java.awt.*;
import java.awt.event.*; 
import java.util.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.event.*;
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;

/** 
 * This class uses a ComponentPanel and an associated TabelModel to display a
 * ItemTable
 */
public class ItemTablePanel extends JPanel implements ExtraCleanup,
    ItemTableProperties {
    
    private static final String CLASS_NAME = "ItemTablePanel";
    private ItemTableModel _tm;
    private ComponentTable _table;
    private UIContext _uic;
    private Vector _launchListenerList = new Vector(); 
    private Item _item;
    private JScrollPane _scrollpane;
    private int _numRows;
    private int _totalColumnWidth = 0;
    private int _numColumns;
    private Vector _destroyListeners = new Vector();

    /*
     * Interface for notifying ItemTableController when category
     * listener should be removed.
     */
    /*package*/ interface DestroyListener {
	void itemTableDestroyed(ItemTablePanel panel);
    }

    /** 
     * Constructor
     *
     * @param uic The UIContext that the Panel can use to display
     *           messages.
     */
    public ItemTablePanel(UIContext uic) {
        // Set backgound color to the Table's background color to work
        // around a Swing 1.1.1 bug.
        setBackground((Color)UIManager.getLookAndFeelDefaults().get("Table.background"));
	setLayout(new GridLayout(1, 1));
	_uic = uic;
	ResourceStack rs = uic.getResourceStack();
	_numRows = rs.getInt(NUM_ROWS);
	_table  = new ComponentTable(_tm = new ItemTableModel());
	_tm.addTableModelListener(_table);
	_scrollpane = new JScrollPane(_table);
	_scrollpane.setVerticalScrollBarPolicy(
	    ScrollPaneConstants.VERTICAL_SCROLLBAR_ALWAYS);

        // Because we set the panel's background to white, the corner
        // shows up white, but we want it gray.
        _scrollpane.setCorner(JScrollPane.UPPER_RIGHT_CORNER,new JPanel());
	add(_scrollpane);
	String selectionStr = rs.getString(SELECTION_MODE);
	int selectionType;
	if (selectionStr.equals("SINGLE_SELECTION")) {
	    selectionType = ListSelectionModel.SINGLE_SELECTION;
	} else if (selectionStr.equals("SINGLE_INTERVAL_SELECTION")) {
	    selectionType = ListSelectionModel.SINGLE_INTERVAL_SELECTION;
	} else if (selectionStr.equals("MULTIPLE_INTERVAL_SELECTION")) {
	    selectionType = ListSelectionModel.MULTIPLE_INTERVAL_SELECTION;
	} else {
	    Log.warning(CLASS_NAME, selectionStr + 
			" is not a valid selection mode.  "+
			"Defaulting to SINGLE_SELECTION.");
	    selectionType = ListSelectionModel.SINGLE_SELECTION;
	}
	_table.getSelectionModel().setSelectionMode(selectionType);
	_table.setShowVerticalLines(false);
	_table.setShowHorizontalLines(false);
	_table.setIntercellSpacing(new Dimension(
	    rs.getPixels(INTERCELL_WIDTH), rs.getPixels(INTERCELL_HEIGHT)));
	_table.setRowHeight(rs.getPixels(ROW_HEIGHT));
	_table.setAutoResizeMode(JTable.AUTO_RESIZE_LAST_COLUMN);
	_table.setColumnSelectionAllowed(false);
	_table.setRowSelectionAllowed(true);
    }

    /** 
     * Sets up the columns for the table, as described by the array of
     * ItemTableColumn objects.
     * 
     * @param columns An array of ItemTableColumns specifying the
     *                order and properties of the columns in the
     *                table.
     */
    public void setupColumns(ItemTableColumn[] columns) {
	_totalColumnWidth = 0;
	_numColumns = columns.length;
	for (int ii = 0 ; ii < _numColumns; ii++) {
	    ItemTableColumn itc  = columns[ii];
	    TableColumn tc = new TableColumn(ii,itc.width,
					     itc.renderer,
					     itc.editor);
	    _totalColumnWidth += itc.width;
	    tc.setHeaderValue(itc.name);
	    _table.addColumn(tc);
	}
	setSize();
    }
    
    private void setSize() {
	_table.setPreferredScrollableViewportSize(
	    new Dimension(_totalColumnWidth +
			  _table.getIntercellSpacing().width *
			  _numColumns,
			  _table.getRowHeight() * _numRows  +
			  _table.getIntercellSpacing().height * _numRows));
	_table.invalidate();
    }

    /**
     * Sets the height of the ItemTablePanel by specifying the number
     * of rows.  The height of each row is determined by the
     * ROW_HEIGHT property. If there are more Items than rows, a scroll bar
     * will appear.
     * 
     * @param rows The number of rows this ItemTablePanel will show.
     */
    public void setNumRows(int rows) {
	_numRows = rows;
	setSize();
    }
    
    /**
     * Sets the data to be used by the table.  The data should be in
     * the form of a Vector of rows, where each row is a Vector of
     * Objects representing the cells in the row.
     *
     * @param data The data to use
     */
    public void setData(Vector data) {
	_tm.setData(data);
    }
    
    /**
     * Tells the table that the data in <TT>row</TT> has changed and
     * it should redraw that row.
     * 
     * @param row The row to redraw
     */
    public void fireTableRowUpdated(int row) {
	_table.tableChanged(new TableModelEvent(_tm, row, row,
						TableModelEvent.ALL_COLUMNS,
						TableModelEvent.UPDATE));
    }
    
     /**
     * Tells the table that <TT>row</TT> has been inserted in the table
     * 
     * @param row The row that was inserted
     */
    public void fireTableRowInserted(int row) {
	_table.tableChanged(new TableModelEvent(_tm, row, row,
						TableModelEvent.ALL_COLUMNS,
						TableModelEvent.INSERT));
    }
  
    /**
     * Tells the table that <TT>row</TT> has been deleted from the table
     * 
     * @param row The row that was deleted
     */
    public void fireTableRowDeleted(int row) {
	_table.tableChanged(new TableModelEvent(_tm, row, row,
						TableModelEvent.ALL_COLUMNS,
						TableModelEvent.DELETE));
    }

    /**
     * Tells the table that <TT>rows</TT> has been deleted from the table
     * 
     * @param startRow The first row that was deleted
     * @param endRow The last row that was deleted
     */
    public void fireTableRowsDeleted(int startRow, int endRow) {
	_table.tableChanged(new TableModelEvent(_tm, startRow, endRow,
						TableModelEvent.ALL_COLUMNS,
						TableModelEvent.DELETE));
    }
    
    /**
     * Tells the table that the cell at <TT>row, column</TT> has
     * changed, and the table should redraw that cell.
     * 
     * @param row The row
     * @param column The column 
     */
    public void fireTableCellChanged(int row, int column) {
	_table.tableChanged(new TableModelEvent(_tm, row, row, column));
    }  
    
    /**
     * Tells the table that the data in the table has changed, and it
     * should redraw the entire table.
     */
    public void fireTableDataChanged(){
	_table.tableChanged(new TableModelEvent(_tm));
    }
    
    /**
     * Tells the table that the structure of the table (number of
     * columns) has changed, and it should recalculate and redraw the
     * entire table.
     */
    public void fireTableStructureChanged(){
	_table.tableChanged(new TableModelEvent(_tm,
						TableModelEvent.HEADER_ROW));
    }

    /** Adds a TableSortRequestListener to this table.
     * 
     * @param listener The TableSortRequestListener to add
     */
    public void addTableSortRequestListener(
	TableSortRequestListener listener) {
	_table.addTableSortRequestListener(listener);
    }
     
    /**
     * Called by RFrame when we are destroyed.  Tell
     * ItemTableController to stop listening to our Category.
     */
    public void extraCleanup() {
	Enumeration enum = _destroyListeners.elements();
	while (enum.hasMoreElements()) {
	    DestroyListener listener = (DestroyListener)enum.nextElement();
	    listener.itemTableDestroyed(this);
	}
    }

    /** Removes a TableSortRequestListener from this table.
     * 
     * @param listener the TableSortRequestListener to remove
     */
    public void removeTableSortRequestListener(
	TableSortRequestListener listener) {
	_table.removeTableSortRequestListener(listener);
    }

    /**
     * Add a listener that will get notified when we are destroyed.
     * 
     * @param listener Gets notified when we are destroyed.
     */
    /*package*/ void addDestroyListener(DestroyListener listener) {
	_destroyListeners.addElement(listener);
    }

    /**
     * Remove a listener that would have gotten notified when we are
     * destroyed.
     * 
     * @param listener Won't get notified when we are destroyed.
     */
    /*package*/ void removeDestroyListener(DestroyListener listener) {
	_destroyListeners.removeElement(listener);
    }

    /**
     * Returns the rows that are currently selected
     * 
     * @return Currently selected rows.
     */
    public int[] getSelectedRows() {
	return _table.getSelectedRows();
    }

    /**
     * Adds a ListSelectionListener to the table's selection model
     *
     * @param listener The listener to add.
     */
    public void addListSelectionListener(ListSelectionListener listener) {
	_table.getSelectionModel().addListSelectionListener(listener);
    }
    
    
}
