//
// ItemView.java
//
//	A Class that encapsulates an ItemView
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import javax.swing.*;
import java.awt.event.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.category.*;
import java.util.*;
import java.text.*;

/** 
 * A class that encapsulates an ItemView.
 * An ItemView shows all the relevant information about a particular
 * Item in a way that the user can understand.  There are several
 * sections to an ItemView:
 * <dl>
 * <dt>Icon
 * <dd>Shown in the upper left corner of the ItemView.  Typically,
 * the Icon represents the type of Item being viewed, and perhaps the
 * state of the Item.
 * <dt>Fields
 * <dd>Shown in the upper right corner of the ItemView.  This section
 * is a key/value section, and is designed to show information about
 * the Item that can be represented by fairly short Strings.
 * <dt>Additional Info section
 * <dd> This section occupies the center of the ItemView.  It is an
 * optional section.  This section is designed to show information
 * about the Item that can't easily be represented as a single line of
 * text.  Examples include ItemTables, graphs, or additional icons.
 * Any java component can be shown here.
 * <dt> TaskShelf section
 * <dd> This section occupies the bottom of the ItemView.  It shows a
 * TaskShelf that contains Tasks that make sense to perform with the
 * displayed Item in the Item's current state.
 * </dl>
 *
 * To create an ItemView, you should first call the static
 * <tt>createItemView</tt> to invoke the ItemViewFactory and
 * instantiate an ItemView.  It is then necessary to call setItem
 * on the ItemView to tell the ItemView which Item to display.
 * <p>
 * To learn more about how to use resource files to define an
 * ItemView, see the ItemViewController documentation.
 * To learn how to control the overall look of the ItemView by
 * setting various properties, look at the ItemViewProperties
 * documentation.
 *
 * @see ItemViewFactory
 * @see ItemViewController
 * @see ItemViewProperties
 */
public class ItemView implements ItemViewProperties {
    
    private static final String CLASS_NAME = "ItemView";
    private ItemViewController _ivController;
    private ItemViewContext _ivContext;
    private TaskShelfController _tsController;
    private Category _category;
    private String _itemSelector;
    private String _categoryName;
    private IconRenderer _iconRenderer; 
    private NameRenderer _nameRenderer;
    private int _iconWidth;
    private int _iconHeight;
    private RenderedObjectListener _iconL;
    private RenderedObjectListener _nameL;
    private Vector _titleListeners = new Vector(2);
    private String _currentTitle;

    /** 
     * The constructor
     * 
     * @param ivController the ItemViewController
     * @param tsController the TaskShelfController
     * @param ivContext the ItemViewContext
     * @param category the Category to use
     * @param fullCategoryName The full name of the category to use
     */
    ItemView(ItemViewController ivController, 
	     TaskShelfController  tsController,
	     ItemViewContext ivContext,
	     Category category,
	     String fullCategoryName) {
	_ivContext = ivContext;
	_ivController = ivController;
	_tsController = tsController;
	_category = category;
	_categoryName = fullCategoryName;
	_iconRenderer =
	    _ivContext.getHostContext().getIconRenderer(fullCategoryName);
	ResourceStack rs = ivContext.getResourceStack();
	_iconWidth = rs.getPixels(ICON_WIDTH);
	_iconHeight = rs.getPixels(ICON_HEIGHT);
    }
     
    /**
     * Sets the Item to display in this ItemView
     * 
     * @param selector the selector of the Item to display
     */
    public void setItem(String selector) {
	if (_iconL != null) {
	    _iconRenderer.removeIconListener(_itemSelector, _iconL);
	}
	if (_nameL != null) {
	    _nameRenderer.removeNameListener(_itemSelector, _nameL);
	}
	_itemSelector = selector;
	_ivController.setItem(selector);
        if (_tsController != null) {
            _tsController.setItem(_category,selector);
        }
        
	_iconRenderer.addIconListener(selector, null, _iconWidth,
				      _iconHeight,
				      _iconL = 
				      new RenderedObjectListener() {
	    public void renderedObjectChanged(String selector, Object object) {
		_ivController.getItemViewPanel().setIcon((Icon)object);
	    }
	});
	if (_titleListeners.size() > 0) { 
	    _nameRenderer.addNameListener(selector,
					  NameRenderer.ITEM_VIEW,
					  _nameL = new NameL());
	}
	
    }
    
    /**
     * Gets the selector of the Item displayed in the ItemView
     * 
     * @return The selector of the Item being displayed
     */
    public String getItemSelector() {
	return _itemSelector;
    }

    /**
     * Gets the full name of the Category displayed in the ItemView
     * 
     * @return The name of the category being displayed
     */
    public String getCategoryName() {
	return _categoryName;
    }
    
    /**
     * Destroys an ItemView so that it can be garbage collected.
     */
    public void destroy() {
	_ivController.destroy();
        if (_tsController != null) {
            _tsController.destroy();
        }
	if (_iconL != null) {
	    _iconRenderer.removeIconListener(_itemSelector, _iconL);
	}
	if (_nameL != null) {
	    _nameRenderer.removeNameListener(_itemSelector, _nameL);
	}
	// Null-out the member variables for better garbage collection
	// and so we'll throw an exception if someone calls any
	// methods after calling this one.
	_ivController = null;
	_ivContext = null;
	_tsController = null;
	_category = null;
	_iconL = null;
	_nameL = null;
    }
    
    /**
     * Gets the JPanel that is actually containing the ItemView.
     * 
     * @return the JPanel
     */
    public JPanel getPanel() {
	return _ivController.getItemViewPanel();
    }

    /**
     * Gets the ItemViewContext being used by the ItemView
     *
     * @return the ItemViewContext
     */
    public ItemViewContext getItemViewContext() {
	return _ivContext;
    }
    
    /**
     * Adds a TaskLaunchRequestListener that will be notified if a
     * user requests that a Task be launched (such as by clicking in a
     * TaskShelf contained by this ItemView) 
     * 
     * @param listener the TaskLaunchRequestListener to add
     */
    public void addTaskLaunchRequestListener(
	TaskLaunchRequestListener listener) {
        if (_tsController != null) {
            _tsController.getTaskShelfPanel().
	    addTaskLaunchRequestListener(listener);
        }
    }
    
    /**
     * Removes a TaskLaunchRequestListener
     *
     * @param listener the TaskLaunchRequestListener to remove
     */
    public void removeTaskLaunchRequestListener( 
	TaskLaunchRequestListener listener) {
        if (_tsController != null) {
            _tsController.getTaskShelfPanel().
	    removeTaskLaunchRequestListener(listener);
        }
    }    

    /**
     * Adds an ItemViewLaunchRequestListener that will be notified if
     * a user requests that a ItemView be launched (such as be
     * clicking on a hyperlink on the ItemView) 
     *
     * @param listener the ItemViewLaunchRequestListener to add
     */
    public void addItemViewLaunchRequestListener(
	ItemViewLaunchRequestListener listener) {
	_ivController.addItemViewLaunchRequestListener(listener);
    }

    /**
     * Removes an ItemViewLaunchRequestListener
     *
     * @param listener the ItemViewLaunchRequestListener to remove
     */
    public void removeItemViewLaunchRequestListener( 
	ItemViewLaunchRequestListener listener){
	_ivController.removeItemViewLaunchRequestListener(listener);
    }
    
    /** 
     * Adds a title Listener
     * @param listener The TitleListener that's notified when the
     * title changes.  
     */
    public void addTitleListener(TitleListener listener) {
	if (_nameRenderer == null) {
	    _nameRenderer = 
		_ivContext.getHostContext().getNameRenderer(getCategoryName());
	    if (_itemSelector != null) {
		_nameRenderer.addNameListener(_itemSelector,
					      NameRenderer.ITEM_VIEW,
					      _nameL = new NameL());
	    }
	}
	_titleListeners.addElement(listener);
	if (_currentTitle != null) {
	    listener.titleChanged(new TitleEvent(getPanel(), _currentTitle));
	}
    }
    
    public void removeTitleListener(TitleListener listener) {
	_titleListeners.removeElement(listener);
    }
					 
    private void notifyTitleListeners() {
	final int num = _titleListeners.size();
	for (int ii = 0; ii < num; ii++) {
	    ((TitleListener)_titleListeners.elementAt(ii)).titleChanged(
		new TitleEvent(getPanel(), _currentTitle));
	}
    }
    

    private class NameL implements RenderedObjectListener {
	public void renderedObjectChanged(String selector, Object object) {
	    _currentTitle = (String)object;
	    notifyTitleListeners();
	}
    }
    
    /** 
     * Creates an ItemView by invoking the ItemViewFactory
     * 
     * @param hostContext the hostContext to use
     * @param categoryName the Category to use
     *
     * @exception ItemViewException if there is an error while
     *            creating the ItemView.
     */
    public static ItemView createItemView(HostContext hostContext, 
					  String categoryName)
	throws ItemViewException{
	
	if (hostContext == null) {
	    throw new IllegalArgumentException(
		"Can't call createItemView with hostContext null");
	}
	if (categoryName == null) {
	    throw new IllegalArgumentException(
		"Can't call createItemView with categoryName null");
	}
	
	ItemViewContext ivc = new 
	    ItemViewContext(categoryName, hostContext);

	ItemViewFactory factory = null;
	ResourceStack rs = ivc.getResourceStack();
	try {
	    String factoryName = rs.getString(ITEM_VIEW_FACTORY);
	    try {
		factory = (ItemViewFactory)SysUtil.createObject(
		    factoryName, ItemViewFactory.class, 
                    new Class[] {String.class, ItemViewContext.class}, 
                    new Object[] {categoryName, ivc});
                        
	    } catch (Exception e) {
		Log.error(CLASS_NAME, "Failed to load a class named \""
			  + factoryName + "\".  The error was: " +
			  e);
		throw new ItemViewException(
		    MessageFormat.format(
			rs.getString(CANT_LOAD_CLASS),
			new Object[] { factoryName, 
					   ItemTableFactory.class.getName()}));
	    }
	} catch (MissingResourceException exception) {
	    // Use the default ItemTableFactory
	}
	if (factory == null) {
	    factory = new ItemViewFactory(categoryName,ivc);
	}
	    return factory.createItemView();
    }
    
    private static final String URL_BEGIN = "itemView://category=";
    private static final String URL_ITEM = "&item=";

    /**
     * Creates a URL that can be used to launch an ItemView
     * 
     * @param category The name of category that contains the item to view
     * @param item The name of the item to view.
     * @return A URL that contains <TT>category</TT> and <TT>item</TT>
     */
    public static String createURLToLaunch(String category, String item) {
	return URL_BEGIN + category + URL_ITEM + item;
    }
    
    /**
     * Gets the category out of a URL that was created with
     * <TT>createURLToLaunch</TT>
     *
     * @param url The string to extract the category from
     * @return the name of the category
     * @see com.sgi.sysadm.ui.ItemView#createURLToLaunch
     */
    public static String getCategoryFromURL(String url) {
	
	Log.assert(url.startsWith(URL_BEGIN),"Malformed itemView URL");
	int ampersand = url.indexOf('&');  
	return url.substring(URL_BEGIN.length(),ampersand);
    }

    /**
     * Gets the Item's name out of a URL that was created with
     * <TT>createURLToLaunch</TT>
     *
     * @param url The string to extract the Item's name  from
     * @return the name of the item
     * @see com.sgi.sysadm.ui.ItemView#createURLToLaunch
     */
    public static String getItemFromURL(String url) {
	Log.assert(url.startsWith(URL_BEGIN),"Malformed itemView URL");
	int ampersand = url.indexOf('&');
	return url.substring(ampersand+URL_ITEM.length());  
    }
}

	
