//
// ItemViewAdditionalInfoRenderer
//
//	An interface that describes how to render the additional
//	information about an Item for an ItemView.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.category.*;
import javax.swing.*;

/**
 * An interface that describes how to render the additional
 * information about an Item for an ItemView. 
 * The sequence that the methods will be called in is as follows:
 * <ol>
 * <li>initializeAdditionalInfoRenderer
 * <li>renderInfo
 * <li>renderInfoAgain (zero or more times)
 * <li>renderInfoBlank
 * <li>repeat from step 2 (only if ItemView is used to display another
 * Item)
 * </ol>
 * <p>
 * This class is used by ItemViewController to display the section of
 * the ItemView known as the "Additional Info" section.   To
 * use a ItemViewAdditionalInfodRenderer, the programmer should set the
 * ADDITIONAL_INFO_RENDERER property to a class that implements this interface.
 * See the
 * ItemView and ItemViewController documentation for more
 * information about how to use this class.
 *
 * @see ItemView
 * @see ItemViewController
 * @see ItemViewProperties#ADDITIONAL_INFO_RENDERER
 */
public interface ItemViewAdditionalInfoRenderer {

    /**
     * Initializes the AdditionalInfoRenderer.  The
     * renderer need take no action, but can initialize variables,
     * etc.  It is also customary to use this method to create
     * components and add them to <tt>panel</tt>  To add values that
     * look like the values that the infrastructure uses in the
     * "fields" section of the ItemView, use an
     * RLabel and pass VALUE_RLABEL_NAME as the name of the RLabel to
     * the RLabel constructor.
     *
     * @param panel The LabelComponentPanel on which to draw.
     * @param ivc The ItemViewContext to use
     * @param controller The ItemViewController using the renderer
     * @see ItemViewProperties#VALUE_RLABEL_NAME
     * @see RLabel
     */
    public void initializeAdditionalInfoRenderer(LabelComponentPanel panel, 
						 ItemViewContext ivc, 
						 ItemViewController controller);
    
    /**
     * Updates the additional info panel. This will be called the
     * first time the Item arrives in the ItemView.  The Renderer
     * should setup anything that depends on the Item's selector, and
     * update the components on the panel as needed.  If the panel
     * changes after this method returns then you should call the
     * <tt>validate</tt> method on the <tt>panel</tt> that was passed
     * to <tt>initializeAdditionalInfoRenderer</tt>.
     *
     * @param item The Item to render */
    public void renderInfo(Item item);
      
    /**
     * Updates the additional info panel after the Item has
     * changed.  This gets called if
     * the Item changes after <tt>renderInfo</tt> has been called.  You know
     * that the Item has the same selector as the Item as the previous
     * renderInfo call;
     * only the Attributes have changed.  If the panel changes after this
     * method returns then you should call the <tt>validate</tt> method on the
     * panel that was passed to <tt>initializeAdditionalInfoRenderer</tt>.
     *
     * @param item The item to draw
     */
    public void renderInfoAgain(Item item);

    /**
     * Updates the additional info panel to be blank.  This will be called
     * when an item is deleted, or the item panel otherwise needs to
     * be blank.  You can still show components when blank, but don't
     * show any state for the Item.  The renderer should forget all
     * about the Item that it learned about in <tt>renderInfo</tt>, and it's
     * also the appropriate time to undo any actions that have been
     * done in the <tt>rendererInfo</tt> method so that this object can be
     * garbage collected.  This may be the last call to the renderer,
     * but if it's not, then it's guarenteed that the next call will
     * be to the <tt>renderInfo</tt> method.
     */
    public void renderInfoBlank();
}
