//
// ItemViewController
//
//	A class that keeps an ItemViewPanel up-to-date with an
//	Item.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.net.*;
import java.text.*;
import javax.swing.*;
import javax.swing.event.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.richText.*;

/**
 * There are three ways to use this class.  The first is a default
 * version.  In this version it is not necessary to write any
 * Item-specific code, or any Item specific resource files.
 * The second method is to add some resources that give the
 * ItemViewController hints on what type of display is desired.  The third
 * way is to subclass this class and provide code to achieve the
 * desired results.
 * <p>
 * The first method is to simply create the object by calling the
 * constructor, pass a ItemViewPanel to the
 * <tt>setItemViewPanel</tt> method, and then
 * pass the ItemViewController a Item selector to
 * display via the <tt>setItem</tt> method.  All of these details are
 * handled automatically by calling ItemView's <tt>createItemView</tt>
 * method. In this default version of an ItemView,
 * all of the Attributes in the Item that is being shown will be
 * converted to strings with the <tt>toString</tt> method and displayed in the
 * ItemViewPanel.
 * <p>
 * To provide more flexibility, this ItemViewController will also
 * attempt to use a ResourceStack to obtain display information.  To
 * use this feature, you must add some resources to the Category's resource
 * file that is contained in the ResourceStack that's contained in
 * the ItemViewContext that is passed to the constructor.  This
 * ResourceStack is build automatically when calling ItemView's
 * <tt>createItemView</tt> method.  The resources that can be used to
 * control the ItemViewController are outlined below.  For more
 * information, see the ItemViewProperties documentation.  For
 * information about ItemViews in general, see the ItemView documentation.
 * <p>
 * To use the resources, there are three types of resources to add the
 * the resource files:
 * <p>
 * First, there must be a set of resources called FIELDS.  The strings in
 * this string array are names that you give each of the fields to display
 * in the "fields" section ItemView. The order of strings in the set
 * will determine the order in which the fields will be displayed.
 * For example, an ItemView that is displaying user
 * accounts might have a properties file that contains:
 * <PRE>
 * com.sgi.product.MyCategory.ItemView.fields0=name
 * com.sgi.product.MyCategory.ItemView.fileds1=gecos
 * com.sgi.product.MyCategory.ItemView.fields2=uid
 * com.sgi.product.MyCategory.ItemView.fields3=shell
 * </PRE>
 * <p>
 * Second, for each field that will be displayed, there may also be a
 * resource that specifies the string to use as the label for that
 * field. The resource specifying the label is LABEL. For example,
 * to label the field "type", add a resource
 * <tt>com.sgi.product.MyCategory.ItemView.label.type</tt>.
 * If this resource is not specified, the name of the field will be
 * used as the label.
 * <p>
 * Finally, each field may have a resource METHOD that specifies the
 * manner in which a particular field of the Item will be
 * generated for display in the ItemView.
 * All of the methods of rendering cells except for the "renderer"
 * method assume that the name of the column is the same as the
 * name of the Attribute in the Item that the column is
 * representing.  If this is not the case, then you must use the
 * BASED_ON attribute to specify which Attribute the column should be
 * based on.
 * For example, to specify the method in which an attribue "type" is
 * displayed, add a resource with the name
 * <tt>com.sgi.product.MyCategory.ItemView.method.type</tt>
 * and give it one of the values mentioned below.  The methods are:
 * <dl>
 * <dt><tt>toString</tt>
 * <dd> Specifies that the value of an Attribute should be converted
 * into a string using the toString() method on the Attribute's
 * value.  This is the default method if no conversion is specified.
 * If the name of the field is not the key of an Attribute in the
 * Item, you must provide a resource BASED_ON which
 * specifies the name of the Attribute in the Item to use.
 *
 * <dt><tt>renderer</tt>
 * <dd>Specifies that field should be derived by using an ItemViewFieldRenderer
 * object.  The ItemViewFieldRenderer object used is the one that
 * was passed to the setItemViewFieldRenderer method of this class
 *
 * <dt><tt>lookup</tt>
 * <dd>Specifies that the value of an Attribute should be converted
 * into a String by appending the value of the Attribute as returned
 * by the <tt>toString</tt> method to the resource key defined by LOOKUP
 * and then using that string as
 * a resource key specifying the string to display.
 * For example, if an Attribute ("type","23") is displayed with the
 * lookup method, the resource
 * <i>com.sgi.product.MyCategory.ItemView.lookup.type.23</i>
 * will be used to get the string to display for the value of that
 * Attribute.
 * If the name of the field is not the key of an Attribute in the
 * Item, you must provide a resource BASED_ON which
 * specifies the name of the Attribute in the Item to use.
 * <p>
 * This method is good for cases when the value of the
 * Attributes comes from a limited set of possible values, and you
 * wish to provide a mapping from the Attribute to some more
 * easily understandable String.  For example, if the Attribute
 * was a boolean attribute, the following could be used in the
 * propery file:
 * <pre>
 * com.sgi.product.UserCategory.ItemView.lookup.turnedOn.true = True
 * com.sgi.product.UserCategory.ItemView.lookup.turnedOn.false = False
 * </pre>
 * The values displayed for true and false can now be localized
 * or modified easily.
 *
 * <dt><tt>richText</tt>
 * <dd>Specifies that the value of the Attribute is the selector of an
 * Item in a particular Category.  The Attribute will be displayed as
 * a RichText component that will launch an ItemView.  See the
 * CATEGORY field for details on how to specify which Category the
 * Item belongs to.  See the SELECTOR field for details on how to
 * specify the selector of the Item to show.
 * If the name of the field is not the key of an Attribute in the
 * Item, you must provide a resource BASED_ON which
 * specifies the name of the Attribute in the Item to use.
 *
 * </dl>
 * <p>
 * Sometimes, an Attribute will be missing from an Item, and in this
 * case the ItemView should display a particular string.  For example,
 * consider an Item that can optionally contain an Attribute "name".
 * If the Item contains that Attribute, then the ItemView should
 * display the name, otherwise, it should display "(None)".  For this
 * situation, you can use the MISSING resource.  The MISSING resource
 * allows you to specify a string that will be displayed if an Attribute
 * is missing from an Item.  The MISSING resource can be used with the
 * <tt>toString</tt>, <tt>lookup</tt>, or <tt>richText</tt>
 * methods. See the MISSING resource for information about using the
 * MISSING resource.
 * <p>
 * Sometimes, an Attribute will be present in an Item, but its value
 * is the empty string ("").  In this case you might want the the
 * ItemView to display a particular string.  For example, consider an
 * Item that can contains an Attribute "nickname".  If that Attribute
 * is set to a non-zero length string, then the ItemView should
 * display that string, otherwise, it should display "(None)".  For
 * this situation, you can use the EMPTY resource.  The EMPTY resource
 * allows you to specify a string that will be displayed if an
 * Attribute is present but contains the empty string.  The EMPTY
 * resource can be used with the <tt>toString</tt> or
 * <tt>richText</tt> methods. See the EMPTY resource for information
 * about using the EMPTY resource.
 * <p>
 * @see ItemViewProperties#FIELDS
 * @see ItemViewProperties#LABEL
 * @see ItemViewProperties#BASED_ON
 * @see ItemViewProperties#METHOD
 * @see ItemViewProperties#CATEGORY
 * @see ItemViewProperties#SELECTOR
 * @see ItemViewProperties#MISSING
 * @see ItemViewProperties#EMPTY
 * @see ItemViewProperties
 * @see ItemView */
public class ItemViewController implements ItemViewProperties {

    private static final String TO_STRING_METHOD_STR = "toString";
    private static final String RENDERER_METHOD_STR = "renderer";
    private static final String LOOKUP_METHOD_STR  = "lookup";
    private static final String RICH_TEXT_METHOD_STR = "richText";

    private static final byte TO_STRING_METHOD =0;
    private static final byte RENDERER_METHOD=1;
    private static final byte LOOKUP_METHOD =2;
    private static final byte RICH_TEXT_METHOD = 3;

    private static final String CLASS_NAME = "ItemViewController";
    private ItemViewPanel _ivp;
    private ResourceStack _rs;
    private boolean _itemFound;
    private boolean _usingResources = false;
    private String[] _fields;
    private byte[] _displayMethods;
    private String[] _basedOns;
    private String _currentItemSelector;
    private boolean _inChangeBlock;
    private Hashtable _itemViewInfoTable = new Hashtable();
    private Category _category;
    private ItemViewFieldRenderer _ivfr;
    private Hashtable _fieldRendererComponents = new Hashtable();
    private ItemViewAdditionalInfoRenderer _ivair;
    private Component _tsp;
    private ItemViewContext _ivc;
    private Vector _itemViewLaunchRequestListeners = new Vector();
    private String _name;
    private Item _currentItem;
    private boolean _gotEndExists;
    private CategoryListener _catListener = new CatListener();
    private static final String[] EMPTY_STRING_ARRAY = new String[0];
    private LListener llistener = new LListener();

    /**
     * Constructor for ItemViewController
     *
     * @param category The category that contains the Item to view
     * @param name The name of this ItemView.  This name is prepended
     *             to all the resources keys.  This should be the
     *             package qualified name of the Category that this
     *             ItemView will display Items of.
     * @param ivc The ItemViewContext for the ItemView.
     */
    public ItemViewController(Category category,
			      String name,
			      ItemViewContext ivc) {
	_rs = ivc.getResourceStack();
	_category = category;
	_ivc = ivc;
	_name = name;

	// Determine if we can use a resource file to provide default
	// behavior.  If we can't, we'll just have to show all the
	// attributes as straight text.
	String lookup = _name + FIELDS;
	try {
	    _fields = _rs.getStringArray(lookup);
	    _usingResources = true;
	} catch (MissingResourceException mre) {
	    Log.warning(CLASS_NAME,
			"Didn't find a resource with the name: \""+
			lookup + "\".  " +
			"Defaulting to no-resource ItemViewController "+
			"methods.");
	}
    }

    /**
     * Sets the ItemViewPanel to use with this ItemViewController.  If
     * this ItemView will be using an ItemViewFieldRenderer, it
     * needs to be set via the <tt>setItemViewFieldRenderer</tt> method
     * before this method is called.
     *
     * @param ivp The ItemViewPanel to control.
     *
     * @see #setItemViewFieldRenderer
     */
    public void setItemViewPanel(ItemViewPanel ivp) {
	_ivp = ivp;
	if (_tsp != null) {
	    _ivp.setTaskShelf(_tsp);
	}
	if (_fields != null) { // if we found a fields resource
	    _displayMethods = new byte[_fields.length];
	    _basedOns =  new String[_fields.length];
	    for (int ii = 0 ; ii < _fields.length ; ii++) {
		// We look up the label in the resource file.  If it's not
		// there, we'll just use the field as is and issue a
		// warning.
		String field = _fields[ii];
		String label = _name + LABEL + field;
		byte displayMethod;
		String lookup = _name + METHOD + field;
		try {
		    displayMethod = ResourceStack.mapStringToByte(
			_rs.getString(lookup),
			new String[]
			{  TO_STRING_METHOD_STR,
			   RENDERER_METHOD_STR,
			   LOOKUP_METHOD_STR,
			   RICH_TEXT_METHOD_STR,
			}
			, new byte[] {
			    TO_STRING_METHOD,
			    RENDERER_METHOD,
			    LOOKUP_METHOD,
			    RICH_TEXT_METHOD,
			});
		} catch (MissingResourceException mre) {
		    Log.warning(CLASS_NAME,
				"Resource \"" + lookup +
				" was not found.  " +
				"Defaulting to \"" + TO_STRING_METHOD_STR +
				"\"");
		    displayMethod = TO_STRING_METHOD;
		}
		_displayMethods[ii] = displayMethod;

		if (displayMethod == RENDERER_METHOD) {
		    Log.assert(_ivfr != null,
			       "The display method for \"" + field +
			       "\" was set to a renderer, but the " +
			       " renderer was never set.");
		}

		if (displayMethod == RICH_TEXT_METHOD) {
		    RichTextComponent rtc =
			new RichTextComponent(field,_rs, null);
		    rtc.setMargins(new Insets(0,0,0,0));
		    rtc.addLinkListener(llistener);
		    _itemViewInfoTable.put(
			field,_ivp.addItemViewInfo(label, rtc));
		} else if (displayMethod == RENDERER_METHOD) {
		    Component c = _ivfr.getComponentForField(field);
		    _fieldRendererComponents.put(field, c);
		    _itemViewInfoTable.put(
			field, _ivp.addItemViewInfo(label, c));

		} else {
		    // Put the key/value pair into the ItemViewPanel, and
		    // remember the returned ItemViewInfo so that we can
		    // modify the value later.
		    _itemViewInfoTable.put(field, _ivp.addItemViewInfo(label,
								     ""));
		}
		String basedOn;
		try {
		    basedOn = _rs.getString(_name + BASED_ON + field);
		} catch (MissingResourceException mre) {
		    basedOn = field;
		}
		_basedOns[ii] = basedOn;
	    }
	}
    }

    /**
     * Sets the TaskShelfPanel to place in the ItemView
     *
     * @param tsp The TaskShelfPanel
     */
    public void setTaskShelfPanel(TaskShelfPanel tsp) {
	_tsp = tsp;
	if (_ivp != null) {
	    _ivp.setTaskShelf(_tsp);
	}
    }

    /**
     * Sets the ItemViewFieldRenderer to use with this ItemView
     *
     * @param renderer The ItemViewFieldRenderer
     */
    public void setItemViewFieldRenderer(ItemViewFieldRenderer renderer) {
	Log.assert(_ivp == null,
		   "Must call this method (setItemViewFieldRenderer) " +
		   "before setting the  ItemViewPanel via the " +
		   "setItemViewPanel method.");
	_ivfr = renderer;
	if (renderer != null) {
	    renderer.initializeFieldRenderer(_ivc, this);
	}
    }

    /**
     * Sets the ItemViewAdditionalInfoRenderer to use with this ItemView
     *
     * @param renderer The ItemViewAdditionalInfoRenderer
     */
    public void setItemViewAdditionalInfoRenderer(
	ItemViewAdditionalInfoRenderer renderer) {
	if (renderer != null) {
	    _ivair = renderer;
	    LabelComponentPanel additionalInfoPanel =
		new LabelComponentPanel(_rs);
	    renderer.initializeAdditionalInfoRenderer(
		additionalInfoPanel,_ivc, this);
	    _ivp.setAdditionalInfo(additionalInfoPanel);
	    additionalInfoPanel.invalidate();
	    if (_ivp.isShowing()) {
		_ivp.repaint();
	    }
	}
    }

    /**
     * Set the Item that is displayed in the ItemView
     *
     * @param selector The Item whose selector matches this parameter
     *     will be displayed.  Any Item that was previously being displayed
     *     will be forgotton.
     */
    public void setItem(String selector) {
	// If we happened to be in a change block, then we won't be
	// after removing the CategoryListener.  So set our flag to false.
	_inChangeBlock = false;
	if (_currentItemSelector != null) {
	    // We've already got an Item set, so do some cleanup.
	    _category.removeCategoryListener(_catListener);
	    clearItemView();
	}
	// remember that we haven't got notification about an Item yet.
	_currentItemSelector = selector;
	_currentItem = null;

	NotificationFilter nf = new NotificationFilter();
	nf.monitorItem(selector);
	_category.addCategoryListener(_catListener, nf);
    }

    /**
     * Removes any CategoryListeners that have been installed
     */
    public void destroy() {
	_category.removeCategoryListener(_catListener);
	_category = null;
	if (_ivair != null) {
	    _ivair.renderInfoBlank();
	}
	if (_ivfr != null) {
	    _ivfr.renderFieldsBlank();
	}
    }

    /**
     * Refresh the view.  Forces a repaint of the ItemView, taking
     * into account any changes that have been made.  It refreshes
     * only if we're not in a change block.
     */
    public void refresh() {
	if (!_inChangeBlock) {
	    _ivp.validate();
	    _ivp.repaint();
	}
    }

    /**
     * Returns the ItemViewPanel that was passed to setItemPanel
     * @return an ItemViewPanel
     */
    public ItemViewPanel getItemViewPanel() {
	return _ivp;
    }

    /**
     * Adds a ItemViewLaunchRequestListener to the list of listeners
     * that will be notified when a user clicks on a hyperlink to
     * launch a new ItemView.  If no listeners are added, then the
     * default behavior is to start the ItemView in a new ItemViewFrame.
     *
     * @param listener The ItemViewLaunchRequestListener to add.
     */
    public void addItemViewLaunchRequestListener(
	ItemViewLaunchRequestListener listener) {
	_itemViewLaunchRequestListeners.addElement(listener);
    }

    /**
     * Remove a ItemViewLaunchRequestListener from the list of
     * listeners
     *
     * @param listener The ItemViewLaunchRequestListener to remove
     */
    public void removeItemViewLaunchRequestListener(
	ItemViewLaunchRequestListener listener) {
	_itemViewLaunchRequestListeners.removeElement(listener);
    }

//------------------ CategoryListener methods -----------------------

    private class CatListener extends CategoryAdapter {

	/**
	 * Called when the Item we're interested in gets added to the
	 * category.
	 *
	 * @param item The Item that was added.
	 * @see com.sgi.sysadm.category.CategoryListener#itemAdded
	 */
	public void itemAdded (Item item) {
	    _currentItem = item;
	    drawFields();
	    // Take care of the additional info panel
	    if (_ivair != null) {
		_ivair.renderInfo(item);
	    }

	    // Tell the Fields renderer that the Item changed.
	    if (_ivfr != null) {
		_ivfr.renderFields(item);
	    }
	    // redraw the panel
	    refresh();
	}

	/**
	 * Called if the Item we're interested in gets changed
	 *
	 * @param item The Item that was changed.
	 * @param newItem The new Item
	 * @see com.sgi.sysadm.category.CategoryListener#itemChanged
	 */
	public void itemChanged(Item item, Item newItem) {
	    // An easy way to change an Item.  We'll just blank everything and
	    // reset it to the new values.  Not the most efficient, but
	    // works for the time being.
	    _currentItem = newItem;
	    drawFields();

	    // Take care of the additional info panel
	    if (_ivair != null) {
		_ivair.renderInfoAgain(newItem);
	    }

	    // Tell the Field renderer that the Item changed.
	    if (_ivfr != null) {
		_ivfr.renderFieldsAgain(newItem);
	    }
	    // redraw the panel
	    refresh();
	}

	private void postItemDeletedMessage(Item item) {
	    String formatString = _rs.getString(_name + ITEM_DELETED, "");
	    if (formatString.length() != 0) {
		Object[] args = new Object[] {item.getSelector()};
		String errorString =
		    MessageFormat.format(formatString,
					 args);
		_ivc.postError(errorString);
		// XXX Consider unposting if Item is added again.
	    }
	}

	/**
	 * Called if the Item we're interested in gets removed from
	 * the category.
	 *
	 * @param item The Item that was removed.
	 * @see com.sgi.sysadm.category.CategoryListener#itemRemoved
	 */
	public void itemRemoved(Item item) {
	    _currentItem = null;
	    if (!_ivp.isShowing()) {
		// We're done if the panel's not showing.
		return;
	    }
	    clearItemView();
	    postItemDeletedMessage(item);
	}

	/**
	 * Called if we're entering a change block
	 * @see com.sgi.sysadm.category.CategoryListener#beginBlockChanges
	 */
	public void beginBlockChanges() {
	    _inChangeBlock = true;
	}

	/**
	 * Called if we're exiting a change block.  Calls <A
	 * HREF="#refresh()">refresh()</a>
	 * @see com.sgi.sysadm.category.CategoryListener#endBlockChanges
	 */
	public void endBlockChanges() {
	    _inChangeBlock = false;
	    refresh();
	}

	/**
	 * Called if we're done with notification of current state.  Posts
	 * an error dialog if the Item that we are supposed to be
	 * displaying has not been seen yet.
	 * @see com.sgi.sysadm.category.CategoryListener#endExists
	 */
	public void endExists() {
	    _gotEndExists = true;
	    if (_currentItem == null && _currentItemSelector != null) {
		try {
		    String formatString =
			_rs.getString(_name + ITEM_NOT_FOUND_ERROR, "");
		    if (formatString.length() != 0) {
			Object[] args = new Object[] {_currentItemSelector};
			String errorString =
			    MessageFormat.format(formatString,
						 args);
			_ivc.postError(errorString);
			// XXX Consider unposting if Item is added.
		    }
		}
		catch (MissingResourceException e) {
		    Log.warning(CLASS_NAME,
				"Item wasn't found, but there was no format " +
				"String in \"" + _name +
				ITEM_NOT_FOUND_ERROR + "\".");
		}
	    }
	}
    }

    private void drawFields () {
	if (_fields == null) {
	    // Create a fake _fields array with the values of
	    // all the Attribute keys of the Item.
	    Enumeration attrEnum = _currentItem.getAttrEnum();
	    _fields = new String[_currentItem.size()];
	    _displayMethods = new byte[_currentItem.size()];
	    _basedOns = new String[_currentItem.size()];
	    for (int ii = 0 ; ii < _currentItem.size(); ii++) {
		String key = ((Attribute)(attrEnum.nextElement()))
		    .getKey();
		_fields[ii] = key;
		_displayMethods[ii] = TO_STRING_METHOD;
		_basedOns[ii] = key;
		_itemViewInfoTable.put
		    (key,_ivp.addItemViewInfo(new JLabel(key), ""));
	    }
	}

	// Iterate through the fields that we should display
	for (int ii = 0 ; ii < _fields.length ; ii++) {
	    String field = _fields[ii];
	    Attribute attr = _currentItem.getAttr(_basedOns[ii]);
	    // The constructor already added labels and empty
	    // values for all of these fields.  We look them
	    // up in the table and fill in the values.
	    ItemViewInfo ivi = (ItemViewInfo)_itemViewInfoTable.get(field);
	    byte displayMethod = _displayMethods[ii];

	    if (attr == null &&
		(displayMethod == TO_STRING_METHOD ||
		 displayMethod == RICH_TEXT_METHOD ||
		 displayMethod == LOOKUP_METHOD)) {
		// Get "missing" string
		String lookup = _name + MISSING + field;
		String missing = null;
		try {
		    missing = _rs.getString(lookup);
		} catch (MissingResourceException mre) {
		    Log.warning(CLASS_NAME,"There is no attribute: \"" +
				_basedOns[ii] +
				"\" in the Item with selector: \"" +
				_currentItem.getSelector() +
				"\".  Will use blanks " +
				"to represent this attribute, since \"" +
				lookup + "\" was not set");
		    missing = "";
		}
		if (displayMethod == RICH_TEXT_METHOD) {
		    ((RichTextComponent)ivi.getValue()).setText(missing);
		} else {
		    ivi.setValueText(missing);
		}
		continue;
	    }

	    if (displayMethod == LOOKUP_METHOD) {
		// If the method is LOOKUP_METHOD, then lookup the
		// value in the resource group specified by
		// <name>.lookup.<field>.<attr value>
		String displayString;
		String lookup = _name + LOOKUP + field +
		    "." + attr.getValueString();

		// If we can't find a particular value, fall
		// back to just using the attr's value.
		// Don't issue a warning here because
		// LOOKUP_METHOD could be used to only provide
		// string for some possible values, such as
		// "Zero" for 0.
		displayString = _rs.getString(lookup,
					      attr.getValueString());

		ivi.setValueText(displayString);
	    } else if (displayMethod == RICH_TEXT_METHOD) {
		// If the method is RICH_TEXT_METHOD set the
		// RichTextComponent

		String lookup = _name + CATEGORY + field;
		String category = null;
		try {
		    category = _rs.getString(lookup);
		} catch (MissingResourceException mre) {
		    Log.fatal("Didn't find a resource \"" +
			     lookup + "\" that was expected " +
			     "because \"" + _name + METHOD + field +
			     "\" was " + RICH_TEXT_METHOD_STR);
		}
		lookup = _name + SELECTOR + field;
		String itemSelector = null;
		try {
		    itemSelector= _rs.getString(lookup);
		} catch (MissingResourceException mre) {
		    Log.fatal("Didn't find a resource \"" +
			     lookup + "\" that was expected " +
			     "because \"" + _name + METHOD + field +
			     "\" was " + RICH_TEXT_METHOD_STR);
		}
		if (itemSelector.equals("ITEM_SELECTOR")) {
		    itemSelector = _currentItem.getSelector();
		} else {
		    itemSelector = _currentItem.getString(itemSelector);
		}
		String rt;
		String itemStr = attr.getValueString();
		if (itemSelector.length() == 0) {
		    // Get "empty" string
		    lookup = _name + EMPTY + field;
		    String empty = null;
		    try {
			empty = _rs.getString(lookup);
		    } catch (MissingResourceException mre) {
			Log.warning(CLASS_NAME,"The attribute: \"" +
				    _basedOns[ii] +
				    "\" in the Item with selector: \"" +
				    _currentItem.getSelector() +
				    "\" is empty.  Defaulting to \"" +
				    itemStr + "\"");
			empty = itemStr;
		    }
		    rt = empty;
		} else {
		    String url =
			ItemView.createURLToLaunch(category,itemSelector);
		    rt = "<A HREF=" +
			RichTextComponent.quoteString(url) + ">" +
			RichTextComponent.quoteString(itemStr) + "</A>";
		}
		((RichTextComponent)ivi.getValue()).setText(rt);
	    } else if (displayMethod != RENDERER_METHOD) {
		if (displayMethod != TO_STRING_METHOD) {
		    Log.warning(CLASS_NAME,
				"Resource \"" + _name + METHOD + field +
				"\" was not "+
				"\"toString\", " +
				"\"renderer\", or " +
				"\"lookup\".  "+
				"Defaulting to \"toString\"");
		}
		// If the display method is TO_STRING_METHOD, then
		// just get the string representation of the
		// attr's value.
		String itemStr = attr.getValueString();
		if (itemStr.length() == 0) {
		    // Get "empty" string
		    String lookup = _name + EMPTY + field;
		    try {
			itemStr = _rs.getString(lookup);
		    } catch (MissingResourceException mre) {
		    }
		}
		ivi.setValueText(itemStr);
	    }
	}
    }

    private void clearItemView() {
	// We can just blank out the values, if any have been set
	if (_fields != null) {
	    for (int ii = 0 ; ii < _fields.length ; ii++) {
		String field = _fields[ii];
		byte displayMethod = _displayMethods[ii];
		ItemViewInfo ivi =
		    (ItemViewInfo)_itemViewInfoTable.get(field);
		if (displayMethod == RICH_TEXT_METHOD) {
		    ((RichTextComponent)ivi.getValue()).setText("");
		} else if(displayMethod != RENDERER_METHOD) {
		    ivi.setValueText("");
		}
	    }

	    // clear the additional info panel
	    if (_ivair != null) {
		_ivair.renderInfoBlank();
	    }

	    // clear the fields controlled by the field
	    // renderer
	    if (_ivfr != null) {
		_ivfr.renderFieldsBlank();
	    }

	    // Redraw the ItemViewPanel
	    refresh();
	}
    }

    /**
     * A method for ItemViewFieldRenderers to call if an field
     * they're rendering changes after the <tt>renderFields</tt> or
     * <tt>renderFieldsAgain</tt> methods return.  This forces a redraw of
     * the field specified by <tt>field</tt>.
     *
     * @param field The string identifying the field that changed.
     *              This should match a string defined in the FIELDS
     *              resource set.
     *
     * @see ItemViewFieldRenderer
     * @see ItemViewProperties#FIELDS
     */
    public void fireFieldChanged(String field) {
	((Component)(_fieldRendererComponents.get(field))).invalidate();
	refresh();
    }

    /**
     * A utility method that returns a RichTextComponent for an
     * ItemViewFieldRenderers' use.  The RichTextComponent is all
     * set up to look like other RichTextComponents in the ItemView,
     * and has a link listener set appropriately.  To launch an
     * ItemView when clicked, the text of the RichTextComponent should
     * contain a link that was formed from ItemView's <tt>createURLToLaunch</tt>.
     *
     * @param name The name that will be passed the the
     *             RichTextComponent constructor
     * @return The RichTextComponent
     *
     * @see ItemViewFieldRenderer
     * @see ItemView#createURLToLaunch
     */
    public RichTextComponent getRichTextForFieldRenderer(String name) {
	RichTextComponent rtc =
	    new RichTextComponent(name, _rs, null);
	rtc.setMargins(new Insets(0,0,0,0));
	rtc.addLinkListener(llistener);
	return rtc;
    }

    /**
     * A utility method that returns a RLabel for a
     * ItemViewFieldRenderer's use.  The RLabel is all set up to
     * look like other RLabels in the ItemView.
     *
     * @return The RLabel.
     * @see ItemViewFieldRenderer
     */
    public RLabel getRLabelForFieldRenderer() {
	return new RLabel("", _rs, VALUE_RLABEL_NAME);
    }

    /**
     * Returns the LinkListener that the ItemViewController uses to handle
     * clicks on RichTextComponents that contain links to launch
     * ItemViews, where the link was created by ItemView's
     * createURLToLaunch.  The AdditionalInfoRenderer can use this if
     * it wishes to let the ItemViewController handle launching the
     * ItemViews.
     *
     * @see ItemView#createURLToLaunch
     */
    public LinkListener getDefaultLinkListener() {
	return llistener;
    }

    private class LListener implements LinkListener {

	/**
	 * Called when a user clicks on a hyperlink in a RichTextComponent
	 *
	 * @param event A LinkEvent describing what the user clicked.
	 */
	public void linkActivated(LinkEvent event) {
	    String target = event.getTarget();
	    String category = ItemView.getCategoryFromURL(target);
	    String item = ItemView.getItemFromURL(target);
	    ItemViewLaunchRequestEvent launchEvent = new
		ItemViewLaunchRequestEvent(ItemViewController.this, category, item);
	    int numListeners = _itemViewLaunchRequestListeners.size();
	    if (numListeners == 0) {
		ItemViewFrame.launchItemViewFrame(launchEvent,_ivc);
	    } else {
		for (int ii = 0 ; ii < numListeners; ii++) {
		    ((ItemViewLaunchRequestListener)
		     _itemViewLaunchRequestListeners.elementAt(ii)).
			itemViewLaunchRequested(launchEvent);
		}
	    }
	}
    }
}
