//
// ItemViewFactory
//
//	A class that knows how to construct an ItemView
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import com.sgi.sysadm.category.*;
import com.sgi.sysadm.util.*;
import java.util.*;
import java.text.*;

/**
 * A class that knows how to construct an ItemView.  If you don't want
 * to use the default ItemViewPanel, ItemViewController, TaskShelfController, or
 * TaskShelfPanel, you must subclass this class.  You <B>do not</B>
 * need to subclass this class to get an item view that uses an
 * ItemViewFieldRenderer or an ItemViewAdditionalInfoRenderer,
 * See the descriptions of getItemViewFieldRenderer() and
 * getItemViewAdditionalInfoRenderer() for more information.
 */
public class ItemViewFactory implements ItemViewProperties {


    /** 
     * The ItemViewContext that was passed to the constructor
     */
    private ItemViewContext _itemViewContext;

    /**
     * The fully qualified name of the Category this ItemView is
     * displaying 
     */
    private String _fullCategoryName;

    /**
     * The name of the category that this ItemView is displaying
     */
    private String _categoryName;

    /** 
     * The Category that this ItemView is displaying
     */
    private Category _category;

    /**
     * The HostContext that was passed in the ItemViewContext to the
     * constructor.
     */
    private HostContext _hostContext;


    private static final String CLASS_NAME = "ItemViewFactory";

    /**
     * Constructor
     *
     * @param categoryName The fully-qualified name of the category
     *                     that contains the item you wish to view.
     * @param itemViewContext The ItemViewContext to use
     */
    public ItemViewFactory(String categoryName, 
			   ItemViewContext itemViewContext) {
	_fullCategoryName = categoryName;
	if (categoryName == null) {
	    throw new IllegalArgumentException(
		"Can't call ItemViewFactory constructor with categoryName null");
	}
	if (itemViewContext == null) {
	    throw new IllegalArgumentException(
		"Can't call ItemViewFactory constructor with itemViewContext null");
	}
	_hostContext = itemViewContext.getHostContext();
	_categoryName = ResourceStack.getClassName(categoryName);
	_category = _hostContext.getCategory(_categoryName);
	_itemViewContext = itemViewContext;
    } 
	
    /** 
     * Creates an ItemView using the results of the various get()
     * methods and the values passed into the constructor.
     *
     * @return An ItemView
     * @exception ItemViewException if there is an error while
     *            creating the ItemView.
     */
    public ItemView createItemView() throws ItemViewException{
	ItemViewPanel ivp = getItemViewPanel();
	ItemViewController controller = getItemViewController();
	TaskShelfController tsc = getTaskShelfController();
	TaskShelfPanel tsp = getTaskShelfPanel();
	if (tsc != null) {
            tsc.setTaskShelfPanel(tsp);
        }
	controller.setItemViewFieldRenderer(
	    getItemViewFieldRenderer());
	controller.setTaskShelfPanel(tsp);
	controller.setItemViewPanel(ivp);
	controller.setItemViewAdditionalInfoRenderer(
	    getItemViewAdditionalInfoRenderer());
	_itemViewContext.setDialogParent(ivp);
	return new ItemView(controller, tsc, _itemViewContext,
			    _category, _fullCategoryName);
    }


    /**
     * Returns the ItemViewController to use in the ItemView.  By
     * default, this method returns the ItemViewController base class
     * that is constructed by passing <tt>categoryName</tt> and
     * <tt>itemViewContext</tt> (as passed to the ItemViewFactory
     * constructor) to the  constructor of ItemViewController.
     * Override this method to use  your own ItemViewController.
     */
    protected ItemViewController getItemViewController() {
	return new ItemViewController(_category,
				      _fullCategoryName,
				      _itemViewContext);
    }
    
    /** 
     * Returns the ItemViewPanel to use for the ItemView.  By
     * default, this method returns the ItemViewPanel base class that
     * is constructed by passing <tt>itemViewContext</tt> (as passed to the
     * ItemViewFactory constructor) to the constructor of
     * ItemViewPanel.  Override this method to use your own
     * ItemViewPanel subclass.
     */
    protected ItemViewPanel getItemViewPanel() {
	return new ItemViewPanel(_itemViewContext);
    }
    
    /** 
     * Returns the TaskShelfController to use in the ItemView.  By
     * default, this method returns the TaskShelfController base class
     * that is constructed by passing the TaskRegistry found in
     * <tt>hostContext</tt> (as passed to the ItemViewFactory
     * constructor) to the constructor of
     * TaskShelfController. Override this method to use your own
     * TaskShelfController subclass.
     */
    protected TaskShelfController getTaskShelfController() {
	return new TaskShelfController(_hostContext, null);
    }
    
    /**
     * Returns the TaskShelfPanel to use with this ItemView.  By
     * default, this method returns the TaskShelfPanel base class that
     * is constructed by passing <tt>itemViewContext></tt> (as passed
     * to the ItemViewFactory constructor) to the constructor of
     * TaskShelfPanel.  Override this method to use your own
     * TaskShelfPanel subclass.
     */
    protected TaskShelfPanel getTaskShelfPanel() {
	return new TaskShelfPanel(_itemViewContext);
    }
     
    /**
     * Loads a class by first checking the resource stack for the
     * class' name, then by appending the appendStr to the
     * _fullCategoryName.
     * 
     * @param resourceLookupStr The key to lookup in the resourceStack
     *                          for the class name.
     * @param appendString The string to append to the
     *                     _fullCategoryName to create a class name.
     * @param returnType The type of the object to return.  If the
     *                   loaded class is not an instance of this type,
     *                   an exception will be thrown.
     *
     * @exception ItemViewException if the class specified by
     *            the resource can't be located, or if a class is
     *            located but can't be instantiated for some reason.
     */
    private Object loadRenderer(String resourceLookupStr,
				String appendStr,
				Class returnType)  
	throws ItemViewException { 
	ResourceStack rs = _itemViewContext.getResourceStack();
	String rendererClassName = rs.getString(resourceLookupStr,
						_fullCategoryName + appendStr);
	try {
	    return SysUtil.createObject(rendererClassName,
					returnType, null, null);
	} catch (ClassNotFoundException cnfe) {
	    if (!rendererClassName.equals(_fullCategoryName + appendStr)) {
		// if we found a class in the resource stack but it
		// wasn't there
		Log.warning(CLASS_NAME, "The ResourceStack specified that \"" +
			    rendererClassName + "\" was to be used as a " +
			    "renderer, but that class was not found.");
	    }
	    return null;
	}   
	catch (SysUtil.ClassLoadException cle) {
	    throw new ItemViewException(MessageFormat.format(
		rs.getString(ItemView.CANT_LOAD_CLASS),
		new Object[] {rendererClassName, returnType.getName()}));
	}
    }
    
    /**
     * Returns the ItemViewFieldRenderer to use with this
     * ItemView.  By default, this method first checks to see if there
     * is a resource specifying the class to use
     * (FIELD_RENDERER).  If this resource is not found, this
     * method attempts to load a class with the name
     * <i>{categoryName}FieldRenderer</i>  If that class is not
     * found, this method returns null.
     *
     * @exception ItemViewException if the class specified in
     *            the resource file does not exist, or if the class
     *            cannot be loaded.
     */
    protected ItemViewFieldRenderer getItemViewFieldRenderer()
	throws ItemViewException{
	return (ItemViewFieldRenderer)
	    loadRenderer(_fullCategoryName + FIELD_RENDERER, 
			 "FieldRenderer",
			 ItemViewFieldRenderer.class);
    }
    
    /**
     * Returns the ItemViewAdditionalInfoRenderer to use with this
     * ItemView.  By default, this method first checks to see if there
     * is a resource specifying the class to use
     * (ADDITIONAL_INFO_RENDERER).  If this resource is not found, this
     * method attempts to load a class with the name
     * <i>{categoryName}AdditionalInfoRenderer</i>  If that class is not
     * found, this method returns null.
     *
     * @exception ItemViewException if the class specified in
     *            the resource file does not exist, or if the class
     *            cannot be loaded.
     */
    protected ItemViewAdditionalInfoRenderer
	getItemViewAdditionalInfoRenderer() throws ItemViewException{
	
	return (ItemViewAdditionalInfoRenderer)
	    loadRenderer(_fullCategoryName + ADDITIONAL_INFO_RENDERER,
			 "AdditionalInfoRenderer",
			 ItemViewAdditionalInfoRenderer.class);
    }
}
    
    
