//
// ItemViewFieldRenderer.java
//
//	An interface that describes how to render a particular field
//	of an ItemView.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.category.*;
import java.awt.*;
/**
 * An interface that describes how to convert an Item into various fields
 * that are displayed in an ItemView.
 * The sequence that the methods will be called in is as follows:
 * <ol>
 * <li>initializeFieldRenderer
 * <li>getComponentForField (once for each field using the
 * renderer)
 * <li>renderFields
 * <li>renderFieldsAgain (zero or more times)
 * <li>renderFieldsBlank 
 * <li>repeat from step 3 (only if ItemView is used to display another
 * Item)
 * </ol>
 * The ItemViewFieldRenderer is used by the ItemViewController to
 * render fields in ways that are not automatically handled by the
 * ItemViewController.  To
 * use a ItemViewFieldRenderer, the programmer should set the METHOD
 * resource to "renderer" in the Category's property file and set the
 * FIELD_RENDERER property to a class that implements this interface.
 * See the ItemView and ItemViewController documentation for more information.
 *
 * @see ItemViewProperties#METHOD
 * @see ItemViewProperties#FIELD_RENDERER
 * @see ItemView
 * @see ItemViewController
 */
public interface ItemViewFieldRenderer {

    /**
     * A method to initialize the ItemViewFieldRenderer.  The renderer
     * need take no action, but can initialize variables, etc.
     * Typically, this is also the place where the renderer creates
     * the components that are later returned in <tt>getComponentForField</tt>
     * 
     * @param ivc The ItemViewContext
     * @param controller The ItemViewController.
     */
    public void initializeFieldRenderer(ItemViewContext ivc, 
					ItemViewController controller);
    
    /**
     * Returns the Component that this ItemViewFieldRenderer
     * is using to render a particular field. After initializing
     * the renderer with <tt>initializeFieldRenderer</tt>, this method
     * will be called once for each
     * field that is using the "renderer" method of display (set
     * via the METHOD property).
     * For convenience, ItemViewController has methods
     * <tt>getRLabelForFieldRenderer</tt> and
     * <tt>getRichTextForFieldRenderer</tt> that
     * return components that look like other Labels and RichText
     * components.  Other components may be returned, though.
     *
     * @param field The field that is using the renderer, as
     *              defined in the FIELDS string array in the
     *              Category's property file.
     * @see ItemViewController#getRichTextForFieldRenderer(String)
     * @see ItemViewController#getRLabelForFieldRenderer()
     * @see ItemViewProperties#METHOD
     * @see ItemViewProperties#FIELDS
     */
    public Component getComponentForField(String field);
    
    /**
     * Tells the renderer to render all the fields that it is
     * supporting.  This will be called the
     * first time the Item arrives in the ItemView.  The Renderer
     * should setup anything that depends on the Item's selector, and
     * fill in all the Components that were returned by
     * <tt>getComponentForField</tt>.  If any of the components change
     * after this method returns, then you should
     * call the <tt>fireFieldChanged</tt> method on the
     * ItemViewController that was passed to
     * <tt>initializeFieldRenderer</tt>.
     *
     * @param item The Item to render.
     * @see ItemViewController#fireFieldChanged(String)
     */
    public void renderFields(Item item);

    /**
     * Tells the renderer to render the fields after the Item has changed.
     * This will be called if the Item changes.  The renderer can be
     * guaranteed that the Item's selector is unchanged from the prior
     * <tt>renderFields</tt> call, and therefore only needs to re-render
     * the fields that depend on the Item's Attributes.   It's
     * only necessary to update the components whose apperance has
     * changed, but it's okay to update all components.  If any of the
     * components change after this  method returns,
     * then you should call the <tt>fireFieldsChanged</tt> method on the
     * ItemViewController that was passed to <tt>initializeFieldRenderer</tt>.
     *
     * @param item The Item to render.
     * @see ItemViewController#fireFieldChanged(String)
     */
    public void renderFieldsAgain(Item item);

    /**
     * Tells the renderer to render all the fields as blanks.  The
     * renderer should forget all about the Item that it learned about
     * in <tt>renderFields</tt>, and it's also the appropriate time to undo
     * any actions that have been done in the <tt>rendererFields</tt>
     * method so that this object can be garbage collected.  This may
     * be the last call to the renderer, but if it's not, then it's
     * guaranteed that the next call will be to the <tt>renderFields</tt>
     * method.
     */
    public void renderFieldsBlank();

}
