//
// ItemViewFrame
//
//	A class that holds an ItemViewPanel, buttons, and menu in a JFrame
//
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.event.*;

/**
 * A subclass of RFrame that holds an ItemView.  The ItemView will
 * occupy the majority of the Frame, with an OK button at the bottom.
 * When an ItemView is added the the ItemViewFrame, the ItemViewFrame
 * adds ItemViewLaunchRequestListeners and TaskLaunchRequestListeners
 * to implement a policy of creating a new frame for either one of
 * those components.  The ItemViewFrame also adds a TitleListener to
 * the ItemTable and keeps the frame's title up to date.
 */
public class ItemViewFrame extends RFrame implements ItemViewProperties {
  
    private ItemViewContext _ivc;
    private ItemView _itemView;

    /**
     * A constructor that is used when you need the frame, but don't
     * yet know the ItemView.  You can call setItemView later to set
     * the ItemView that gets displayed.
     */
    public ItemViewFrame() {
	setDefaultCloseOperation(DISPOSE_ON_CLOSE);
	addWindowListener(new WindowAdapter() {
	    public void windowClosed(WindowEvent e) {
 		if (_itemView != null) {
		    _itemView.destroy();
		    _itemView = null;
		}
	    }
	});
    }
    
    /** 
     * A constructor that is used when you have the ItemView to
     * display
     * 
     * @param itemView The ItemView to display in the frame.
     */
    public ItemViewFrame(ItemView itemView) {
	this();
	setItemView(itemView);
    }
    
    /**
     * Sets the ItemView that is displayed in this frame.
     *
     * @param itemView The ItemView to display in the frame.
     */
    public void setItemView(ItemView itemView) {
	if (_itemView != null) {
	    _itemView.destroy();
	}
	_itemView = itemView;
	_ivc = itemView.getItemViewContext();
	ResourceStack rs = _ivc.getResourceStack();
	int vgap = rs.getPixels(FRAME_VGAP);
	getContentPane().setLayout(new BorderLayout(0,vgap));
	JPanel buttonPanel = new JPanel();
	JButton button = new JButton(rs.getString(FRAME_OK));
	buttonPanel.add(button);
	button.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
		setVisible(false);
		dispose();
	    }
	});
	getContentPane().add(buttonPanel, BorderLayout.SOUTH);
	getContentPane().add(itemView.getPanel(),BorderLayout.CENTER);
	itemView.addTaskLaunchRequestListener(new TaskLaunchRequestListener() {
	    public void taskLaunchRequested(TaskLaunchRequestEvent event) {
		TaskFrame.launchTaskFrame(event,_ivc, TaskFrame.POST_RESULT_PANEL);
	    }
	});
	itemView.addItemViewLaunchRequestListener(
	    new ItemViewLaunchRequestListener() {
	    public void itemViewLaunchRequested(
		ItemViewLaunchRequestEvent event) {
		launchItemViewFrame(event,_ivc);
	    }
	});
	itemView.addTitleListener(new TitleListener() {
	    public void titleChanged(TitleEvent event) {
		setTitle(event.getTitle());
	    }
	});
    }
	
    /**
     * A static method to launch an ItemView in a new Frame.
     * 
     * @param event An ItemViewLaunchRequestEvent containing
     *              information about the ItemView to show.
     * @param uic The UIContext to set the busy cursor over while the
     *            ItemView is being launched.
     */
    public static void launchItemViewFrame(
	final ItemViewLaunchRequestEvent event,
	final UIContext uic) {
	uic.busy(new ResultListener() {
	    public void succeeded(ResultEvent ev) {
		try {
		    String categoryName = event.getCategorySelector();
		    String itemName = event.getItemSelector();
		    String key;
		    HostContext hc = uic.getHostContext();
		    if ((key = runOnce(categoryName, itemName, hc)) != null) {
			ItemView iv =
			    ItemView.createItemView(hc, categoryName);
			iv.setItem(itemName);
			launchIV(iv, hc, categoryName, itemName, key);
		    }
		} catch (ItemViewException ive) {
		    uic.postError(ive.getMessage());
		}
		uic.notBusy();
	    }
	    public void failed(ResultEvent event) {
	    }
	});
    }

   
    private static void launchIV(ItemView iv, final HostContext hc, 
				 String categoryName, String itemName,
				 String key) {
	ItemViewFrame frame = new ItemViewFrame(iv);
	final HostContext.Client itemViewClient =
	    new HostContext.Client(frame, iv.getItemViewContext());
	hc.registerClient(itemViewClient, key);
	frame.addWindowListener(new WindowAdapter() {
	    public void windowClosed(WindowEvent event) {
		hc.unregisterClient(itemViewClient);
	    }
	});
	frame.pack();
	frame.setVisible(true);
    }

    /**
     * Pops the IV to the front and returns null if an IV was
     * already running.  Otherwise, returns the run once key.
     */
    private static String runOnce(String categoryName, String itemName,
			          HostContext hc) {
	
	// Run once logic: Concatenate category name and item
	// name; if HostContext already knows about a
	// corresponding ItemView, use it.
	String key = categoryName + "." + itemName;
	Vector vec = hc.getClients(key);
	if (vec.size() > 0) {
	    ((HostContext.Client)vec.elementAt(0)).getFrame().toFront();
	    return null;
	}
	return key;
    }
}

