//
// LabelComponentConstraints.java
//
//      A constriants object for LabelComponentLayoutManager
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import java.awt.*;

/**
 * An object that represents the constraints that determine how a
 * component is laid out in the LabelComponentLayoutManager
 */
public class LabelComponentConstraints implements Cloneable {
   
    /**
     * Pass this as the constraints object when you're adding a label.
     */
    public static LabelComponentConstraints LABEL  = new
       LabelComponentConstraints();

    /**
     * The label to attach to the component.  The label must have been
     * previously added to the container.  The default is null
     */
    public Component label;

    /**
     * Used internally by the LabelComponentLayoutManager
     */
    Component component;
    
    /**
     * The type of layout to use.  Can be set to HORIZONTAL or
     * VERTICAL.  The default is HORIZONTAL.
     */
    public byte layoutType;
 
    /**
     * Use a hoizontal alignment
     */
    public static final byte HORIZONTAL = 0;
    
    /**
     * Use a vertical alignment
     */
    public static final byte VERTICAL = 1;

    /**
     * The number of pixels to put between a label and the component
     * when the layoutType is VERTICAL.  The default is 0.
     */
    public int vgap;

    /**
     * The numner of pixels to put between the label and the component
     * when the layoutType is HORIZONTAL.  The default is 0.
     */
    public int hgap;

    /**
     * The number of pixels to put below each component in the layout
     * and the label or component directly below it.  This is ignored
     * for the bottommost component in the layout.  The default is 0.
     */
    public int bottomGap;

    /**
     * How the component should be attached to the right hand side.
     * The choices are: <dl>
     * <dt>DEFAULT
     * <dd> If the component implements DynamicSize, then attach to
     * the right, else, just give the component its preferred width,
     * even if that means that it will be clipped.
     * <dt>ALWAYS
     * <dd> Always attach the component the the right hand side
     * <dt>NEVER
     * <dd> Always give the component its preferred width
     * </dl>
     * The default is DEFAULT.
     */
    public byte rightAttachment;
    
    /**
     * If the component implements DynamicSize, then attach the
     * the right, else, just give the component its preferred width.
     */
    public static final byte DEFAULT = 0;

    /**
     * Always attach the component the the right hand side
     */
    public static final byte ALWAYS = 1;
    
    /**
     * Always give the component its preferred width
     */
    public static final byte NEVER = 2;

    /** 
     * The group that identifies this component.  This is only used
     * for components with HORIZONTAL layoutType.  All the components
     * of a single group will be laid out so that the left edges of
     * the components are aligned.  The default is "".
     */
    public String group = "";
    
    /**
     * The type of justification to use for the label.  The options
     * are LEFT or RIGHT.  The default is RIGHT.
     */
    public byte labelJustification;

    /**
     * Give the label right justification.
     */
    public static final byte RIGHT = 0;
     
    /**
     * Give the label left justification.
     */
    public static final byte LEFT = 1;

  
    /**
     * The bottommost Component in the layout will be attached to the
     * bottom of the Container if <tt>attachBottom</tt> is set to true for that
     * component.  The default is false.
     */
    public boolean attachBottom = false;
    
    /**
     * Used internally by the LabelComponentLayoutManager
     */
    int split;

    /**
     * Validates that this object contains legal values.
     * @return true if legal, false if not.
     */
    public boolean validate() {
	if (layoutType < 0 || layoutType > 1) {
	return false;
	}
	if (rightAttachment < 0 || rightAttachment > 2) {
	    return false;
	}
	if (labelJustification < 0 || labelJustification > 1) {
	    return false;
	}
	return true;
    }

    /**
     * Make a copy of this object.
     *
     * @return An exact copy of the object.
     */
    public Object clone() {
	LabelComponentConstraints o = null; 
	try {
	    o = (LabelComponentConstraints)super.clone();
	    o.group = new String(group);
	    return o;
	} catch (CloneNotSupportedException e) {
	    throw new InternalError();
	}
    }

    /**
     * Returns a string version of the constraints for debugging
     * purposes.
     * 
     * @return A string representing the object.
     *
     */
    public String toString() {
	return "\n\tlabel: " + label +
	    "\n\tcomponent: " + component +
	    "\n\tlayoutType: " + (layoutType == 0 ? "VERTICAL" :
				"HORIZONTAL") + 
	    "\n\tvgap: " + vgap + 
	    "\n\thgap: " + hgap + 
	    "\n\tbotomGap" + bottomGap + 
	    "\n\trightAttachment: " + rightAttachment +
	    "\n\tlabelJustification: " + labelJustification +
	    "\n\tattachBottom: " + attachBottom +
	    "\n\tgroup: " + group +
	    "\n\tsplit: " + split;
    }
}


    
    

    
