//
// LabelComponentLayoutManager.java
//
//      A layout manger for laying out two columns, the left of labels
//      and the right of components.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import java.awt.*;
import java.util.*;
import com.sgi.sysadm.util.*;

/** 
 * A layout manger for laying out sets of labels and components.
 * There are two layouts available for each set, horizontal with the
 * label on the left and the component on the right, and vertical with
 * the label above the component.  This LayoutManager is used by
 * LabelComponentPanel.
 *
 * See LabelComponentConstraints for more information.
 * @see LabelComponentPanel
 * @see LabelComponentConstraints
 */
/*package*/ class LabelComponentLayoutManager implements LayoutManager2,
                                                    DynamicSizeLayoutManager {

    private Vector _componentConstraints = new Vector();
    // holds LabelComponentConstraints objects for all components

    private Hashtable _constraints = new Hashtable();
    // holds a hashtable of component->constraints

    private Hashtable _groups = new Hashtable();
    // holds a hashtable of string->vector(components)

    private Vector _oneColumnComponents = new Vector();
    // holds components

    private Hashtable _labels = new Hashtable();
    // holds components

    private Dimension _currentSize;
    
    private static Dimension ZERO = new Dimension(0,0);

    /**
     * Returns the constraints object associated with a component
     * @return The LabelComponentConstraints
     */
    public LabelComponentConstraints getConstraints(Component component) {
	return  (LabelComponentConstraints)_constraints.get(component);
    }

    /**
     * Notification that a new component is being added to the
     * container. This method should not be used.  Use
     * Container.add(Component, Object) instead.
     *
     * @param name A name associated with the component.  Not
     *             currently used
     * @param comp The component to add to the layout 
     */                 
    public void addLayoutComponent(String name, Component comp) {
	Log.fatal("Use add(Component, Object) instead");
    }
 
    /*
     * A function that figures out the sizes of the left and right
     * columns.  It returns the width of the left and right columns, and
     * the height of both columns.
     *
     * @param components The components to put in the columns
     * @param width The total max width that the columns can have
     */
    private Dimension getColumnSizes(Vector components, int width) {
	int totalHeight = 0;  // keep track of the height
	int leftColWidth = 0;
	int rightColWidth = 0;
	Component component;
	Component label;
	LabelComponentConstraints constraints;
	Dimension componentSize;

	// Determine the width of the left column
 	int size = components.size();
	for (int ii = 0; ii < size; ii++) {
	    component = (Component)components.elementAt(ii);
	    constraints = 
		(LabelComponentConstraints)_constraints.get(component);
	    if (!component.isVisible()) {  
		/* only compute the size of this component and label
		 * if the component is visible
		 */
		continue;
	    }
	    label = constraints.label;
	    if (label != null && label.isVisible()) {
		leftColWidth =
		    Math.max(leftColWidth, label.getPreferredSize().width);
	    }
	    
	}

	// Determine the height and the width of the right column
	width -= leftColWidth;
	for (int ii = 0; ii < size; ii++) {
	    component = (Component)components.elementAt(ii);
	    constraints = 
		(LabelComponentConstraints)_constraints.get(component);
	    if (!component.isVisible()) {  
		/* only compute the size of this component and label
		 * if the component is visible
		 */
		continue;
	    }
	    label = constraints.label;
	    componentSize = getComponentSize(component, 
					     width - constraints.hgap,
					     constraints.rightAttachment);
	    rightColWidth =
		Math.max(rightColWidth,
			 componentSize.width + constraints.hgap);
	    
	    if (label == null || !label.isVisible()) {
		// if there's no  label, or if it's not visible
		// then we just add the height of this component  
		totalHeight += componentSize.height + constraints.bottomGap;
	    } else { 
		/* if there's a label, then we determine
		   which is taller - the label or the
		   component, and add that height to the
		   running total */
		totalHeight +=
		    Math.max(componentSize.height,
			     label.getPreferredSize().height) + 
		    constraints.bottomGap;
		;
	    }
	    // Store the split value for later use
	    constraints.split = leftColWidth;    
	}
	return new Dimension(leftColWidth + rightColWidth, totalHeight);
    }

    /**
     * Calculates the size of the components that are in the vertical
     * layout
     * @param components The components whose size to calculate
     * @param parentWidth The width of the parent.
     */
    private Dimension getOtherSizes(Vector components, int parentWidth) {
	int width = 0;
	int height = 0;
	Component component;
	Component label;
	LabelComponentConstraints constraints;
	Dimension componentSize;
	Dimension labelSize;
	int size = components.size();
	for (int ii = 0; ii < size; ii++) {
	    component = (Component)components.elementAt(ii);
	     constraints = 
		(LabelComponentConstraints)_constraints.get(component);
	    if (!component.isVisible()) {  
		/* only compute the size of this component and label
		 * if the component is visible
		 */
		continue;
	    }
	    label = constraints.label;
	    componentSize = getComponentSize(component,
					     parentWidth,
					     constraints.rightAttachment);
	    if (label != null) {
		labelSize = label.getPreferredSize();
	    } else {
		labelSize = ZERO;
	    }
	    width = Math.max(width, Math.max(componentSize.width,
					     labelSize.width));
	    height += componentSize.height + labelSize.height +
		constraints.vgap + constraints.bottomGap;
	    
	}
	return new Dimension(width, height);
    }
    
    /*
     * Called by all of the get*Size methods to determine the correct
     * size for the container.
     */
    private Dimension getSize(Container parent, int parentWidth) {
	if (_currentSize != null) {
	    return _currentSize;
	}
        Insets insets = parent.getInsets();
	parentWidth -= (insets.left + insets.right);
	Dimension colSize;
	int widthOfColumns = 0;
	int heightOfColumns = 0;
	Enumeration enum = _groups.elements();
	while(enum.hasMoreElements()) {
	    colSize = getColumnSizes((Vector)enum.nextElement(), parentWidth);
	    widthOfColumns = Math.max(widthOfColumns,colSize.width);
	    // use the larger of the minLeft and the actual left
	    heightOfColumns += colSize.height;
	}
	Dimension otherSize = getOtherSizes(_oneColumnComponents, parentWidth);
	
	int width = Math.max(widthOfColumns, otherSize.width) +
	    +insets.right
	    +insets.left;
	int height =
	    heightOfColumns    
	    +otherSize.height
	    +insets.top
	    +insets.bottom;

	// subtract off the bottomGap associated with the bottom-most
	// visible component.
	int size = _componentConstraints.size();
	for (int ii = size -1; ii >= 0; ii--) {
	    LabelComponentConstraints c =
		(LabelComponentConstraints)_componentConstraints.elementAt(ii);
	    if (c.component.isVisible()) {
		height -= c.bottomGap;
		break;
	    }
	}
        return new Dimension(width,height);
    }
    /**
     * Figure out the height that <tt>parent</tt> wants to be given
     * <tt>width</tt>.
     * 
     * @param parent Container to determine height of.
     * @param width proposed width.
     * 
     * @return desired height.
     */
    public int getPreferredHeight(Container parent, int width) {
	return getSize(parent, width).height;
    }

    /**
     * Calculate the component's size
     *
     * @param c The component
     * @param width The max possible width for the component
     * @param rightAttachment whether or not this component is
     *                        attached to the right-hand edge
     */
    private Dimension getComponentSize(Component c, 
				       int width, 
				       byte rightAttachment) {
	boolean dynamic = c instanceof DynamicSize;
	if (width < 0 ||
	    rightAttachment == LabelComponentConstraints.NEVER ||
	    (rightAttachment == LabelComponentConstraints.DEFAULT && 
	     !dynamic)) {
	    return c.getPreferredSize();
	}
	if (dynamic) {
	    return new Dimension(width,
				 ((DynamicSize)c).getPreferredHeight(width));
	}
	return new Dimension (width, c.getPreferredSize().height);
    }
    
    
    /**
     * Does the layout.
     *
     * @param parent The container containing the objects to lay out.
     */             
    public void layoutContainer(Container parent) {
	Dimension parentSize = parent.getSize();
        // Don't bother doing the layout if the parent doesn't have a
	// valid width yet.
	if (parentSize.width <= 0) {
	    return;
	}
	// We need to respect the container's insets.
        Insets insets = parent.getInsets();
        int currentYPos = insets.top;
	Dimension componentSize;
	Dimension labelSize;
	LabelComponentConstraints constraints;
	Component component;
	Component label;
	int protoRightColWidth =
	    parentSize.width - insets.left - insets.right;
	
	int num = _componentConstraints.size();

	// make sure that the constraints have corrent split values
	getSize(parent, protoRightColWidth);

	for (int ii = 0; ii < num; ii++) {
	    constraints = 
		(LabelComponentConstraints) _componentConstraints.elementAt(ii);
            component = constraints.component;
	    label = constraints.label;
	    
	    if (!component.isVisible()) {
		// Make sure component and label will be hidden
		component.setBounds(0,0,0,0);
		if (label != null) {
		    label.setBounds(0,0,0,0);
		}
		continue;
	    }
	    if (constraints.layoutType == LabelComponentConstraints.VERTICAL) { 
		componentSize = getComponentSize(component,
						 protoRightColWidth,
						 constraints.rightAttachment);
		if (label != null && label.isVisible()) {
		    labelSize = label.getPreferredSize();
		    label.setBounds(insets.left,
				    currentYPos,
				    labelSize.width,
				    labelSize.height);
		    currentYPos += labelSize.height + constraints.vgap;
		}
		if (ii == num -1 && constraints.attachBottom) {
		    // if this is the last component, then see if it
		    // wants to stretch to the bottom of the parent.
		    componentSize.height = parentSize.height -
			currentYPos - insets.bottom;
		}
		component.setBounds(insets.left,
				    currentYPos,
				    componentSize.width,
				    componentSize.height);
		currentYPos += componentSize.height + constraints.bottomGap;
		
	    } else {
		componentSize =
		    getComponentSize(component, protoRightColWidth -
				     constraints.split - constraints.hgap,
				     constraints.rightAttachment);
		if (ii == num -1 && constraints.attachBottom) {
		    // if this is the last component, then see if it
		    // wants to stretch to the bottom of the parent.
		    componentSize.height = parentSize.height -
			currentYPos - insets.bottom;
		}
		int centerLine = componentSize.height / 2;
		if (label != null && label.isVisible()) {
		    // There's a label for this component, so lay it out.
		    labelSize = label.getPreferredSize();
		    if (componentSize.height < labelSize.height) {
			centerLine = labelSize.height / 2;
		    }
		    label.setBounds(
			insets.left +
			(constraints.labelJustification == 
			 LabelComponentConstraints.RIGHT ?
			 constraints.split - labelSize.width :
			 0),
			currentYPos + centerLine -
			labelSize.height/2,
			labelSize.width,
			labelSize.height);
		}
		else labelSize = ZERO;
		
		// Now lay out the component itself, 
				
		component.setBounds(insets.left +
				    constraints.split +
				    constraints.hgap,
				    currentYPos + centerLine -
				    componentSize.height/2,
				    componentSize.width,
				    componentSize.height);
		currentYPos +=
		    Math.max(componentSize.height,labelSize.height) +
		    constraints.bottomGap;
	    }
	}
    }
    
    /** 
     * Returns the minumum size of <tt>parent</tt>.
     * @see java.awt.Component#getMaximumSize()
     * @see java.awt.LayoutManager#minimumLayoutSize
     */
    public Dimension minimumLayoutSize(Container parent) {
        return getSize(parent, -1);
    }

    /** 
     * Returns the preferred size of <tt>parent</tt>.
     * @see java.awt.Component#getPreferredSize()
     * @see java.awt.LayoutManager#preferredLayoutSize
     */
    public Dimension preferredLayoutSize(Container parent) {
        return getSize(parent, -1);
    }

    /** 
     * Returns the maximum size of <tt>parent</tt>.
     * @see java.awt.Component#getMaximumSize()
     * @see java.awt.Component#getPreferredSize()
     * @see java.awt.LayoutManager2#maximumLayoutSize
     */
    public Dimension maximumLayoutSize(Container target) {
        return getSize(target, -1);
    }
    
    /**
     * Removes the specified component from the layout.
     * @param comp the component to be removed
     */
    public void removeLayoutComponent(Component component) {
	Component label = (Component)_labels.get(component);
	if (label == null) {
	    // wasn't a label - must be a component
	    LabelComponentConstraints constraints = 
		(LabelComponentConstraints)_constraints.remove(component);
	    _componentConstraints.removeElement(constraints);
	    _oneColumnComponents.removeElement(component);
	    Enumeration enum = _groups.elements();
	    Vector v;
	    while (enum.hasMoreElements()) {
		v = (Vector)enum.nextElement();
		if (v.removeElement(component)) {
		    break;
		}
	    }
	} else {
	    // make sure no component's constraints references that
	    // lable
	    int num = _componentConstraints.size();
	    for (int ii = 0; ii < num; ii++) {
		if (((LabelComponentConstraints)
		    _componentConstraints.elementAt(ii)).label == component) {
		    Log.fatal("Must remove component that uses a label " +
			     "before removing label.");
		}
	    }
	}
    }
    
    /**
     * Adds the specified component to the layout, using the specified
     * constraint object.
     * @param component the component to be added
     * @param constraints  where/how the component is added to the layout.
     * @see java.awt.LayoutManager2#addLayoutComponent
     */
    public void addLayoutComponent(Component component, 
				   Object origConstraints) {
	Log.assert(component != null, "can't add a null component.");
	Log.assert(origConstraints != null, "can't add null constraints");
	Log.assert(origConstraints instanceof LabelComponentConstraints,
		   "constraints must be an instance of " +
		   "LableComponentConstraints");
	if (origConstraints == LabelComponentConstraints.LABEL) {
	    _labels.put(component,component);
	    return;
	}
	LabelComponentConstraints constraints =
	    (LabelComponentConstraints)
	    ((LabelComponentConstraints)origConstraints).clone();
	
	if (constraints.label != null) {
	    Log.assert(_labels.get(constraints.label) != null,
		       "Must add label to container before component.");
	}
	Log.assert(constraints.validate(),
		   "Constraints are not valid.");
	constraints.component = component;
	_constraints.put(component, constraints);
	if (constraints.layoutType == LabelComponentConstraints.VERTICAL) {
	    _oneColumnComponents.addElement(component);
	} else {
	    Vector v = (Vector)_groups.get(constraints.group);
	    if (v == null) {
		v = new Vector();
		_groups.put(constraints.group,v);
	    }
	    v.addElement(component);
	}
	_componentConstraints.addElement(constraints);
    }
    
    /**
     * Returns the alignment along the x axis. 
     */
    public float getLayoutAlignmentX(Container target) {
        return 0.5F;
    }
    
    
    /**
     * Returns the alignment along the y axis. 
     */
    public float getLayoutAlignmentY(Container target) {
        return 0.5F;
    }
    

    /**
     * Invalidates the layout, indicating that if the layout manager
     * has cached information it should be discarded.
     */
    public void invalidateLayout(Container target) {
	_currentSize = null;
    }    
}
