//
// LabelComponentPanel.java
//
//      A Panel for laying out labels and components in one and two columns.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import java.awt.*;
import javax.swing.*;
import com.sgi.sysadm.util.*;

/**
 * A Panel for laying out labels and Components is one and two
 * columns.  The layout of the Components is controlled with
 * resources.  All of the resources have three forms.  The prefix is
 * one of:
 * <ol>
 * <li>component name -  This is the name of the Component, as passed to the
 *     <tt>addComponent</tt> method.
 * <li>component's class name - The package qualified name of the
 *     Component that was passed tothe <tt >addComponent</tt> method.
 * <li>"LabelComponentPanel" - The liternal string "LabelComponentPanel".
 * </ol>
 * The resources are looked up, in order, until one is found.  The
 * defaults for the third case ("LabelComponentPanel") are in
 * SysadmUIP.properties.
 */
public class LabelComponentPanel extends RPanel {
    private static final String CLASS_NAME = "LabelComponentPanel";

    /**
     * A resource <i>&lt;component name | component's class name |
     * "LabelComponentPanel"&gt;.layoutType</i> determines what type of
     * layout the component will have.  The choices are "horizontal"
     * or "vertical"
     * @see com.sgi.sysadm.ui.LabelComponentConsraints.layoutType
     */
    public static final String LAYOUT_TYPE = ".layoutType";
    private static final String HORIZONTAL = "horizontal";
    private static final String VERTICAL = "vertical";

    /**
     * A resource <i>&lt;component name | component's class name |
     * "LabelComponentPanel"&gt;.vgap</i> determines the vertical gap (in
     * points) between the label and the component when the
     * LAYOUT_TYPE is "vertical"
     * @see com.sgi.sysadm.ui.LabelComponentConsraints.vgap
     */
    public static final String VGAP = ".vgap";

    /**
     * A resource <i>&lt;component name | component's class name |
     * "LabelComponentPanel"&gt;.hgap</i> determines the horizontal gap
     * (in points)  between the label and the component when the
     * LAYOUT_TYPE is "horizontal"
     * @see com.sgi.sysadm.ui.LabelComponentConsraints.hgap
     */
    public static final String HGAP = ".hgap";

    /**
     * A resource <i>&lt;component name | component's class name |
     * "LabelComponentPanel"&gt;.bottomGap</i> determines the vertical gap
     * (in points)  between one component and the next.  It is ignored
     * for the bottom most component.
     * @see com.sgi.sysadm.ui.LabelComponentConsraints.bottomGap
     */
    public static final String BOTTOM_GAP = ".bottomGap";

    /**
     * A resource <i>&lt;component name | component's class name |
     * "LabelComponentPanel"&gt;.rightAttachment</i> determines the type
     * of attachment to give a component.
     * The choices are "default", "always" or "never"
     * @see com.sgi.sysadm.ui.LabelComponentConsraints.rightAttachment
     */
    public static final String RIGHT_ATTACHMENT = ".rightAttachment";
    private static final String DEFAULT = "default";
    private static final String ALWAYS = "always";
    private static final String NEVER = "never";

    /**
     * A resource <i>&lt;component name | component's class name |
     * "LabelComponentPanel"&gt;.group</i> determines the group of
     * horizontally laid out components that this component belongs
     * to.  The members of a group will have the left edges of their
     * components aligned.
     * @see com.sgi.sysadm.ui.LabelComponentConsraints.group
     */
    public static final String GROUP = ".group";

    /**
     * A resource <i>&lt;component name | component's class name |
     * "LabelComponentPanel"&gt;.labelJustification</i> determines the
     * justification for the label.  The choices are "left" and
     * "right"
     * @see com.sgi.sysadm.ui.LabelComponentConsraints.labelJustification
     */
    public static final String LABEL_JUSTIFICATION = ".labelJustification";
    private static final String LEFT = "left";
    private static final String RIGHT = "right";

    /**
     * A resource <i>&lt;component name | component's class name |
     * "LabelComponentPanel"&gt;.attachBottom</i> determines whether the
     * bottom-most component in the layout is stretched to touch the
     * bottom of the panel.
     * @see com.sgi.sysadm.ui.LabelComponentConsraints.attachBottom
     */
    public static final String BOTTOM = ".attachBottom";

    private LabelComponentConstraints _c =
	new LabelComponentConstraints();
    private LabelComponentLayoutManager _manager =
	new LabelComponentLayoutManager();
    private ResourceStack _rs;

    /**
     * Create a LabelComponentPanel
     *
     * @param rs The ResourceStack to use to lookup resources.
     */
    public LabelComponentPanel(ResourceStack rs) {
	setLayout(_manager);
	_rs = rs;
    }

    /**
     * Adds a component and a LinkLabel to a page.
     * The LinkLabel and the <tt>component</tt> will share
     * <tt>name</tt> as
     * their name. The new component and label will appear beneath any
     * previously added components.
     *
     * @param component The component to add
     * @param label The label that describes this component
     */
    public void addComponent(Component component,
			     String name) {
	LinkLabel label = new LinkLabel(_rs, name);
	addComponent(component, name, label);
    }

    /**
     * Adds a component and a label to the page.  The
     * <tt>component</tt> and <tt>label</tt> will appear beneath any
     * previously added components.
     *
     * @param component The component to add
     * @param name The name of the component.  Will be used in
     *		resource lookups.
     * @param label The label that describes this component
     */
    public void addComponent(Component component,
			     String name,
			     Component label) {
	String className = component.getClass().getName();
	_c.label = label;
	try {
	    _c.layoutType = ResourceStack.mapStringToByte(
		_rs.getString(createLookup(name, className, LAYOUT_TYPE)),
		new String[] {HORIZONTAL, VERTICAL },
		new byte[] {LabelComponentConstraints.HORIZONTAL,
			    LabelComponentConstraints.VERTICAL
		});
	} catch (IndexOutOfBoundsException e) {
	    Log.fatal("The layout type of the component named \"" +
		     name + "\" was not either \"" + HORIZONTAL +
		     "\" or \"" + VERTICAL + "\" as expected");
	}
	_c.vgap = _rs.getPixels(createLookup(name, className, VGAP));
	_c.hgap = _rs.getPixels(createLookup(name, className, HGAP));
	_c.bottomGap = _rs.getPixels(createLookup(name, className, BOTTOM_GAP));
	try {
	    _c.rightAttachment = ResourceStack.mapStringToByte(
		_rs.getString(createLookup(name, className, RIGHT_ATTACHMENT)),
		new String[] { DEFAULT, NEVER, ALWAYS },
		new byte[] {LabelComponentConstraints.DEFAULT,
			    LabelComponentConstraints.NEVER,
			    LabelComponentConstraints.ALWAYS
		});
	} catch (IndexOutOfBoundsException e) {
	    Log.fatal("The right attachement of the component named \"" +
		     name + "\" was not either \"" + DEFAULT +
		     "\", \"" + NEVER + "\"  or \"" + VERTICAL +
		     "\" as expected");
	}
	_c.group = _rs.getString(createLookup(name, className, GROUP));
	_c.attachBottom = _rs.getBoolean(createLookup(name, className, BOTTOM));
	try {
	    _c.labelJustification = ResourceStack.mapStringToByte(
		_rs.getString(
		    createLookup(name, className, LABEL_JUSTIFICATION)),
		new String[] {LEFT , RIGHT },
		new byte[] {LabelComponentConstraints.LEFT,
			    LabelComponentConstraints.RIGHT});
	} catch (IndexOutOfBoundsException e) {
	    Log.fatal("The left justification of the component named \"" +
		     name + "\" was not either \"" + LEFT +
		     "\"  nor \"" + RIGHT +
		     "\" as expected");
	}

	if (label != null) {
	    add(label, LabelComponentConstraints.LABEL);
	}
	add(component, _c);
    }

    private String[] createLookup(String name,
				  String className,
				  String postfix) {
	return new String[] {
	    name + postfix,
		className + postfix,
		CLASS_NAME + postfix
		};
    }

    /**
     * Removes a component and any label associated with it.
     *
     * @param component The component to remove
     */
    public void removeComponent(Component component) {
	Component label = _manager.getConstraints(component).label;
	remove(component);
	if (label != null) {
	    remove(label);
	}
    }
}
