//
// LinkLabelBase.java
//
//      A subclass of RichTextArea that uses a standardized look.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;
import java.awt.*;
import java.util.*;

/**
 * This class can be used wherever a RichTextArea would be used,
 * but where a standardized look is desired.
 * <p>
 * A LinkLabel's text and link is determined by resources.
 * The string that is displayed in the label is determined by using
 * the <tt>LABEL</tt> property.  The link that the label launches is
 * determined by the <tt>GLOSSARY</tt> property.  Both of these
 * properties are of the form  <i>&lt;name&gt;.something</i>, where
 * <ii>name</ii> is the <code>name</code> argument passed to the constructor.
 * Since this class extends RichTextArea, it can handle launching
 * everything that RichTextArea handles.  Because this class will just
 * show a normal label instead of a link if the <tt>GLOSSARY</tt>
 * property is not found, it can be used for normal labels when there
 * will be a possibility of desiring a link label at a later time.
 *
 * <p><i>Example</i>
 * <br>
 * A call to <code>new LinkLabelBase(rs, "MyLabel", "LinkLabel")</code>, and a resource
 * file with <pre>
 * MyLabel.label = "My Nice Label:
 * MyLabel.glossary = glossary.myGloss
 * </pr>
 * Would show a LinkLabel with text "My Nice Label:".  When the user
 * clicked on the link, the "glossary.myGloss" glossary entry would be
 * displayed.
 */

public class LinkLabelBase extends RichTextArea {

    private String _glossary;
    private String _preTags;
    private String _postTags;
    private String _label;

    /**
     * A resource <i>&lt;name&gt;.label</i> is the text that will
     * be displayed in the link.  <i>name</i> is the name argument
     * passed to the constructor.  If this resource
     * is not found, the name will be used as the displayed string.
     */
    public static final String LABEL = ".label";

    /**
     * A resource <i>&lt;name&gt;.glossary</i> is the glossary
     * entry to display when the link is clicked.  <i>name</i> is
     * the name argument passed to the constructor.  If this resource
     * is not found, the link will be rendered as a normal label
     * instead of as a link.
     */
    public static final String GLOSSARY = ".glossary";

    /**
     * A resource <i>&lt;className&gt;.preTags</i> is a string that
     * will be inserted at the beginning of the displayed text.
     * <i>className</i> is  the className argument passed to the
     * constructor. For example, if this resource were set to
     * <i>&lt;B&gt;</i>, then  the label would be bold.
     */
    public static final String PRE_TAGS = ".preTags";

    /**
     * A resource <i>&lt;className&gt;.postTags</i> is a string that
     * will be inserted at the end of the displayed text.
     * <i>className</i> is the className argument passed to the
     * constructor.  For example, this resource should be
     * <i>&lt;/B&gt;</I> if PRE_TAGS was set to <i>&lt;B&gt;</i>.
     */
    public static final String POST_TAGS = ".postTags";

    private static final Insets zeroInsets = new Insets(0,0,0,0);
    
    /**
     * The constructor
     *
     * @param rs The ResourceStack to use to lookup the resources
     * @param name The name used for the LABEL and GLOSSARY
     *               resources.
     * @param className The className used for the PRE_TAGS and
     *                   POST_TAGS resources, as well as the name of
     *                   the RichTextArea that is created.
     */
    public LinkLabelBase(ResourceStack rs,
			 String name,
			 String className) {

	super(className, rs);
	String label;
	String glossary;

	String lookup = name + LABEL;
	label = rs.getString(lookup, null);
	if (label == null) {
	    if (Log.isLevelOn(Log.DEBUG)) {
		// String concats are expensive - skip it
		// if it's not going to be used.
		Log.debug(CLASS_NAME,
			  "Didn't find a resource \"" + lookup +
			  "\" in the resource stack.  Will use \"" +
			  name + "\".");
	    }
	    label = name;
	}
	lookup = name + GLOSSARY;
	_glossary  = rs.getString(lookup, null);
	if (_glossary == null) {
	     if (Log.isLevelOn(Log.DEBUG)) {
		 // String concats are expensive - skip it
		 // if it's not going to be used.
		 Log.debug(CLASS_NAME,
			   "Didn't find a resource \"" + lookup +
			   "\" in the resource stack.  Will not set a " +
			   "glossary entry");
	     }
	}
	
	_preTags = rs.getString(className + PRE_TAGS, "");
	_postTags = rs.getString(className + POST_TAGS, "");

	setText(label);
	setAutoWrap(false);
	setMargins(zeroInsets);
    }

    /**
     * Set the text to be displayed.
     *
     * @param text The text to be displayed.
     */
    public void setText(String text) {
	_label = text;
	if (_glossary == null) {
	    super.setText(_preTags + _label + _postTags);
	} else {
	    super.setText(_preTags + "<A HREF=" + _glossary + ">" + _label +
			  "</A>" + _postTags);
	}
    }

    /**
     * Set the glossary entry to be displayed.
     *
     * @param text The link name of the glossary entry to be displayed.
     */
    public void setGlossary(String text) {
	_glossary = text;
	setText(_label);
    }
}
