//
// LinkPageLayout.java
//
//	Layout manager for vertical list of links with associated icons.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;
import java.awt.*;
import java.util.*;

/**
 * LinkPageLayout is a layout manager intended for conainers that want to
 * display a hierarchical list of objects, including links that have an
 * associated icon, text, separators, and a container title.  Objects
 * are added using one of the pre-defined constraints.
 */
public class LinkPageLayout implements LayoutManager2 {
    /**
     * The title layout constraint.  Indicates that the component
     * should be centered at the top of the container.  The container may have
     * only one title.  If multiple TITLE components are added, the
     * last one added will be displayed at the top of the container.
     */
    static public final String TITLE = "title";

    /**
     * The text layout constraint.  Indicates that the component
     * should be placed at the current indent level and stretch to the
     * full width of the container.
     */
    static public final String TEXT = "text";

    /**
     * The icon layout constraint.  Indicates that the next add will
     * be of a link that will be displayed to the right of this icon.
     */
    static public final String ICON = "icon";

    /**
     * The link layout constraint.  Indicates that an icon was just
     * added and this link should be displayed to the right of that
     * icon.
     */
    static public final String LINK = "link";

    /**
     * The separator layout constraint.  Indicates that the component
     * should stretch the width of the container.
     */
    static public final String SEPARATOR = "separator";

    /**
     * The indent layout constraint.  Increases the indent level by
     * INDENT_SPACING for all subsequent components added.
     */
    static public final String INDENT = "indent";

    /**
     * The outdent layout constraint.  Reduces the indent spacing by
     * INDENT_SPACING for all subsequent components added.
     */
    static public final String OUTDENT = "outdent";

    /**
     * The extra-vertical-space layout constraint.  While all
     * components have a small amount of space in between them (see
     * VERTICAL_SPACING) the layout client may wish to insert
     * additional space before or after a component.  The
     * extra-vertical-space constraint adds blank vertical
     * space that is the height of the added component.  For example:
     * <PRE>
     * Component extraSpace = new JComponent();
     * extraSpace.setSize(width, height);
     * linkPage.add(extraSpace, LinkPageLayout.EXTRA_VERTICAL_SPACE);
     * </PRE>
     * If the client wishes to add the same amount of extra vertical
     * space in multiple locations in the container, the same
     * component can be added multiple times.  However, any change in
     * size of the component will appear in all of the places the
     * component was added.
     */
    static public final String EXTRA_VERTICAL_SPACE = "EXTRA_VERTICAL_SPACE";
    
    /**
     * The property <I>LinkPageLayout.verticalSpacing</I> controls the
     * amount of space, in points,  that is added between each
     * component.
     */
    static public final String VERTICAL_SPACING =
				    "LinkPageLayout.verticalSpacing";

    /**
     * The property <I>LinkPageLayout.iconSpacing</I> is the amount of
     * space, in points, that is placed in between an icon and its
     * link.
     */
    static public final String ICON_SPACING =
	"LinkPageLayout.iconSpacing";

    /**
     * The property <I>LinkPageLayout.indentSpacing</I> is the amount
     * of space, in points, that the components are indented each time
     * the INDENT layout constraint is used.
     */
    static public final String INDENT_SPACING =
	"LinkPageLayout.indentSpacing";

    /**
     * The property <I>LinkPageLayout.minimumWidth</I> is the minimum
     * width, in points, of the container being laid out.
     */
    static public final String MINIMUM_WIDTH =
	"LinkPageLayout.minimumWidth";

    /**
     * The property <I>LinkPageLayout.minimumHeight</I> is the minimum
     * height, in points, of the container being laid out.
     */
    static public final String MINIMUM_HEIGHT = "LinkPageLayout.minimumHeight";

    private final int TITLE_TYPE = 0;
    private final int TEXT_TYPE = 1;
    private final int LINK_TYPE = 2;
    private final int SEPARATOR_TYPE = 3;
    private final int INDENT_TYPE = 4;
    private final int OUTDENT_TYPE = 5;
    private final int EXTRA_VERTICAL_SPACE_TYPE = 6;
    
    private Container _parent;
    private Component _title;
    
    private class LayoutItem {
	private int _type;
	private Component _comp;
	private Component _icon;
	private Dimension _indentGroupIconSize;
	
	LayoutItem(int type) {
	    Log.assert(type == TITLE_TYPE || type == TEXT_TYPE ||
		       type == LINK_TYPE || type == SEPARATOR_TYPE ||
		       type == INDENT_TYPE || type == OUTDENT_TYPE ||
		       type == EXTRA_VERTICAL_SPACE_TYPE,
		       "invalid LayoutItem type");
	    _type = type;
	}
    }

    private Vector _itemList = new Vector();
    private int _titleIndex;
    private Dimension _firstIndentGroupIconSize = new Dimension(0, 0);
    private Dimension _currentIndentGroupIconSize;
    private LayoutItem _pendingLink = null;
    private Stack _iconSizeStack = new Stack();

    private int _indentSpacing;
    private int _verticalSpacing;
    private int _iconSpacing;
    private int _minimumWidth;
    private int _minimumHeight;

    private String _badParent =
	"parent parameter does not match value passed to constructor.";
    
    /**
     * Constructor.
     *
     * @param parent The container being laid out.
     * @param ResourceStack The ResourceStack that contains the
     *                      LinkPageLayout properties.
     */
    public LinkPageLayout(Container parent, ResourceStack rs) {
	_parent = parent;
	
	_verticalSpacing = rs.getPixels(VERTICAL_SPACING);
	_iconSpacing = rs.getPixels(ICON_SPACING);
	_indentSpacing = rs.getPixels(INDENT_SPACING);
	_minimumWidth = rs.getPixels(MINIMUM_WIDTH);
	_minimumHeight = rs.getPixels(MINIMUM_HEIGHT);
	
	// Add a fixed title item.  There is only one title per
	// container.  Multiple adds will result in the last title
	// added being displayed.
	LayoutItem titleItem = new LayoutItem(TITLE_TYPE);
	_itemList.addElement(titleItem);
	_titleIndex = _itemList.size() - 1;

	_currentIndentGroupIconSize = _firstIndentGroupIconSize;
    }
    
    /**
     * @see java.awt.LayoutManager#addLayoutComponent
     */
    public void addLayoutComponent(String name, Component comp) {
	Log.fatal("Use add(Component, Object) instead.");
    }
    
    /**
     * This method has an empty implementation.  LinkPageLayout does
     * not support removal of components at this time.
     *
     * @see java.awt.LayoutManager#removeLayoutComponent
     */
    public void removeLayoutComponent(Component comp) {
	// Do nothing so that clients can add components multiple times.
    }
    
    private Dimension getSize(Container parent) {
	Dimension size = new Dimension(0, 0);
	
	int indentLevel = 0;
	int indent = 0;
	Dimension iconSize = _firstIndentGroupIconSize;
	Enumeration items = _itemList.elements();
	while (items.hasMoreElements()) {
	    LayoutItem item = (LayoutItem)items.nextElement();
	    Dimension compSize = (item._comp != null ?
				    getComponentSize(item._comp) :
				    new Dimension(0,0));
	    
	    switch (item._type) {
	    case TITLE_TYPE:
		size.width = Math.max(size.width, compSize.width);
		size.height += compSize.height;
		break;
		
	    case TEXT_TYPE:
		size.width = Math.max(size.width, indent + compSize.width);
		size.height += compSize.height;
		break;
	    
	    case LINK_TYPE:
		size.width = Math.max(size.width,
				      indent + iconSize.width + _iconSpacing +
				      compSize.width);
		size.height += Math.max(iconSize.height, compSize.height);
		break;
		
	    case SEPARATOR_TYPE:
		// Separators do not affect the width
		size.height += compSize.height;
		iconSize = item._indentGroupIconSize;
		break;
		
	    case INDENT_TYPE:
		if (iconSize.width == 0) {
		    // There were no icons at the previous level.
		    // Simply indent by _indentSpacing.
		    indent += _indentSpacing;
		} else {
		    // Indent past the icons and spacing for the icons
		    // from the previous level.
		    indent += iconSize.width + _iconSpacing;
		}
		indentLevel++;
		iconSize = item._indentGroupIconSize;
		
		// Since this is not a component to lay out, continue
		// to the next item.
		continue;
		
	    case OUTDENT_TYPE:
		indentLevel--;
		Log.assert(indentLevel >= 0, "too many OUTDENTs");
		iconSize = item._indentGroupIconSize;
		if (iconSize.width == 0) {
		    indent -= _indentSpacing;
		} else {
		    indent -= iconSize.width + _iconSpacing;
		}
		
		// Since this is not a component to lay out, continue
		// to the next item.
		continue;
	
	    case EXTRA_VERTICAL_SPACE_TYPE:
		size.height += compSize.height;
		
		// Since this is not a component to lay out, continue
		// to the next item.
		continue;
	    }
	    
	    size.height += _verticalSpacing;
	}
	
	if (size.height > 0) {
	    size.height -= _verticalSpacing;
	}
	
	Insets insets = _parent.getInsets();
	size.width += insets.left + insets.right;
	size.height += insets.top + insets.bottom;
	
	return size;
    }
    
    /**
     * @see java.awt.LayoutManager#preferredLayoutSize
     */
    public Dimension preferredLayoutSize(Container parent) {
	Log.assert(parent == _parent, _badParent);
	
	return getSize(parent);
    }
    
    /**
     * @see java.awt.LayoutManager#minimumLayoutSize
     */
    public Dimension minimumLayoutSize(Container parent) {
	Log.assert(parent == _parent, _badParent);
	
	Insets insets = parent.getInsets();
	return new Dimension(_minimumWidth + insets.left + insets.right,
			     _minimumHeight + insets.top + insets.bottom);
    }

    /**
     * @see java.awt.LayoutManager#layoutContainer
     */
    public void layoutContainer(Container parent) {
	Dimension parentSize = new Dimension(parent.getSize());
	if (parentSize.width <= 0) {
	    // Don't bother doing the layout if the parent isn't
	    // a reasonable size yet.
	    return;
	}
	
	// Calculate the parent width after insets
	Insets insets = parent.getInsets();
	parentSize.width -= (insets.left + insets.right);
	
	int currentYPos = insets.top;
	int availWidth = parentSize.width;
	
	int indentLevel = 0;
	int indent = 0;
	Dimension iconSize = _firstIndentGroupIconSize;
	Enumeration items = _itemList.elements();
	while (items.hasMoreElements()) {
	    LayoutItem item = (LayoutItem)items.nextElement();
	    Dimension compSize = (item._comp != null ?
				    getComponentSize(item._comp) :
				    new Dimension(0,0));
	    
	    int XPos = insets.left + indent;
	    int YPos = currentYPos;
	    int width = Math.min(availWidth, compSize.width);
	    int height = getPreferredComponentHeight(item._comp, width);
	    
	    switch (item._type) {
	    case TITLE_TYPE:
		// 642942: titles are no longer centered.
		width = availWidth;
		height = getPreferredComponentHeight(item._comp, width);
		break;
		
	    case TEXT_TYPE:
		break;
	    
	    case LINK_TYPE:
		// Layout the icon first (if any) and then the link
		width = availWidth - iconSize.width - _iconSpacing;
		height = getPreferredComponentHeight(item._comp, width);
		if (item._icon != null) {
		    int iconYPos;
		    if (iconSize.height >= height) {
			YPos += (iconSize.height - height)/2;
			iconYPos = currentYPos;
		    } else {
			iconYPos = currentYPos + (height - iconSize.height)/2;
		    }
		    
		    item._icon.setBounds(XPos, iconYPos,
					 iconSize.width, iconSize.height);
		}
		XPos += iconSize.width + _iconSpacing;
		break;
		
	    case SEPARATOR_TYPE:
		width = availWidth;
		height = compSize.height;
		iconSize = item._indentGroupIconSize;
		break;
		
	    case INDENT_TYPE:
		if (iconSize.width == 0) {
		    // There were no icons at the previous level.
		    // Simply indent by _indentSpacing.
		    indent += _indentSpacing;
		} else {
		    // Indent past the icons and spacing for the icons
		    // from the previous level.
		    indent += iconSize.width + _iconSpacing;
		}
		indentLevel++;
		iconSize = item._indentGroupIconSize;
		availWidth = parentSize.width - indent;
		
		// Since this is not a component to lay out, continue
		// to the next item.
		continue;
		
	    case OUTDENT_TYPE:
		indentLevel--;
		iconSize = item._indentGroupIconSize;
		if (iconSize.width == 0) {
		    indent -= _indentSpacing;
		} else {
		    indent -= iconSize.width + _iconSpacing;
		}
		availWidth = parentSize.width - indent;
		
		// Since this is not a component to lay out, continue
		// to the next item.
		continue;
	    
	    case EXTRA_VERTICAL_SPACE_TYPE:
		currentYPos += compSize.height;
		
		// Since this is not a component to lay out, continue
		// to the next item.
		continue;
	    }
	    
	    // Lay out the component.
	    if (item._comp != null) {
		item._comp.setBounds(XPos, YPos, width, height);
		currentYPos += (item._type == LINK_TYPE ?
				    Math.max(iconSize.height, height) : height)
			       + _verticalSpacing;
	    }
	}
    }

    /**
     * @see java.awt.LayoutManager2#addLayoutComponent
     */
    public void addLayoutComponent(Component comp, Object constraints) {
	Log.assert(constraints instanceof String,
		   "Constraints must be a String");

	String linkExpected =
	    "add of LinkPageLayout.ICON must be followed by an add " +
	    "of a LinkpageLayout.LINK";

	String type = (String)constraints;
	LayoutItem item;
	
	if (type == TITLE) {
	    Log.assert(_pendingLink == null, linkExpected);
	    item = (LayoutItem)_itemList.elementAt(_titleIndex);
	    item._comp = comp;
	} else if (type == TEXT) {
	    Log.assert(_pendingLink == null, linkExpected);
	    item = new LayoutItem(TEXT_TYPE);
	    item._comp = comp;
	    _itemList.addElement(item);
	} else if (type == ICON) {
	    Log.assert(_pendingLink == null, linkExpected);
	    item = new LayoutItem(LINK_TYPE);
	    item._icon = comp;
	    Dimension compSize = getComponentSize(comp);
	    _currentIndentGroupIconSize.width = 
		Math.max(_currentIndentGroupIconSize.width,
			 compSize.width);
	    _currentIndentGroupIconSize.height =
		Math.max(_currentIndentGroupIconSize.height,
			 compSize.height);
	    _pendingLink = item;
	} else if (type == LINK) {
	    Log.assert(_pendingLink != null,
		       "add of LinkPageLayout.LINK must be preceeded "
		       + "by an add of LinkPageLayout.ICON");
	    _pendingLink._comp = comp;
	    _itemList.addElement(_pendingLink);
	    _pendingLink = null;
	} else if (type == SEPARATOR) {
	    Log.assert(_pendingLink == null, linkExpected);
	    item = new LayoutItem(SEPARATOR_TYPE);
	    item._comp = comp;
	    item._indentGroupIconSize = new Dimension(0, 0);
	    _itemList.addElement(item);
	    _currentIndentGroupIconSize = item._indentGroupIconSize;
	} else if (type == INDENT) {
	    Log.assert(_pendingLink == null, linkExpected);
	    item = new LayoutItem(INDENT_TYPE);
	    item._indentGroupIconSize = new Dimension(0, 0);
	    _itemList.addElement(item);
	    _iconSizeStack.push(_currentIndentGroupIconSize);
	    _currentIndentGroupIconSize = item._indentGroupIconSize;
	} else if (type == OUTDENT) {
	    Log.assert(_pendingLink == null, linkExpected);
	    item = new LayoutItem(OUTDENT_TYPE);
	    item._indentGroupIconSize = (Dimension)_iconSizeStack.pop();
	    _itemList.addElement(item);
	    _currentIndentGroupIconSize = item._indentGroupIconSize;
	} else if (type == EXTRA_VERTICAL_SPACE) {
	    Log.assert(_pendingLink == null, linkExpected);
	    item = new LayoutItem(EXTRA_VERTICAL_SPACE_TYPE);
	    item._comp = comp;
	    _itemList.addElement(item);
	} else {
	    Log.fatal("invalid component type added");
	}
    }
    
    /**
     * @see java.awt.LayoutManager2#maximumLayoutSize
     */
    public Dimension maximumLayoutSize(Container parent) {
	return preferredLayoutSize(parent);
    }
    
    /**
     * @return java.awt.Component.CENTER_ALIGNMENT
     *
     * @see java.awt.LayoutManager2#getLayoutAlignmentX
     */
    public float getLayoutAlignmentX(Container parent) {
	Log.assert(parent == _parent, _badParent);
    
	return java.awt.Component.CENTER_ALIGNMENT;
    }
    
    /**
     * @return java.awt.Component.CENTER_ALIGNMENT
     *
     * @see java.awt.LayoutManager2#getLayoutAlignmentY
     */
    public float getLayoutAlignmentY(Container parent) {
	Log.assert(parent == _parent, _badParent);
    
	return java.awt.Component.CENTER_ALIGNMENT;
	
    }
    
    /**
     * @see java.awt.LayoutManager2#invalidateLayout
     */
    public void invalidateLayout(Container parent) {
	Log.assert(parent == _parent, _badParent);
    }
    
    private Dimension getComponentSize(Component comp) {
	if (!(comp instanceof RichTextComponent)) {
	    return comp.getPreferredSize();
	}
	
	Dimension size = new Dimension(comp.getSize());
	if (size.width <= 0) {
	    return comp.getPreferredSize();
	}

	size.height =
	    ((RichTextComponent)comp).getPreferredHeight(size.width);
	
	return size;
    }
    
    private int getPreferredComponentHeight(Component comp, int width) {
	if (comp == null) {
	    return 0;
	}
	
	if (!(comp instanceof RichTextComponent)) {
	    return comp.getPreferredSize().height;
	}
	
	return
	    ((RichTextComponent)comp).getPreferredHeight(width);
    }
}
