//
//  ProductAttributeSetter.java
//
//	An interface for classes that can set product Attributes
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import com.sgi.sysadm.util.*;

/** 
 * An interface for classes that can set Product Attributes.
 * Product Attributes are pieces of TaskData that are applicable to
 * many of the Tasks in a particular product.  A Task can request that
 * Product Attributes for its product be loaded by using the
 * PRODUCT_ATTRIBUTES property.  The HostContext will then instantiate
 * the ProductAttributeSetter that has been specifed with the SETTER
 * property.
 * <p>
 * After instantiating the ProductAttributeSetter, HostContext calls
 * the <tt>initializeProductAttributeSetter</tt>.  HostContext then calls
 * <tt>setProductAttributes</tt>.  <tt>setProductAttributes</tt> can
 * be called multiple times, because it is possible for HostContext to
 * want to refresh the Product Attributes.
 * 
 * @see Task#PRODUCT_ATTRIBUTES
 * @see HostContext#SETTER
 */
public interface ProductAttributeSetter {
    
    /**
     * Initializes the setter.  The object need not take any action,
     * but can perform any setup here.  It's also a good idea to save
     * the parameters passed in here for later use.
     *
     * @param hostContext the HostContext to call setProductAttributes() on
     * @param uic The UIContext to use for showing any UI
     */
    public void initializeProductAttributeSetter(HostContext hostContext,
						 UIContext uic);

    /**
     * Sets the Product Attributes.  The object should do whatever is
     * necessary to obtain the Product Attributes (which may involve
     * calls to the server, showing UI's, etc).  When the Product Attributes
     * are known, this method should call the
     * <tt>setProductAttribute</tt> method on HostContext for each
     * attribute.  When done
     * setting the Product Attributes, this method must call the
     * success method of <tt>listener</tt> to notify HostContext that
     * it should proceed.  If there was some reason that the Product
     * Attributes could not be set, then this method should call the
     * <tt>failed</tt> method of <tt>listener</tt> with the reason set
     * appropriately.
     * <p> 
     * While Tasks are one consumer of Product Attributes, any code can
     * make a request to HostContext that Product Attributes be loaded.  It
     * may be desireable in these cases to control how the attributes
     * are collected.  For that reason, a <tt>context</tt> object is passed.
     * This is the same <tt>context</tt> that is passed to
     * HostContext's <tt>getProductAttribute</tt> or
     * <tt>getProductAttributes</tt> method.  It is up to the
     * implementor to decide on what <tt>context</tt> should
     * represent.  If <tt>setProductAttributes</tt> is called in
     * response to a Task being loaded by a TaskLoader, then the
     * context will be <tt>null</tt>
     * 
     * @param context An Object that can be used to control the setter.
     * @param listener The listener to notify when all of the product
     *                 Attributes have been set.
     *
     * @see HostContext#setProductAttribute
     * @see HostContext#getProductAttribute
     * @see HostContext#getProductAttributes
     */
    public void setProductAttributes(Object context,
				     ResultListener listener);
}
