//
// ProductInfo.java
//
//	Determines product information about a particular product.
//	Currently only support product attributes.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import java.util.*;
import java.io.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.util.*;

public abstract class ProductInfo {

    protected static final String PRODUCT_INFO = "productInfo";

    private Hashtable _taskLists = new Hashtable();
    private Vector _listeners = new Vector();
    
    // ListenerElem is an element in our _listeners Vector.  Each
    // element corresponds to a getProductInfo call that needs to have a
    // result returned via ResultEvent.
    private class ListenerElem {
	String _product;
	ResultListener _listener;

	ListenerElem(String product, ResultListener listener) {
	    _product = product;
	    _listener = listener;
	}
    };

    /**
     * Set the HostContext needed for creating ProductInfo instances.
     * 
     * @param hostContext HostContext for creating ProductInfo
     *                    instances.
     */
    public void setHostContext(HostContext hostContext) {
    }

    /**
     * Determines the product attributes pertaining to a product.
     * 
     * @param product The name of the product
     * @param listener Gets result of this operation.  The
     *                 <tt>getResult</tt> method should be called on
     *                 the ResultEvent passed to <tt>listener.succeeded</tt>
     *                 to get an ResourceBundle containing the Attributes.
     */
    public synchronized void getProductAttrs(String product,
					    ResultListener listener) {
	_listeners.addElement(new ListenerElem(product, listener));
	computeProductAttrs(product);
    }
    
    /**
     * Subclasses override this method that ProductInfo calls when it
     * needs the product attributes for a particular product.  Subclass
     * must eventually call setProductAttrs when it is done computing the
     * product attributes.
     * 
     * @param product The product to get the attributes of
     */
    protected abstract void computeProductAttrs(String product);

    /**
     * Called by subclass to report result of computeProductAttrs
     * 
     * @param product The product that this result is for
     * @param attrs String the attrs for this product
     */
    protected synchronized void setProductAttrs(String product, 
						String attrs) {
	ResourceBundle bundle = null;
	try { 
	    bundle = new PropertyResourceBundle(
		new ByteArrayInputStream(attrs.getBytes()));
	} catch (IOException e) {
	    // Ignore.  It's coming from a String, so it'll be okay
	}
	ResultEvent event = new ResultEvent(ProductInfo.this, bundle);
	notifyListeners(product, event, true);
    }

    private void notifyListeners(String product,
				 ResultEvent event,
				 boolean success) {
	for (int ii = _listeners.size() -1; ii >= 0; ii--) {
	    ListenerElem elem = (ListenerElem)_listeners.elementAt(ii);
	    if (elem._product.equals(product)) {
		if (success) {
		    elem._listener.succeeded(event);
		} else {
		    elem._listener.failed(event);
		}
		_listeners.removeElementAt(ii);
	    }
	}
    }
}

