//
// RFrame.java
//
//	Rhino Frame, which can be de-iconified.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;
import javax.swing.*;
import java.awt.event.*;
import java.awt.*;
import java.lang.reflect.*;
import com.sgi.sysadm.util.*;

/**
 * RFrame is a JFrame that de-iconifies itself if <tt>toFront</tt> is
 * called and it is iconified.  RFrame also calls
 * <tt>extraCleanup</tt> on every Component in its hierarchy that
 * implements the ExtraCleanup interface when <tt>RFrame.dispose</tt>
 * is called.  This gives Components a convenient hook for cleaning up
 * when they are no longer needed.   As the component tree is traversed
 * all parent - child relationships are broken to facilitate garbage
 * collection in light of certain Java bugs such as 4129207.  
 * <tt>removeNotify</tt> is not
 * suitable as a cleanup hook because when an RFrame is de-iconified
 * <tt>removeNotify</tt> and then <tt>addNotify</tt> will be called on
 * all Components in the hierarchy.
 * @see ExtraCleanup
 */
public class RFrame extends JFrame {

    // Whether we should place our own windows
    private final static boolean _placeWindows;

    // Where we put our last window
    private static int _lastWindowPlacement = 0;

    // How much to offset the x and y locations of successive windows
    private final static int _windowOffset = 50;

    // Avoid placing windows too close to top or left side by always
    // having at least this much space between the top and left.
    private final static int _bufferSize = 50;

    // The size (in pixels) of the screen
    private final static Dimension _screenSize;

    private boolean _iconic = false;
    private Rectangle _bounds = getBounds();

    private final static String CLASS_NAME = "RFrame";

    // Initialize the _screenSize
    static {
        // To work around Java bug 4102292, which results in windows 
        // always appearing at 0,0 on the screen, we'll place the
        // windows ourselves, if necessary.  Right now, 

        _placeWindows =
            !System.getProperties().getProperty("os.name").equals("Irix");
        if (_placeWindows) {
            Log.debug(CLASS_NAME, "We'll place our own windows");
            Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
            _screenSize = new Dimension(d.width/2,
                                        d.height/2);
        } else {
            Log.debug(CLASS_NAME, 
                      "We'll let the window manager place our windows");
            _screenSize = null;
        }
    }
    
    /**
     * Construct an RFrame with a title.
     *
     * @param title Title bar string.
     */
    public RFrame(String title) {
	super(title);
	addWindowListener(new WindowAdapter() {
	    public void windowIconified(WindowEvent event) {
		_iconic = true;
	    }
	    public void windowDeiconified(WindowEvent event) {
		_iconic = false;
	    }
	});
 	addComponentListener(new ComponentAdapter() {
	    public void componentMoved(ComponentEvent event) {
		Rectangle rect = getBounds();
		// In Windows, we get iconified by being moved to a
		// distant location.  This logic is attempting to
		// differentiate an iconification from a normal move.
		if (rect.x >= -rect.width) {
		    _bounds = rect;
		}
	    }
	    public void componentResized(ComponentEvent event) {
		_bounds = getBounds();
	    }
 	});

        if (_placeWindows) {
            Point p = new Point(
                _lastWindowPlacement % _screenSize.width + _bufferSize,
                _lastWindowPlacement % _screenSize.height + _bufferSize);
            setLocation(p);
            Log.debug(CLASS_NAME, "Setting location to: " + p);
            _lastWindowPlacement += _windowOffset;
        }
    }

    /**
     * Construct an RFrame.
     */
    public RFrame() {
	this("");
    }

    /**
     * Called when we are garbage-collected.  For debugging.
     *
     * @exception Throwable if there is an error during destruction.
     */
    protected void finalize() throws Throwable {
	Log.trace(CLASS_NAME, "Finalize: " + this);
	super.finalize();
    }

    /**
     * In addition to bringing the window to the front, de-iconify it
     * if it is iconified.
     */
    public void toFront() {
	if (!isVisible()) {
	    return;
	}

	try {
	    // The removeNotify/addNotify hack in the catch clause
	    // below works in Java 1.1, but not in Java 1.2.  So we
	    // use reflection to try to get the real method for
	    // unminimizing ourselves before resorting to the hack.
	    Class frameClass = getClass();
	    Method setStateMethod = frameClass.getMethod(
		"setState", new Class[] { Integer.TYPE });
	    Field normalField = frameClass.getField("NORMAL");
	    
	    setStateMethod.invoke(this, new Object[]
				  { normalField.get(this) });
	} catch (Exception ex) {
	    if (_iconic) {
		setVisible(false);
		/*
		 * This is a hack I got from java.sun.com.  The idea
		 * is that we destroy the window peer, getting rid of
		 * the old window, and then create a new one.  We
		 * have a real API for doing this in Java 1.2 (see
		 * above try clause).
		 * --rogerc
		 */
		removeNotify();
		addNotify();
		setBounds(_bounds);
		_iconic = false;
		setVisible(true);
	    }
	}
	super.toFront();
    }

    /**
     * Called when this RFrame is no longer to be used.  Calls the
     * <tt>extraCleanup</tt> method on every Component in our
     * hierarchy that implements the ExtraCleanup interface. 
     * As the component tree is traversed
     * all parent - child relationships are broken to facilitate garbage
     * collection in light of certain Java bugs such as 4129207.  
     * @see ExtraCleanup
     */
    public void dispose() {
	super.dispose();
	doExtraCleanup(this);
    }

    /**
     * Call the extraCleanup() method on every Component in our
     * hierarchy that implements the ExtraCleanup interface.  As we go
     * through the component tree, remove all parent - child
     * relationships to facilitate garbage collection in light of
     * certain Java bugs such as 4129207.
     *
     * @param comp Top of hierarchy to cleanup.
     */
    private void doExtraCleanup(Component comp) {
	try {
	    ((ExtraCleanup)comp).extraCleanup();
	} catch (ClassCastException ex) {
	}

	try {
	    Component kids[] = ((Container)comp).getComponents();
	    for (int ii = 0; ii < kids.length; ii++) {
		doExtraCleanup(kids[ii]);
	    }
	    ((Container)comp).removeAll();
	} catch (ClassCastException ex) {
	}
    }
}
