//
// RLabel.java
//
//      A subclass of JLabel that uses a standard look.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.util.*;
import javax.swing.*;
import java.awt.*;
import java.util.*;

/**
 * This class can be used wherever a JLabel would be used,
 * but when it is desireable to control the font and color with
 * resources.
 */
public class RLabel extends JLabel {
	
    
    /**
     * A resource <i>RLabel.font</i> is a Font resource that specifies the
     * default font to use for the RLabel if FONT can't be found.
     * @see #FONT
     */
    public static final String DEFAULT_FONT = "RLabel.font";
    
    /**
     * A resource <i>RLabel.color</i> is a Color resource that specifies the
     * default color to use for the RLabel if COLOR can't be found.
     * @see #COLOR
     */
    public static final String DEFAULT_COLOR  = "RLabel.color";
  
    /**
     * A resource <i>&lt;name&gt;.font</i> is a resource that specifies the
     * font to use for the RLabel, where &lt;name&gt; is the <tt>name</tt>
     * passed to the constructor.
     */
    public static final String FONT = ".font";
    
    /**
     * A resource <i>&lt;name&gt;.color</i> is a resource that specifies the
     * color to use for the RLabel, where &lt;name&gt; is the <tt>name</tt>
     * passed to the constructor.
     */
    public static final String COLOR  = ".color";

    /**
     * Construct a nameless RLabel.  DEFAULT_FONT and DEFAULT_COLOR
     * will be used.
     * 
     * @param rs The ResourceStack containing the resources to control
     *           this RLabel.
     */
    public RLabel(ResourceStack rs) {
	super();
	setup(rs, null);
    }
    
    /**
     * Construct a RLabel with a particular name.  The <tt>name</tt> will be
     * used to lookup the FONT and COLOR resources.
     *
     * @param rs The ResourceStack containing the resources to control
     *           this RLabel.
     * @param name The name of the RLabel.
     */ 
    public RLabel(ResourceStack rs,
		  String name) {
	super();
	setup(rs, name);
    }
    
    /** 
     * Construct a nameless RLabel with some initial text and a
     * specific alignment.
     * DEFAULT_FONT and DEFAULT_COLOR will be used.  The alignment
     * will be as specified.
     *
     * @param text The text to display in the RLabel
     * @param horizontalAlignment The alignment for the text.  See
     *                             JLabel for the alignment options.
     * @param rs The ResourceStack containing the resources to control
     *           this RLabel.
     * @see JLabel
     */
    public RLabel(String text,
		  int horizontalAlignment,
		  ResourceStack rs) {
	super(text, horizontalAlignment);
	setup(rs, null);
    }
    
    /** 
     * Construct a nameless RLabel with some initial text.
     * DEFAULT_FONT and DEFAULT_COLOR will be used. 
     *
     * @param text The text to display in the RLabel
     * @param rs The ResourceStack containing the resources to control
     *           this RLabel.
     */
    public RLabel(String text,
		  ResourceStack rs) {
	super(text);
	setup(rs, null);
    }
    
    /** 
     * Construct a RLabel with some initial text, a particular
     * name, and a particular alignment.  The <tt>name</tt> will be
     * used to lookup the FONT and COLOR resources.  The alignment
     * will be as specified.
     *
     * @param text The text to display in the RLabel
     * @param horizontalAlignment The alignment for the text.  See
     *                             JLabel for the alignment options.
     * @param rs The ResourceStack containing the resources to control
     *           this RLabel.  
     * @param name The name of the RLabel.
     *
     * @see JLabel
     */
    public RLabel(String text,
		  int horizontalAlignment,
		  ResourceStack rs,
		  String name) {
	super(text, horizontalAlignment);
	setup(rs, name);
    }
	
    /** 
     * Construct a RLabel with some initial text and a particular
     * name.  The <tt>name</tt> will be
     * used to lookup the FONT and COLOR resources. 
     *
     * @param text The text to display in the RLabel
     * @param rs The ResourceStack containing the resources to control
     *           this RLabel.  
     * @param name The name of the RLabel.
     *
     * @see JLabel
     */
    public RLabel(String text,
		  ResourceStack rs,
		  String name) {
	super(text);
	setup(rs, name);
    }
    private void setup(ResourceStack rs, String name) {
	if (name != null) {
	    setFont(rs.getFont(new String[] {name + FONT, DEFAULT_FONT}));
	    setForeground(rs.getColor(
		new String[] {name + COLOR, DEFAULT_COLOR}));
	} else {
	    setFont(rs.getFont(DEFAULT_FONT));
	    setForeground(rs.getColor(DEFAULT_COLOR));
	}
    }
}



	
