//
// RPanel.java
//
//	A subclass of JPanel that handles dynamic sizing.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;
import java.awt.*;
import java.util.*;
import javax.swing.*;
import com.sgi.sysadm.util.*;

/**
 * RPanel is a subclass of JPanel which handles children which are
 * dynamically sizable, meaning that their height depends on their
 * width and they implement the DynamicSize interface.  Normal
 * Components are handled as well.
 * <p>
 * Since RPanel itself implements DynamicSize, RPanels can be nested
 * inside one another and cooperate to implement multiple levels of
 * dynamic sizing.
 * <p>
 * In order to properly implement dynamic sizing, RPanel must be used
 * in conjunction with a LayoutManager that implements the
 * DynamicSizeLayoutManager interface.
 * @see DynamicSize
 * @see DynamicSizeLayoutManager
 */
public class RPanel extends JPanel implements DynamicSize {

    private Sizer _sizer;

    /**
     * Get the preferred size of this RPanel.  If
     * <tt>setPreferredWidth</tt> has been called, and if the
     * LayoutManager implements the DynamicSizeLayoutManager
     * interface, this will calculate the height based on the width
     * passed to <tt>setPreferredWidth</tt>.
     * 
     * @return Preferred size of this panel.
     */
    public Dimension getPreferredSize() {
	if (_sizer == null) {
	    _sizer = new DefaultSizer();
	}
	return _sizer.getPreferredSize(this);
    }

    /**
     * Get the preferred height of this RPanel given a width.  This
     * method will fail with a ClassCastException if this RPanel's
     * LayoutManager does not implement the DynamicSizeLayoutManager
     * interface.
     * 
     * @param width Proposed width
     * 
     * @return height we'd like to be if we're <tt>width</tt> wide.
     */
    public int getPreferredHeight(int width) {
	return ((DynamicSizeLayoutManager)getLayout())
	    .getPreferredHeight(this, width);
    }

    /**
     * Set the preferred width of this component.  Calling this method
     * causes RPanel to use <tt>getPreferredHeight</tt> in its
     * implementation of <tt>getPreferredSize</tt>.  This means that
     * <tt>getPreferredSize</tt> will throw a ClassCastException if
     * this RPanel's LayoutManager does not implement the
     * DynamicSizeLayoutManager interface.
     * 
     * @param width Preferred width for this panel.
     */
    public void setPreferredWidth(int width) {
	setSizer(new DynamicSizer(width));
    }

    /*
     * Sizer interface for implementing getPreferredSize() method.
     */
    /*package*/ interface Sizer {
	public Dimension getPreferredSize(RPanel panel);
    }

    /*
     * Hack.  I'd really like to be able to call
     * panel.super.getPreferredSize() from within an inner class.
     */
    private Dimension superGetPreferredSize() {
	return super.getPreferredSize();
    }

    /*
     * Sizer that makes RPanel act just like JPanel.
     */
    static class DefaultSizer implements Sizer {
	public Dimension getPreferredSize(RPanel panel) {
	    return panel.superGetPreferredSize();
	}
    }
    
    /*
     * Sizer that does dynamic sizing.
     */
    static class DynamicSizer implements Sizer {
	final int _width;
	DynamicSizer(int width) {
	    _width = width;
	}
	public Dimension getPreferredSize(RPanel panel) {
	    return new Dimension(_width, panel.getPreferredHeight(_width));
	}
    }

    /*
     * Sizer that does fixed sizing.
     */
    static class FixedSizer implements Sizer {
	final Dimension _size;
	FixedSizer(Dimension size) {
	    _size = size;
	}
	public Dimension getPreferredSize(RPanel panel) {
	    return new Dimension(_size);
	}
    }

    /*
     * Set the sizer to be used.
     */
    /*package*/ void setSizer(Sizer sizer) {
	_sizer = sizer;
    }

    /**
     * setupSizing looks up resources on the resource stack and sets
     * the size of this RPanel accordingly.
     * 
     * @param rs resource stack for resources.
     * @param dynamicSizeResource Boolean resource which determines
     * 				  whether we will do dynamic sizing.
     * @param widthResource Integer resource determines width in
     *                      points, which will be used in all cases.
     * @param heightResource If dynamicSizing is off, and the
     * 			     heightResource is present, heightResource
     * 			     determines fixed height of this RPanel.
     * 			     If heightResource is not present, golden
     *			     ratio is used.
     */
    public void setupSizing(ResourceStack rs,
				 String dynamicSizeResource,
				 String widthResource,
				 String heightResource) {
	int width = rs.getInt(widthResource);
	if (rs.getBoolean(dynamicSizeResource)) {
	    setSizer(new DynamicSizer(SysUtil.pixelsFromPoints(width)));
	    return;
	}
	Dimension size;
	try {
	    size = new Dimension(SysUtil.pixelsFromPoints(width),
				 rs.getPixels(heightResource));
	} catch (MissingResourceException ex) {
	    size = SysUtil.sizeFromWidth(width);
	}
	setSizer(new FixedSizer(size));
    }
}
