//
// RTextField.java
//
//      A subclass of JTextField that uses monospaced font.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.util.*;
import javax.swing.text.Document;
import javax.swing.*;
import sun.awt.motif.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.datatransfer.*;
import java.util.*;

/**
 * This class is a JTextField that uses a monospace font defined in a
 * properties file.  Using a monospace font in a text field assures
 * that the number of input characters displayed is equal to the
 * column width of the text field.
 * <P>
 * The font and margin resources are shared with RPasswordField and
 * ItemFinder.
 *
 * @see javax.swing.JTextField
 * @see com.sgi.sysadm.ui.RPasswordField
 * @see com.sgi.sysadm.ui.ItemFinder
 */
public class RTextField extends JTextField implements ExtraCleanup {
    private X11Selection _xselection;
    private Vector _focusListeners;

    /**
     * A resource <i>Field.font</i> defined in
     * <TT>com.sgi.sysadm.ui.SysadmUIP.properties</TT> that specifies
     * the font to be used.  It may be overridden in
     * {package}/packageP.properties, but please note that this
     * resource is shared with RPasswordField and ItemFinder.
     */
    public static final String FONT = "Field.font";

    /**
     * A resource <i>Field.topMargin</i> defined in
     * <TT>com.sgi.sysadm.ui.SysadmUIP.properties</TT> that specifies
     * the amount of space between the top of the tallest letter, and the
     * border of the text field.  It may be overridden in
     * {package}/packageP.properties, but please note that this
     * resource is shared with RPasswordField and ItemFinder.
     */
    public static final String TOP_MARGIN = "Field.topMargin";

    /**
     * A resource <i>Field.bottomMargin</i> defined in
     * <TT>com.sgi.sysadm.ui.SysadmUIP.properties</TT> that specifies
     * the amount of space between the bottom of a letter descender (ex.,
     * the tail of a "g"), and the border of the text field.  It may
     * be overridden in {package}/packageP.properties, but please note
     * that this resource is shared with RPasswordField and ItemFinder.
     */
    public static final String BOTTOM_MARGIN = "Field.bottomMargin";

    /**
     * A resource <i>Field.leftMargin</i> defined in
     * <TT>com.sgi.sysadm.ui.SysadmUIP.properties</TT> that specifies
     * the  amount of space between the leftmost letter and the border of
     * the text field.  It may be overridden in
     * {package}/packageP.properties, but please note that this
     * resource is shared with RPasswordField and ItemFinder.
     */
    public static final String LEFT_MARGIN = "Field.leftMargin";

    /**
     * A resource <i>Field.rightMargin</i> defined in
     * <TT>com.sgi.sysadm.ui.SysadmUIP.properties</TT> that specifies
     * the amount of space between the rightmost letter and the border of
     * the textfield.  It may be overridden in
     * {package}/packageP.properties, but please note that this
     * resource is shared with RPasswordField and ItemFinder.
     */
    public static final String RIGHT_MARGIN = "Field.rightMargin";

    private static Insets insets;
    private static Font font;

    static {
	// Static initializer
	ResourceStack rs = new ResourceStack();
	rs.pushBundle("com.sgi.sysadm.ui.SysadmUI" +
		       ResourceStack.BUNDLE_SUFFIX);
	font = rs.getFont(FONT);
	int left =  rs.getInt(LEFT_MARGIN);
	int right  = rs.getInt(RIGHT_MARGIN);
	int top  = rs.getInt(TOP_MARGIN);
	int bottom = rs.getInt(BOTTOM_MARGIN);
	insets = new Insets(top,left,bottom,right);
    }

    /*
     * This is what makes the font monospace, and gives the textfield
     * a skosh more room (to match the size of JComboBoxes).
     */
    private void setUpCommonUI() {
        setFont(font);
        setMargin(insets);
	setupXTransfers();
    }

    /**
     * Paste from the primary X selection when the user presses the
     * middle mouse button.  Unfortunately, the X11Selection methods
     * for going the other way (copying to the primary selection) are
     * not public, so we can't call them.
     */
    private void setupXTransfers() {
	try {
	    _xselection = new X11Selection("PRIMARY");
	} catch (Throwable err) {
	    Log.trace("RTextField", "No X11 selection");
	    return;
	}

	addMouseListener(new MouseAdapter() {
	    public void mousePressed(MouseEvent ev) {
		if ((ev.getModifiers() & MouseEvent.BUTTON2_MASK) != 0) {
		    try {
			String sel = (String)_xselection.getTransferData(
				DataFlavor.stringFlavor);
			int len = sel.length();
			StringBuffer buf = new StringBuffer(len);
			// Strip out newlines
			for (int ii = 0; ii < len; ii++) {
			    char ch = sel.charAt(ii);
			    if (ch != '\n' && ch != '\r') {
				buf.append(ch);
			    }
			}
			getDocument().insertString(getCaretPosition(),
						   buf.toString(), null);
		    } catch (Throwable ex) {
			// We catch everything, including null pointer
			// exceptions.
		    }
		}
	    }
	});
    }

    /**
     * Construct a new RTextField.
     */
    public RTextField() {
	super();
	setUpCommonUI();
    }

    /**
     * Construct a new RTextField initialized
     * with the specified <TT>text</TT>.
     *
     * @param text The text to be displayed, or null.
     */
    public RTextField(String text) {
	super(text);
	setUpCommonUI();
    }

    /**
     * Construct a new RTextField with width set to the specified
     * number of <TT>columns</TT>.
     *
     * @param columns The width of the TextField in columns.
     */
    public RTextField(int columns) {
	super(columns);
	setUpCommonUI();
    }

    /**
     * Construct a new RTextField with width set to the specified
     * number of <TT>columns</TT> and initialized with the specified
     * <TT>text</TT>.
     *
     * @param text The text to be displayed, or null.
     * @param columns The width of the TextField in columns.
     */
    public RTextField(String text, int columns) {
	super(text, columns);
	setUpCommonUI();
    }

    /**
     * Construct a new RTextField with width set to the specified
     * number of <TT>columns</TT>, initialized with the specified
     * <TT>text</TT> and using the specified text storage
     * model.
     *
     * @param text The text to be displayed, or null.
     * @param columns The width of the TextField in columns.
     * @param doc The text storage to use.
     */
    public RTextField(Document doc, String text, int columns) {
	super(doc, text, columns);
	setUpCommonUI();
    }

    /**
     * Add a FocusListener to our list.
     *
     * @param l A FocusListener interested in FocusEvents on this text
     *          field.
     *
     * @see java.awt.Component#addFocusListener
     */
    public void addFocusListener(FocusListener l) {
	if (_focusListeners == null ) {
	    _focusListeners = new Vector();
	}
	_focusListeners.addElement(l);
	super.addFocusListener(l);
    }

    /**
     * Remove focus listener from our list.
     *
     * @param l A FocusListener no longer interested in FocusEvents on
     *          this text field.
     *
     * @see java.awt.Component#removeFocusListener
     */
    public void removeFocusListener(FocusListener l) {
	_focusListeners.removeElement(l);
	super.removeFocusListener(l);
    }

    /**
     * Cleanup and remove any focus listeners.
     *
     * @see com.sgi.sysadm.ui.ExtraCleanup
     */
    public void extraCleanup() {
	if (_xselection != null) {
	    try {
		_xselection.dispose();
	    } catch (Throwable ex) {
	    }
	}
	for (int ii = 0; ii < _focusListeners.size(); ii++) {
	    super.removeFocusListener((FocusListener)
				      _focusListeners.elementAt(ii));
	}
	_focusListeners.removeAllElements();
    }
}
