//
// ResourceBasedIconRenderer.java
//
//	An icon renderer that renders using resources.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.category.*;
import javax.swing.*;
import java.util.*;
import java.awt.*;

/**
 * An icon renderer that renders using resources.  This class supports
 * both FTR icons and .gif icons
 *
 * @see FtrIcon
 */
public class ResourceBasedIconRenderer extends IconRenderer {
 
    /** 
     * A resource <i>&lt;name&gt;.iconModifiers</i> is a
     * string array that specifies which Attributes of the Item the
     * icon's appearance
     * will be based on.  The Attributes specified will be passed to
     * the <tt>set</tt> method of FtrIcon.
     *
     * @see FtrIcon#set
     */
    public static final String ICON_MODIFIERS = 
        ".iconModifiers";

    /** 
     * A resource <i>&lt;name&gt;.iconBasedOn</i> is a
     * string that specifies which Attribute the icon appearance
     * will be based on.
     */
    public static final String ICON_BASED_ON = 
        ".iconBasedOn";

    /**
     * A resource <i>&lt;name&gt;.icon</i> is a string that
     * specifies the icon to display if ICON_BASED_ON was not
     * specified, or if the value of ICON_BASED_ON's attribute is not
     * one of the values for which a particular icon has been
     * defined via the ICON property.  The value of this resource can
     * either be the package qualified class
     * name of an FtrIcon or a classpath relative pathname to an image
     * file.
     */
    public static final String DEFAULT_ICON = ".icon";
 
    /**
     * A resource <i>&lt;name&gt;.icon.&lt;Attribute's
     * value&gt;</i> is a string that gives the name of the icon to
     * display if &lt;Attribute's value&gt; matches
     * the value of the attribute specified in ICON_BASED_ON.  The
     * value of this resource can either be the package qualified class
     * name of an FtrIcon or a classpath relative pathname to an image
     * file.
     */
    public static final String ICON = ".icon.";    

    /**
     * A resource <i>&lt;name&gt;.blinkAttr</i> is a string array that gives
     * the names of Attributes in the Item.  If the value any of the
     * Attributes named in the array matches the corresponding string
     * in the BLINK_VALUES array, then then icon will blink.
     * <p>
     * For example, to blink if the Attribute "State" is "error" 
     * add the following resources:
     * <PRE>
     * com.sgi.mypackage.category.myCategory.blinkAttr0 = State
     * com.sgi.mypackage.category.myCategory.blinkValue0 = error
     * </PRE>
     * To blink if the Attribute "State" is "error" or
     * "unknown", add the following resources:     
     * <PRE>
     * com.sgi.mypackage.category.myCategory.blinkAttr0 = State
     * com.sgi.mypackage.category.myCategory.blinkAttr1 = State
     * com.sgi.mypackage.category.myCategory.blinkValue0 = error
     * com.sgi.mypackage.category.myCategory.blinkValue1 = unknown
     * </PRE>
     * To blink if Attribute "State" is "error" or Attribute
     * "Location" is "unknown", add the following resources:
     * <PRE>
     * com.sgi.mypackage.category.myCategory.blinkAttr0 = State
     * com.sgi.mypackage.category.myCategory.blinkAttr1 = Location
     * com.sgi.mypackage.category.myCategory.blinkValue0 = error
     * com.sgi.mypackage.category.myCategory.blinkValue1 = unknown
     * </PRE>
     */
    public static final String BLINK_ATTRS = ".blinkAttr";
    
    /**
     * A resource <i>&lt;name&gt;.blinkValue</i> is a string array that gives
     * the values of Attributes that an Item may have.  If the values
     * of any of the Attributes named in the BLINK_ATTRS array match
     * the corresponding strings in the BLINK_ATTRS array, then then
     * icon will blink. See BLINK_ATTRS for more information.
     *
     * @see BLINK_ATTRS
     */
    public static final String BLINK_VALUES = ".blinkValue";
   
    private String[] _modifiers;
    private String _defaultIconName;
    private String _basedOn;
    private boolean _blinking = false;
    private boolean _blank = false;
    private String[] _blinkAttrs;
    private String[] _blinkValues;
    private Blinker _blinker;
    private static final String[] EMPTY_STRING_ARRAY = new String[0];
    private static final String CLASS_NAME = "ResourceBasedIconRenderer";
    private Hashtable _blinkingItems = new Hashtable();
    private int _numListeners = 0;
    private boolean _notifyingBlinkers = false;

    /**
     * Construct a ResourceBasedIconRenderer for a particular
     * Category.
     * 
     * @param fullCategoryName The package qualified name of the
     *                         Category
     * @param rs The ResourceStack to use to look up the properties
     * @param hc The HostContext to use to get Categories.
     */
    public ResourceBasedIconRenderer(String fullCategoryName,
				     ResourceStack rs,
				     HostContext hc) {
	super(fullCategoryName, rs, hc);
	_basedOn = getResourceStack().getString(
	    fullCategoryName + ICON_BASED_ON, null);
	_modifiers = getResourceStack().getStringArray(
	    fullCategoryName + ICON_MODIFIERS, EMPTY_STRING_ARRAY);
	_defaultIconName = fullCategoryName + DEFAULT_ICON;

	_blinkAttrs = getResourceStack().getStringArray(
	    fullCategoryName + BLINK_ATTRS, EMPTY_STRING_ARRAY);
	_blinkValues = getResourceStack().getStringArray(
	    fullCategoryName + BLINK_VALUES, EMPTY_STRING_ARRAY);
	
	Log.assert((_blinkAttrs == EMPTY_STRING_ARRAY && 
		    _blinkValues == EMPTY_STRING_ARRAY) || 
		   (_blinkAttrs != EMPTY_STRING_ARRAY &&
		    _blinkValues != EMPTY_STRING_ARRAY),
		   _blinkAttrs == EMPTY_STRING_ARRAY ? 
		   fullCategoryName + BLINK_VALUES + " was set, but " +
		   fullCategoryName + BLINK_ATTRS +  " was not." :
		   fullCategoryName + BLINK_ATTRS +  " was set, but " +
		   fullCategoryName + BLINK_VALUES + " was not.");
	Log.assert(_blinkAttrs == EMPTY_STRING_ARRAY ||
		   _blinkAttrs.length == _blinkValues.length,
		   fullCategoryName + BLINK_VALUES + " array was not the " + 
		   " same length as the " + fullCategoryName + BLINK_ATTRS +  
		   " array.");
	_blinker = new Blinker() {
	    public void blinkOn() {
		_blank = false;
		Enumeration enum = _blinkingItems.elements();
		_notifyingBlinkers = true;
		while (enum.hasMoreElements()) {
		    notifyListeners((Item)enum.nextElement(), false);
		}
		_notifyingBlinkers = false;
	    };
	    public void blinkOff() {
		_blank = true;
		Enumeration enum = _blinkingItems.elements();
		_notifyingBlinkers = true;
		while (enum.hasMoreElements()) {
		    notifyListeners((Item)enum.nextElement(), false);
		}
		_notifyingBlinkers = false;
	    }
	};
    }

    public void addIconListener(String itemSelector, Object context,
				int iconWidth, int iconHeight,
				RenderedObjectListener listener) {
	_numListeners++;
	super.addIconListener(itemSelector, context, iconWidth,
			      iconHeight, listener);
     }

    public void removeIconListener(String itemSelector,
				   RenderedObjectListener listener) {
	_numListeners--;
	if (_numListeners == 0) {
	    BlinkThread.removeBlinker(_blinker);
	    _blinking = false;
	}
	super.removeIconListener(itemSelector, listener);
    }

    /**
     * Create an icon.  First try to create an FtrIcon; if that
     * doesn't work, try to create an ImageIcon.
     * 
     * @param item The item
     * @param width The width of the Icon to create, in pixels
     * @param height The height of the Icon to create, in pixels
     * @param context The context to use.  See GenericItemRenderer for
     *                 a description of the context.
     *
     * @return Icon corresponding to the item
     * @see GenericItemRenderer
     */
    protected Icon createIcon(Item item,
			      int width, 
			      int height,
			      Object context) {
	boolean blinkIcon = false;
	if (!_notifyingBlinkers) {
	    if (item != null) {
		for (int ii = 0; ii < _blinkAttrs.length; ii++) {
		    if (item.getString(_blinkAttrs[ii]).equals(_blinkValues[ii])) {
			blinkIcon = true;
			break;
		    }
		}
		String itemSelector = item.getSelector();
		if (blinkIcon) {
		    _blinkingItems.put(itemSelector, item);
		} else {
		    _blinkingItems.remove(itemSelector);
		}
	    }
	    
	    if (!_blinking && _blinkingItems.size() > 0) {
		BlinkThread.addBlinker(_blinker);
		_blinking = true;
	    } else if (_blinking && _blinkingItems.size() == 0) {
		BlinkThread.removeBlinker(_blinker);
		_blinking = false;
	    }
	} else {
	    blinkIcon = true;
	}
	
	if (blinkIcon && _blank) {
	    return new BlankIcon(width, height);
	}
	
	String iconName;
	if (_basedOn == null || item == null) {
	    iconName = _defaultIconName;
	} else {
	    iconName = getFullCategoryName() + ICON +
		item.getValueString(_basedOn);
	}
	Icon icon;
	try {
	    icon = getResourceStack().getIcon(
		new String[]{iconName, _defaultIconName});
	} catch (MissingResourceException mre) {
	    // no icon was found.
	    Log.debug(CLASS_NAME, "An icon with the name of " +
		      iconName + " was not found.  Will not set the icon.");
	    return null;
	}
	
	if (icon instanceof FtrIcon) {
	    ((FtrIcon)icon).setSize(width, height);
	    int num = _modifiers.length;
	    AttrBundle bundle = null;
	    if (num > 0 && item != null) {
		bundle = new AttrBundle();
		Attribute attr;
		for (int ii = 0; ii < num; ii++) {
		    attr = item.getAttr(_modifiers[ii]);
		    if (attr != null) {
			bundle.setAttr(attr);
		    } else {
			Log.warning(CLASS_NAME,
				    "iconModifier attribute " +
				    _modifiers[ii] + " was not found " +
				    "in the item.");
		    }
		}
	    }	   
	    if  (bundle != null) {
		((FtrIcon)icon).set(bundle);
	    }
	}
	return icon;
    }
}
