//
// ResourceBasedNameRenderer.java
//
//	A NameRenderer that renders using resources.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.util.*;
import com.sgi.sysadm.category.*;
import java.text.*;
import java.util.*;

/**
 * An NameRenderer that renders using resources.  This is used in the
 * infrastructure to create the names of the Items for ItemViews,
 * ItemTables, and ResultViews, and TreeViews.
 *
 */
public class ResourceBasedNameRenderer extends NameRenderer {

    /**
     * A resource <i>&lt;category name&gt;.nameAttr</i> is a
     * string that specifies which Attribute of the Item whose value
     * will be used as the name.
     */
    public static final String NAME =
        ".nameAttr";

    /**
     * A resource <i>&lt;category name&gt;.categoryName</i> is a
     * string that specifies the user visible name of the Category.
     */
    public static final String CATEGORY =
        ".categoryName";

    /**
     * A resource <i>&lt;category name&gt;.pluralCategoryName</i> is a
     * string that specifies the user visible name of the Category in
     * its plural form.
     */
    public static final String CATEGORY_PLURAL =
        ".pluralCategoryName";

    /**
     * A value of <i>ITEM_SELECTOR</i> for the resource <TT>NAME</TT>
     * means to use the selector of the item as the name of the Item.
     * This is the default.
     */
    private static final String ITEM_SELECTOR = "ITEM_SELECTOR";

    /**
     * A value of <i>CATEGORY_SELECTOR</i> for the resource <TT>CATEGORY</TT>
     * means to use the selector of the category as the name of the
     * category.  This is the default.
     */
    private static final String CATEGORY_SELECTOR = "CATEGORY_SELECTOR";

    /**
     * A resource <i>ItemAndCategoryFormat</i> is a string that gives the
     * Format String to use to combine the Category name and the Item
     * name. {0} will be replaced by the Item name, {1} by the
     * Category name.
     */
    public static final String ITEM_NAME_FORMAT = "ItemAndCategoryFormat";

    /**
     * A resource <i>ItemView.titleFormatString</i> is the string that
     * is used to create the ItemView title.  The name of the Item
     * will be substituted in {0}, the name of the Category in {1},
     * and the name of the host that the GUI is connected to in {2}
     */
    public static final String IV_TITLE_FORMAT =
        "ItemView.titleFormatString";

    /**
     * A resource <i>ItemTable.titleFormatString</i> is the string that
     * is used to create the ItemTable title.  The name of the Category
     * will be substituted in {0} and the name of the host that the
     * GUI is connected to in {1}
     */
    public static final String IT_TITLE_FORMAT =
        "ItemTable.titleFormatString";

    private static final String CLASS_NAME = "ResourceBasedNameRenderer";

    private String _itemName;
    private String _categoryName;
    private String _pluralCategoryName;
    private String _hostName;
    private String _ITFormat;
    private String _IVFormat;
    private String _itemAndCategoryFormat;

    /**
     * Create a ResourceBasedNameRenderer for a particular Category.
     * 
     * @param fullCategoryName The package qualified name of the
     *                         Category
     * @param rs The ResourceStack to use to look up the properties
     * @param hc The HostContext to use to get Categories.
     */
    public ResourceBasedNameRenderer(String fullCategoryName,
				     ResourceStack rs,
				     HostContext hc) {
	super(fullCategoryName, rs, hc);
	_itemName = getResourceStack().getString(fullCategoryName + NAME,
            ITEM_SELECTOR);
	_categoryName = getResourceStack().getString(
	    fullCategoryName + CATEGORY, CATEGORY_SELECTOR);
	_pluralCategoryName = getResourceStack().getString(
	    fullCategoryName + CATEGORY_PLURAL, fullCategoryName + "s");
	_hostName = hc.getHostName();
	_ITFormat = rs.getString(IT_TITLE_FORMAT);
	_IVFormat = rs.getString(IV_TITLE_FORMAT);
	_itemAndCategoryFormat = rs.getString(ITEM_NAME_FORMAT);
    }

    /**
     * Create the name of the Item.
     *
     * @param item The Item
     * @param context The context to use. See GenericItemRenderer for
     *                 a description of the context.
     *
     * @return String corresponding to the name of the Item.
     */
    protected String createName(Item item,
				Object context) {
	String name;
	String category;
	String pluralCategory;
	if (item == null) {
	    name = "";
	} else if (_itemName.equals(ITEM_SELECTOR)) {
	    name = item.getSelector();
	} else {
	    name = item.getValueString(_itemName);
	}
	if (_categoryName.equals(CATEGORY_SELECTOR) && item != null) {
	    category = item.getType();
	} else {
	    category = _categoryName;
	}
	if (_pluralCategoryName.equals(CATEGORY_SELECTOR) && item != null) {
	    pluralCategory = item.getType() + "s";
	} else {
	    pluralCategory = _pluralCategoryName;
	}

	if (context == ITEM_VIEW) {
	    return MessageFormat.format(
		_IVFormat,
		new String[] { name, category, _hostName }
		);
	} else if (context == ITEM_TABLE) {
	    return MessageFormat.format(
		_ITFormat,
		new String[] {pluralCategory, _hostName });
	} else if (context == RESULT_VIEW) {
	    return MessageFormat.format(
		_itemAndCategoryFormat,
		new String[] {name, category});
	} else if (context instanceof ItemNameRendererFormat) {
	    name = "";
	    if (item == null) {
		return name;
	    }

	    ItemNameRendererFormat format = (ItemNameRendererFormat)context;

	    if (format._attributes != null) {
		Object[] args = new Object[format._attributes.length];
		for (int ii = 0; ii < format._attributes.length; ++ii) {
		    args[ii] = item.getValueString(format._attributes[ii]);
		}
		name = MessageFormat.format(format._format, args);
	    } else {
		name = item.getValueString(format._format);
	    }
	    return name;
	} else {
	    return name;
	}
    }
}
