//
// ResultViewPanel.java
//
//	A class that displays the results of a Task
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.category.*;

/** 
 * A class that displays the results of a Task.  The ResultPanel
 * contains three sub-components.  The first, located at the top of
 * the panel, is a RichTextComponent that contains text describing the
 * results of the Task.  The text can be simple and just say that the
 * Task sucessfuly completed, or it can give detailed information
 * about the actions carried out by the Task.  The text may also tell
 * the user what steps logically follow after completing this Task.
 * <p>
 * The second section, located in the middle of the panel, contains a
 * Item icon and a link to launch an ItemView.  This section is
 * optional, and is only used when the Task has operated on a
 * particular Item.
 * <p>
 * The third section is a TaskShelf, which displays the Tasks that
 * are logical to perform given the current Item and its current
 * state.
 * <p>
 * There are 7 constructors which offer a wide variety of ways to
 * create a ResultView panel.  Here as some tips to choosing the
 * correct constructor:
 * <dl>
 * <dt>Does the Task operate on a particular Item, and is that Item
 * still in existance when the Task finishes?  
 *  
 * <dd>If so, use one of the
 * constructors that takes an <tt>itemSelector</tt> and
 * <tt>categoryName</tt>, or an <tt>itemTester</tt> and a
 * <tt>categoryName</tt>.  Choosing one of these constructors
 * will allow the ResultViewPanel to show a link to launch the
 * ItemView for the Item that was operated on.  It will also allow the
 * TaskShelf to show Tasks that are applicable to the Item in
 * question.  
 * The choice whether to use
 * the itemSelector or itemTester version of the constructor depends
 * on whether or not the Task knows the selector of the Item it
 * operated on.  If so, pass the selector to the constructor.  If not,
 * then pass an ItemTester that can positively identify the Item
 * <p>
 * <dt>Does the Task operate on a particular Category of Items, rather
 * than a particular Item, or does the Task delete an Item? 
 * <dd>
 * In either of these cases, there is no Item to display, but you
 * should still pass the name of the Category to the constructor so
 * that the Tasks relavent to the Category can be placed in the
 * TaskShelf.
 * <p>
 * <dt> Will the TaskRegistry provide all of the Tasks that should be
 * shown in the TaskShelf.
 * <dd>
 * If you use a constructor that takes a categoryName (or a
 * categoryName + itemSelector or categoryName + itemTester), then the
 * ResultViewPanel will use the TaskRegistry to fill the TaskShelf
 * with relavant Tasks.  If there are some Tasks that should be in the
 * ResultViewPanel that will not be returned by the TaskRegistry, then
 * you should use one of the constructors that accepts an array of
 * TaskLoaders.  The Tasks specified by the TaskLoaders will be placed
 * ahead of any Tasks that the TaskRegistry provides in the TaskShelf.
 * </dl>
 * @see RichTextComponent
 * @see ItemTester
 * @see TaskRegistry
 * @see TaskLoader
 */
public class ResultViewPanel extends JPanel {
   
    /** 
     * A resource <i>ResultViewPanel.textToLinkSpacing</i> is an
     * integer that gives the vertical space (in points) to put
     * between the text area and the ItemView link.
     */
    public static final String TEXT_TO_LINK_SPACING = 
        "ResultViewPanel.textToLinkSpacing";

    /** 
     * A resource <i>ResultViewPanel.textToShelfSpacing</i> is an
     * integer that gives the vertical space (in points) to put
     * between the text area and the TaskShelf.  Only used if there's
     * no link to launch an ItemView.
     */
    public static final String TEXT_TO_SHELF_SPACING = 
        "ResultViewPanel.textToShelfSpacing";
     
    /** 
     * A resource <i>ResultViewPanel.shelfToButtonSpacing</i> is an
     * integer that gives the vertical space (in points) to put
     * between the ItemView link and the TaskShelf.
     */
    public static final String LINK_TO_SHELF_SPACING = 
        "ResultViewPanel.linkToShelfSpacing";

    /** 
     * A resource <i>ResultViewPanel.iconToLinkSpacing</i> is an
     * integer that gives the horizontal space (in points) to put
     * between the icon and the link.
     */
    public static final String ICON_TO_LINK_SPACING = 
        "ResultViewPanel.iconToLinkSpacing";


    /**
     * A resource <i>ResultViewPanel.iconWidth</i> is an integer
     * that specifies the width (in points) of the icon.
     */
    public static final String ICON_WIDTH = 
        "ResultViewPanel.iconWidth";
    
    /**
     * A resource <i>ResultViewPanel.iconHeight</i> is an integer
     * that specifies the height (in points) of the icon.
     */
    public static final String ICON_HEIGHT = 
        "ResultViewPanel.iconHeight";
    

    /**
     * A resource <i>ResultViewPanel.verticalMargin</i> is the height
     * (in points) of the margin to use on the top and bottom of the
     * panel.
     */
    public static final String V_MARGIN = 
        "ResultViewPanel.verticalMargin";
     
    /**
     * A resource <i>ResultViewPanel.horizontalMargin</i> is the width
     * (in points) of the margin to use on the left and right of the
     * panel.
     */
    public static final String H_MARGIN = 
        "ResultViewPanel.horizontalMargin";

    /**
     * A resource <i>ResultViewPanel.pleaseWait</i> is the string that
     * is displayed in place of the Item's name until the name is
     * known.  The ResultViewPanel is often displayed before it has
     * retreived the Item that the Task has operated on from the
     * server.  Until the ItemArrives, the PLEASE_WAIT string will be
     * displayed as the Item's name.
     */
    public static final String PLEASE_WAIT =
        "ResultViewPanel.pleaseWait"; 

    /**
     * A system property that controls whether the TaskShelf is shown
     * in the ResultViewPanel.  If this property is defined,
     * then the TaskShelf will be shown.
     */
    public static final String SHOW_TASK_SHELF = 
        "ResultViewPanel.showTaskShelf";

    private final static Byte TEXT = new Byte((byte)0);
    private final static Byte SHELF = new Byte((byte)1);
    private final static Byte LINK = new Byte((byte)2);
    private final static Byte ICON = new Byte((byte)3);
    
    private int _iconWidth;
    private int _iconHeight;

    private UIContext _uic;
    private TaskShelfPanel _tsp;
    private TaskShelfController _tsc;
    private Vector _itemViewLaunchRequestListeners = new Vector();
    private Vector _taskLaunchRequestListeners = new Vector();
    private Category _category;
    private CategoryListener _catListener;
    private RichTextComponent _itemTitle;
    private JLabel _itemIcon;
    private ItemView _itemView;
    private int _currentLinkState = RichTextComponent.LINK_STATE_INITIAL;
    private static boolean _showTaskShelf;
    private static final String CLASS_NAME = "ResultViewPanel";

    static {
	if (SysUtil.getSystemProperty(SHOW_TASK_SHELF) != null) {
	    _showTaskShelf = true;
	    Log.warning(CLASS_NAME, SHOW_TASK_SHELF + " flag is set");
	} else {
	    _showTaskShelf = false;
	}
    }

    private ResultViewPanel(UIContext uic,
			    ResourceStack rs,
			    String message) {
	setLayout(new ResultViewPanelLayoutManager(rs));
	int v = rs.getPixels(V_MARGIN);
	int h = rs.getPixels(H_MARGIN);
	setBorder(BorderFactory.createEmptyBorder(v,h,v,h));
	_uic = uic;
	add(getRichText(uic,rs,message), TEXT );
	_iconWidth = rs.getPixels(ICON_WIDTH);
	_iconHeight = rs.getPixels(ICON_HEIGHT);
    }

    /** 
     * A constructor that contains a message and a list of Tasks that
     * might logically follow the Task that has finished.  This is
     * meant to be used when there is no Item or Category to give the
     * status of.
     * 
     * @param uic The UIContext to use for this ResultViewPanel.
     * @param rs The ResourceStack to use to control this
     *           ResultViewPanel.
     * @param loaders The list of TaskLoaders whose Tasks should be
     *                displayed in the TaskShelf.
     * @param message A localized string to display in the
     *                ResultViewPanel.  The string can contain any
     *                markup that RichText can understand.
     */
    public ResultViewPanel(UIContext uic, 
			   ResourceStack rs, 
			   TaskLoader[] loaders, 
			   String message) {
	this(uic,rs,message);
	if (loaders != null && _showTaskShelf) {
	    _tsp = new TaskShelfPanel(uic);
	    _tsp.setTaskList(loaders);
	    add(_tsp, SHELF);
	}
    }

    /**
     * A constructor that takes a TaskLoader[], a Category name, Item
     * selector, and  message.  The ResultView will display the message,
     * a list of Tasks that are logical for the Item in its current
     * state (as returned by TaskRegistry) with the Tasks in
     * <tt>loader</tt> prepended, and a link to launch an ItemView for the Item.
     * This is meant to be used when the Task operates on a particular Item.
     *
     * @param uic The UIContext to use for this ResultViewPanel.
     * @param rs The ResourceStack to use to control this
     *           ResultViewPanel.
     * @param loaders An array of Task loaders.  These Tasks will appear
     *                first in the TaskShelf.
     * @param categoryName The name of the Category that contains the
     *                     Item that the Task operated on.
     * @param itemSelector The selector of the Item that the Task
     *                     operated on.
     * @param message A localized string to display in the
     *                ResultViewPanel.  The string can contain any
     *                markup that RichText can understand.
     */
    public ResultViewPanel(final UIContext uic,
			   final ResourceStack rs,
			   TaskLoader[] loaders,
			   final String categoryName,
			   final String itemSelector,
			   final String message) {
	this(uic,rs,message);
	HostContext hc = uic.getHostContext();
	if (_showTaskShelf) {
	    setupRVP(hc, uic, loaders);
	    _tsc.setItem(hc.getCategory(
		ResourceStack.getClassName(categoryName)), 
			 itemSelector);
	}
	reserveSpaceForItemViewLink(rs);
	setupItemViewLink(uic, rs, categoryName, itemSelector, hc);
    }

    /**
     * A constructor that contains a Category name and message.  The
     * Result view will display the message, and a list of Tasks that are
     * logical for that category (as returned by TaskRegistry), with
     * the Tasks in <tt>loaders</tt> prepended.  
     * This is meant to be used when the Task does not have a specific
     * Item to show the status of, but has operated on a particular
     * Category (for example, a delete Task).
     *
     * @param uic The UIContext to use for this ResultViewPanel.
     * @param rs The ResourceStack to use to control this
     *           ResultViewPanel.
     * @param loaders An array of Task loaders.  These Tasks will appear
     *                first in the TaskShelf.
     * @param categoryName The name of the Category that contains the
     *                     Item that the Task operated on.
     * @param message A localized string to display in the
     *                ResultViewPanel.  The string can contain any
     *                markup that RichText can understand.
     */
    public ResultViewPanel(final UIContext uic,
			   final ResourceStack rs,
			   TaskLoader[] loaders,
			   final String categoryName,
			   final String message) {
	this(uic,rs,message);
	if (!_showTaskShelf) {
	    return;
	}
	
	HostContext hc = uic.getHostContext();
	_tsp = new TaskShelfPanel(uic);
	_tsc = new TaskShelfController(hc, loaders);
	
	_tsc.setTaskShelfPanel(_tsp);
	_tsc.setCategory(hc.getCategory(
	    ResourceStack.getClassName(categoryName)));
	add(_tsp, SHELF);
    }
    
    /**
     * A constructor that contains a Category name and message.  The
     * Result view will display the message, and a list of Tasks that are
     * logical for that Category (as returned by TaskRegistry).
     * This is meant to be used when the Task has operated on a
     * Category, but doesn't have a specific Item to display the state
     * of.
     *
     * @param uic The UIContext to use for this ResultViewPanel.
     * @param rs The ResourceStack to use to control this
     *           ResultViewPanel.
     * @param categoryName The name of the Category that contains the
     *                     Item that the Task operated on.
     * @param message A localized string to display in the
     *                ResultViewPanel.  The string can contain any
     *                markup that RichText can understand.
     */
    public ResultViewPanel(final UIContext uic,
			   final ResourceStack rs,
			   final String categoryName,
			   final String message) {
	this(uic, rs, (TaskLoader[])null, categoryName, message);
    }
    
     /**
     * A constructor that takes a TaskLoader[], a Category name, Item
     * name, and  message.  The ResultView will display the message,
     * a list of Tasks that are logical for the Item in its current
     * state (as returned by TaskRegistry), and a way to show the
     * ItemView for the Item.
     * This is meant to be used when the Task creates or modifies an
     * Item, and the Item's selector is known.
     *
     * @param uic The UIContext to use for this ResultViewPanel.
     * @param rs The ResourceStack to use to control this
     *           ResultViewPanel.
     * @param categoryName The name of the Category that contains the
     *                     Item that the Task operated on.
     * @param itemSelector The selector of the Item that the Task
     *                     operated on.
     * @param message A localized string to display in the
     *                ResultViewPanel.  The string can contain any
     *                markup that RichText can understand.
     */
    public ResultViewPanel(final UIContext uic,
			   final ResourceStack rs,
			   final String categoryName,
			   final String itemSelector,
			   final String message) {
	this(uic,rs, null, categoryName, itemSelector, message);
    }
    
    /**
     * A constructor that takes a TaskLoader[], a Category name, Item
     * tester, and  message.  The ResultView will display the message,
     * a list of Tasks that are logical for the Item found by the
     * tester in its current state (as returned by TaskRegistry), and
     * a way to show the ItemView for
     * the Item.
     * This is meant to be used when the Task creates or modifies an
     * Item, and the Item's selector in <b>not</b> known, but the
     * Item can be identified with an ItemTester.
     *
     * @param uic The UIContext to use for this ResultViewPanel.
     * @param rs The ResourceStack to use to control this
     *           ResultViewPanel.
     * @param categoryName The name of the Category that contains the
     *                     Item that the Task operated on.
     * @param itemTester An ItemTester that returns TEST_PASSED when
     *                   it identifies the Item that the Task operated on.
     *                  
     * @param message A localized string to display in the
     *                ResultViewPanel.  The string can contain any
     *                markup that RichText can understand.
     */
    public ResultViewPanel(final UIContext uic,
			   final ResourceStack rs,
			   final String categoryName,
			   final ItemTester itemTester,
			   final String message) {
	this(uic,rs, null, categoryName, itemTester, message);
    }
    
    /**
     * A constructor that takes a TaskLoader[], a Category name, Item
     * tester, and  message.  The ResultView will display the message,
     * a list of Tasks that are logical for the Item found by the
     * tester in its current state (as returned by TaskRegistry) with
     * the tasks from <tt>loaders</tt> prepended, and a way to show
     * the ItemView for the Item.
     * This is meant to be used when the Task creates or modifies an
     * Item and the Item's selector in <b>not</b> known, but the
     * Item can be identified with an ItemTester.
     *
     *
     * @param uic The UIContext to use for this ResultViewPanel.
     * @param rs The ResourceStack to use to control this
     *           ResultViewPanel.
     * @param loaders An array of Task loaders.  These Tasks will appear
     *                first in the TaskShelf.
     * @param categoryName The name of the Category that contains the
     *                     Item that the Task operated on.
     * @param itemTester An ItemTester that returns TEST_PASSED when
     *                   it identifies the Item that the Task operated on.
     *                  
     * @param message A localized string to display in the
     *                ResultViewPanel.  The string can contain any
     *                markup that RichText can understand.
     */
    public ResultViewPanel(final UIContext uic,
			   final ResourceStack rs,
			   TaskLoader[] loaders,
			   final String categoryName,
			   final ItemTester itemTester,
			   final String message) {
	this(uic,rs,message);
	final HostContext hc = uic.getHostContext();
	setupRVP(hc, uic, loaders);
	_category = hc.getCategory(ResourceStack.getClassName(categoryName));
	reserveSpaceForItemViewLink(rs);
	_category.addCategoryListener(_catListener = new CategoryAdapter() {
	    public void itemAdded(Item item) {
		if (itemTester.testItem(item).passed()) {
		    String itemSelector = item.getSelector();
		    _category.removeCategoryListener(this);
		    _catListener = null;
		    if (_showTaskShelf) {
			_tsc.setItem(hc.getCategory(
			    ResourceStack.getClassName(categoryName)), 
				     itemSelector);
		    }
		    setupItemViewLink(uic, rs, categoryName,
				      itemSelector, hc);
		}
	    }
	}, NotificationFilter.ALL_ITEMS);
    }
    

    /**
     * Initializes the panel
     */
     private void setupRVP(HostContext hc, 
			  UIContext uic, 
			  TaskLoader[] loaders) {
	if (!_showTaskShelf) {
	    return;
	}
	_tsp = new TaskShelfPanel(uic);
	_tsc = new TaskShelfController(hc, loaders);
	
	_tsc.setTaskShelfPanel(_tsp);
	add(_tsp, SHELF);
    }
    
    /**
     * Sets up the item view link area
     */
    private void reserveSpaceForItemViewLink(ResourceStack rs) {
	_itemIcon = new JLabel();
	_itemTitle = new RichTextComponent("ResultViewPanel.ItemTitle", rs);
	_itemTitle.setMargins(new Insets(0,0,0,0));
	_itemTitle.setText(rs.getString(PLEASE_WAIT));
	add(_itemTitle, LINK);
	add(_itemIcon, ICON);
    }
    
    /**
     * sets the item view link area.
     */
    private void setupItemViewLink(final UIContext uic,
				   final ResourceStack rs,
				   final String categoryName,
				   final String itemSelector,
				   final HostContext hc) {
	IconRenderer renderer = hc.getIconRenderer(categoryName);
	renderer.addIconListener(itemSelector, null, _iconWidth,
				 _iconHeight, new RenderedObjectListener() {
	    public void renderedObjectChanged(String itemSelector, 
					      Object object) {
		_itemIcon.setIcon((Icon)object);
	    }
	});
	
	NameRenderer nameRenderer = hc.getNameRenderer(categoryName);
	nameRenderer.addNameListener(itemSelector,
				     NameRenderer.RESULT_VIEW,
				     new RenderedObjectListener() {
	    public void renderedObjectChanged(String itemSelector,
					      Object object) {
		_itemTitle.setText("<A HREF=>" + 
				   RichTextComponent.quoteString(
				       (String)object) + 
				   "</A>");
	    }
	});
	
	_itemTitle.addLinkListener(new LinkListener() {
	    public void linkActivated(LinkEvent event) {
		handleItemViewLaunch(uic, hc, categoryName, itemSelector);
	    }
	});
	_itemIcon.addMouseListener(new MouseAdapter() {
	    public void mouseClicked(MouseEvent e) {
		_itemTitle.setLinkState(
		    0, RichTextComponent.LINK_STATE_VISITED);
		_currentLinkState =
		    RichTextComponent.LINK_STATE_VISITED;
		_itemTitle.repaint();
		SwingUtilities.invokeLater(new Runnable() {
		    public void run() {
			handleItemViewLaunch(
			    uic, hc, categoryName, itemSelector);
		    }
		});
	    }
	    public void mousePressed(MouseEvent e) {
		_itemTitle.setLinkState(
		    0, RichTextComponent.LINK_STATE_ACTIVE);
		_itemTitle.repaint();
	} 
	    public void mouseExited(MouseEvent e) {
		_itemTitle.setLinkState(
		    0, _currentLinkState);
		_itemTitle.repaint();
	    } 
	    
	});
    }

    /**
     * Handles an ItemView launch, either from the link or the icon.
     */
    private void handleItemViewLaunch(UIContext uic, HostContext hc,
				      String categoryName, String itemSelector) {
	ItemViewLaunchRequestEvent newEvent= new
	    ItemViewLaunchRequestEvent(
		ResultViewPanel.this,
		categoryName,
		itemSelector);
	int size = _itemViewLaunchRequestListeners.size();

	if (size == 0) {
	    ItemViewFrame.launchItemViewFrame(newEvent, uic);
	    return;
	}
	for (int ii = 0; ii < size; ii ++) {
	    ((ItemViewLaunchRequestListener)
	     _itemViewLaunchRequestListeners.elementAt(ii)).
		itemViewLaunchRequested(newEvent);
	}
    }
    
    /**
     * Adds an ItemViewLaunchRequestListener to the list of listeners
     * that will be notified if an ItemView launch is requested.  If
     * there are no listeners added, then the default behavior is to
     * launch the ItemView in a new ItemViewFrame.
     *
     * @param listener The ItemViewLaunchRequestListener to add.
     */
    public void addItemViewLaunchRequestListener(
	ItemViewLaunchRequestListener listener) {
	_itemViewLaunchRequestListeners.addElement(listener);
    }
    
    /**
     * Adds a TaskLaunchRequestListener to the list of listeners
     * that will be notified if an Task launch is requested. If
     * there are no listeners added, then the default behavior is to
     * launch the Task in a new TaskFrame.
     *
     * @param listener The TaskLaunchRequestListener to add
     */
    public void addTaskLaunchRequestListener(
	TaskLaunchRequestListener listener) {
	_tsp.addTaskLaunchRequestListener(listener);
    }
    
    /**
     * Destroys this ResultViewPanel for garbage collection purposes.
     */
    public void destroy() {
	if (_category != null && _catListener != null) {
	    _category.removeCategoryListener(_catListener);
	    _catListener = null;
	}
	if (_tsc != null) {
	    _tsc.destroy();
	    _tsc = null;
	}
    }

    /**
     * A utility function to create the RichTextComponent from a
     * string
     */
    private static RichTextComponent getRichText(final UIContext uic, 
						 final ResourceStack rs, 
						 final String message){
	RichTextArea rta = new RichTextArea("ResultViewPanel",rs);
	rta.setText(message);
	rta.setMargins(new Insets(0,0,0,0));
	return rta;
    }

    private class ResultViewPanelLayoutManager implements LayoutManager2 {

	private int _textToLinkSpacing;
	private int _linkToShelfSpacing;
	private int _iconToLinkSpacing;
	private int _textToShelfSpacing;
	private Component _text;
	private Component _shelf;
	private Component _link;
	private Component _icon;

	/**
	 * The default constructor for ResultViewPanelLayoutManager 
	 */                 
	public ResultViewPanelLayoutManager(ResourceStack rs) {
	    _textToLinkSpacing = rs.getPixels(TEXT_TO_LINK_SPACING);
	    _linkToShelfSpacing = rs.getPixels(LINK_TO_SHELF_SPACING);
	    _iconToLinkSpacing = rs.getPixels(ICON_TO_LINK_SPACING);
	    _textToShelfSpacing = rs.getPixels(TEXT_TO_SHELF_SPACING);
	}

	/**
	 * Notification that a new component is being added to the
	 * container. This method is not currently used.  Use
	 * add(Component, Object) instead.
	 *
	 * @param name A name associated with the component.  Not
	 *             currently used
	 * @param comp The component to add to the layout 
	 */                 
	public void addLayoutComponent(String name, Component comp) {
	    Log.fatal("Use add(Component, Object) instead");
	}
 
	/*
	 * Called by all of the get*Size methods to determine the correct
	 * size for the container.
	 */
	private Dimension getSize(Container parent) {
	    Insets insets = parent.getInsets();
	    Dimension compSize;
	    int height = 0;
	    int width = 0;
	    Dimension size;
	    if (_text != null) {
		size = _text.getPreferredSize();
		height += size.height;
		width = Math.max(width,size.width);
	    }
	    if (_text != null && _link != null) {
		height += _textToLinkSpacing;
	    }

	    if (_link != null) {
		size = _link.getPreferredSize();
		height += Math.max(size.height, _iconHeight);
		width = Math.max(width,
				 size.width + 
				 _iconWidth + 
				 _iconToLinkSpacing);
	    }
	   
	    if (_link != null && _shelf != null) {
		height += _linkToShelfSpacing;
	    }
	    if (_link == null && _shelf != null && _text != null) {
		height += _textToShelfSpacing;
	    }
	    if (_shelf != null) {
		size = _shelf.getPreferredSize();
		height += size.height;
		width = Math.max(width,size.width);
	    }
	    width += insets.left + insets.right;
	    height += insets.top + insets.bottom;
	    return new Dimension(width,height);
	}

	private Dimension getComponentSize(Component c, int width) {
	    if (c instanceof DynamicSize) {
		return new Dimension(
		    width,((DynamicSize)c).getPreferredHeight(width));
	    } else {
		return c.getPreferredSize();
	    }
	}

	/**
	 * Does the layout.
	 *
	 * @param parent The container containing the objects to lay out.
	 */             
	public void layoutContainer(Container parent) {
	    int parentWidth = parent.getSize().width;
	    // Don't bother doing the layout if the parent doesn't have a
	    // valid width yet.
	    if (parentWidth <= 0) {
		return;
	    }
	    Insets insets = parent.getInsets();	
	    parentWidth -= insets.left + insets.right;
	    int currentY = insets.top;
	    Dimension size;
	    if (_text != null) {
		size = getComponentSize(_text, parentWidth);
		_text.setBounds(insets.right, currentY, 
				size.width, size.height);
		currentY += size.height;
	    }
	    if (_text != null && _link != null) {
		currentY += _textToLinkSpacing;
	    }
	    if (_link != null) {
		size = getComponentSize(_link, parentWidth);
		int taller =  Math.max( _iconHeight, size.height);
		int midline  = taller /2;
		_icon.setBounds(insets.left, 
				currentY + midline - _iconHeight / 2,
				_iconWidth, _iconHeight);
		int linkLeft = insets.left + _iconWidth + _iconToLinkSpacing;
		
		_link.setBounds(linkLeft,
				currentY + midline - size.height /2,
				parentWidth - linkLeft,
				size.height);
		currentY += taller;
	    }
	    if (_link != null && _shelf != null) {
		currentY += _linkToShelfSpacing;
	    }
	    if (_link == null) {
		currentY += _textToShelfSpacing;
	    }
	    if (_shelf != null) {
		_shelf.setBounds(insets.right, 
				 currentY, 
				 parentWidth, 
				 parent.getSize().height - currentY);
	    }	   
	}

	/** 
	 * Returns the minumum size of this component.
	 * @see java.awt.Component#getMaximumSize()
	 * @see java.awt.LayoutManager#minimumLayoutSize
	 */
	public Dimension minimumLayoutSize(Container parent) {
	    return getSize(parent);
	}

	/** 
	 * Returns the preferred size of this component.
	 * @see java.awt.Component#getPreferredSize()
	 * @see java.awt.LayoutManager#preferredLayoutSize
	 */
	public Dimension preferredLayoutSize(Container parent) {
	    return getSize(parent);
	}

	/** 
	 * Returns the maximum size of this component.
	 * @see java.awt.Component#getMaximumSize()
	 * @see java.awt.Component#getPreferredSize()
	 * @see java.awt.LayoutManager2#maximumLayoutSize
	 */
	public Dimension maximumLayoutSize(Container target) {
	    return getSize(target);
	}
    
	/**
	 * Removes the specified component from the layout.
	 * @param comp the component to be removed
	 */
	public void removeLayoutComponent(Component comp) {
	    if (comp == _link) {
		_link = null;
	    } else if (comp == _text) {
		_text = null;
	    } else if (comp == _shelf) {
		_shelf = null;
	    } else if (comp == _icon) {
		_icon = null;
	    } else {
		Log.fatal("Tried to remove a component that wasn't " +
			 "in the layout:\n" + comp);
	    }
	}
    
	/**
	 * Adds the specified component to the layout, using the specified
	 * constraint object.
	 * @param comp the component to be added
	 * @param constraints  where/how the component is added to the layout.
	 * @see java.awt.LayoutManager2#addLayoutComponent
	 */
	public void addLayoutComponent(Component comp, Object constraints) {
	    Log.assert(comp != null, "can't add a null component.");
	    if (constraints instanceof Byte) {
		if (((Byte)constraints).equals(LINK)) {
		    _link = comp;
		} else if (((Byte)constraints).equals(TEXT)) {
		    _text = comp;
		} else if (((Byte)constraints).equals(SHELF)) {
		    _shelf = comp;
		} else if (((Byte)constraints).equals(ICON)) {
		    _icon = comp;
		} else {
		    Log.fatal("The constraints must be one of: " +
			     ICON + ", " + LINK + ", " + 
			     TEXT + ", or " + SHELF);
		}
	    }
	}
    
	/**
	 * Returns the alignment along the x axis.  This specifies how
	 * the component would like to be aligned relative to other 
	 * components.  The value should be a number between 0 and 1
	 * where 0 represents alignment along the origin, 1 is aligned
	 * the furthest away from the origin, 0.5 is centered, etc.
	 */
	public float getLayoutAlignmentX(Container target) {
	    return 0.5F;
	}
    
    
	/**
	 * Returns the alignment along the y axis.  This specifies how
	 * the component would like to be aligned relative to other 
	 * components.  The value should be a number between 0 and 1
	 * where 0 represents alignment along the origin, 1 is aligned
	 * the furthest away from the origin, 0.5 is centered, etc.
	 */
	public float getLayoutAlignmentY(Container target) {
	    return 0.5F;
	}
    

	/**
	 * Invalidates the layout, indicating that if the layout manager
	 * has cached information it should be discarded.
	 */
	public void invalidateLayout(Container target) {
	}    
    }
}
    
