//
// Task.java
//
//	Abstract class for creating sysadm Tasks.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import com.sgi.sysadm.category.Category;
import com.sgi.sysadm.category.CategoryListener;
import com.sgi.sysadm.category.Item;
import com.sgi.sysadm.category.ItemTester;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.text.*;
import java.util.*;
import java.io.*;

/**
 * Task is an abstract class for creating sysadm Task user
 * interfaces.  For more information on writing a Task, see the
 * <A HREF="../tutorials/HowToWriteATask.html">How to Write a Task</A>
 * tutorial.
 * <P>
 * Task provides services such as initializing TaskData, checking privileges,
 * and managing a control panel.
 * <P>
 * Task interacts with its subclasses as follows:
 * <UL>
 * <LI>
 *   Task.Task() initializes itself and sets up public and private TaskData
 *   attributes as defined in {taskname}P.properties.  Task also sets the
 *   default preferred interface (Form or Guide) from the property
 *   Task.PREFERRED_UI (see Task.setPreferredUI() below).</P>
 * <LI>
 *   Task.setTaskDataAttr() is called by clients (such as RunTask) to
 *   set their own values for public TaskData.  TaskData usually
 *   corresponds to data that a user can input to the Task.  For
 *   example, a user name and modem name could be TaskData to the
 *   Create PPP Connection Task.  TaskData can also be used
 *   to control the behavior of a Task.  For example, the Create PPP
 *   Connection Task might define a boolean TaskData attribute that controls
 *   whether or not advanced options are displayed.</P>
 * <LI>
 *   subclass.setOperands() allows clients to set Task operands.  Operands are
 *   the objects that a Task will operate on.  For example, the Create PPP
 *   Connection task would not have any operands, but a user account
 *   Item selector name (or list of user account Item selectors) would
 *   be an operand to the Delete PPP Connection.
 *   <P>
 *   Because operands may be passed to Tasks by a class with no
 *   specific knowledge about the Task (e.g. TaskShelfPanel), no ordering of
 *   operands should be assumed or required by the Task.  The number
 *   and type of operands should be checked by subclass.setOperands(),
 *   but any verification that requires access to the server should be
 *   postponed until subclass.verifyPrereqsBeforeCheckPrivs() or
 *   subclass.verifyPrereqsAfterCheckPrivs() is called because
 *   setOperands() is synchronous.
 *   <P>
 *   Tasks that accept operands must set the property
 *   Task.OPERAND_TYPE_ACCEPTED to the Item type that the Task
 *   accepts, and override Task.setOperands() to check and store any
 *   operand(s) passed.  If the Task will accept any operand type, set
 *   the value of this property to Task.ALL_OPERAND_TYPES.  Tasks that
 *   do not accept operands need not define the
 *   Task.OPERAND_TYPE_ACCEPTED property nor override setOperands();
 *   the base class will post an error if a Task client attempts to
 *   pass in operands.</P>
 * <LI>
 *   Task.setPreferredUI() allows clients to determine whether the Form
 *   or Guide is the preferred user interface.  If a Task has both types of
 *   interfaces, this preference will be used to decide which interface to
 *   use when the Task is first displayed.  Calling this method overrides
 *   the property <A HREF="#PREFERRED_UI">Task.PREFERRED_UI</A>.</P>
 * <LI>
 *   Task.verifyPrereqs() is called by Task clients (such as RunTask) after
 *   they have set TaskData attributes and Task operands.  verifyPrereqs()
 *   first calls verifyPrereqsBeforeCheckPrivs(), then it attempts to obtain
 *   privileges, and finally it calls verifyPrereqsAfterCheckPrivs().  A
 *   failure at any of these three stages aborts the operation and the client
 *   is notified of the failure via a ResultListener.
 *   <P>
 *   Task implements trivial versions of verifyPrereqsBeforeCheckPrivs() and
 *   verifyPrereqsAfterCheckPrivs() that simply call
 *   ResultListener.succeeded(). The subclass should override these
 *   methods as needed.  If verification in either of these methods
 *   consists of multiple asynchronous calls that must be performed in
 *   sequence (also known as chaining), the subclass may use the
 *   version of
 *   <A HREF="com.sgi.sysadm.ui.TaskContext.html">TaskContext</A>.dataOK()
 *   that takes a Vector of TaskDataVerifiers to simplify the chaining logic. 
 *   <P>
 *   NOTE: the reason verification is split into BeforeCheckPrivs() and
 *         AfterCheckPrivs() calls is for convenience to the user.  The idea
 *         is to notify the user of a failure as early as possible in order to
 *         prevent unnecessary input of root password for privileges.  For
 *         example, it would be very annoying for the user to find out that
 *         the server state is invalid for a particular task after entering
 *         the root password when the task could have checked the server and
 *         posted an error before attempting to obtain privileges.</P>
 * <LI>
 *   When Task is notified via the AncestorListener interface that it is
 *   about to become visible for the first time, Task calls
 *   subclass.registerInterfaces().  subclass.registerInterfaces() should
 *   create a Form and/or Guide, and register these interfaces by calling
 *   Task.setForm() and Task.setGuide() respectively.  Task will then display
 *   the preferred interface if both exist, or the one registered interface
 *   otherwise.</P>
 * <LI> 
 *   Task displays a TaskControlPanel to allow the user to drive
 *   the Task interface.  TaskControlListener events are passed along to
 *   the Form and Guide as appropriate.</P>
 * <LI>
 *   Task.getTaskDataAttr() is called by clients to get the value of a 
 *   public TaskData attribute.  For example, the client may want to
 *   get the value of something entered by the User and pass it along to
 *   another Task.</P>
 * <LI>
 *   When the user presses the "OK" button, indicating that the task should
 *   be performed, Task first calls
 *   TaskContext.allDataOK(TaskDataVerifier.MUST_BE_SET) to verify that all
 *   of the TaskData attributes are valid and consistent.  After successful
 *   verification of TaskData, Task then calls subclass.ok() so that the
 *   subclass can run the privileged command(s).  The privileged
 *   command is run via a java.lang.Process object that is either under the
 *   control of the sysadm infrastructure or the subclass, as desired. Here
 *   are six of the <!-- excessive number of --> ways the subclass can run
 *   and interact with the
 *   privileged command(s):</P>
 * <OL>
 * <LI>runPriv(String privcmdName) displays a busy dialog, passes all TaskData
 *     to the named privileged command, and calls taskSucceeded() or
 *     taskFailed().</P>
 * <LI>runPriv(String privcmdName, TaskData taskData) displays a busy dialog,
 *     passes the specified TaskData to the named privileged command, and
 *     calls taskSucceeded() or taskFailed().  An OutputStream is
 *     returned so that the subclass can pass private data (such as
 *     passwords) or control data to the privileged command Process via
 *     standard input.</P>
 * <LI>runPriv(String privcmdName, TaskData taskData, ResultListener listener)
 *     passes the specified TaskData to the named privileged command and calls
 *     listener.succeeded() if the privcmd succeeds or
 *     listener.failed() if the privcmd fails.  An OutputStream is
 *     returned so that the subclass can pass private data (such as
 *     passwords) or control data to the privileged command Process via
 *     standard input.  The subclass is responsible for posting a busy
 *     dialog, and calling taskSucceeded() or taskFailed().</P>
 * <LI>runPriv(String privcmdName, TaskData taskData, ProcessListener listener)
 *     passes the specified TaskData to the named privileged command.  It
 *     adds the ProcessListener to the privileged command Process so
 *     that the subclass gets notified of ProcessEvents.  An OutputStream is
 *     returned so that the subclass can pass private data (such as
 *     passwords) or control data to the privileged command Process via
 *     standard input. When using this version of runPriv(), the
 *     subclass is responsible for posting a busy dialog, if any, and
 *     responding to ProcessEvents.  In particular, the subclass must
 *     call taskSucceeded() or taskFailed() when the privileged
 *     command Process(es) exit(s).</P>
 * <LI>runPriv(String privcmdName, TaskData taskData, Category category,
 *     ItemTester itemTester, int eventType) is like runPriv(String, TaskData),
 *     except that if the privcmd succeeds and the given ItemTester is not
 *     null, runPriv() waits to call taskSucceeded() until the specified event
 *     type has occurred in the Category on an Item matching the ItemTester.
 *     (If this does not happen before <i>Task.eventTimeout</i> seconds, it
 *     calls taskFailed().)
 *     <P>
 *     You might want to do this when the Task is launched from a metatask, and
 *     subsequent steps cannot be performed until after the changes caused by
 *     the Task's privcmd have been propagated back to the client. In general,
 *     it's probably better to <em>not</em> use this feature unless absolutely
 *     necessary, as it means the user spends longer staring at an hourglass.
 *     </P>
 * <LI>The subclass can create its own Process directly (e.g. by using
 *     TaskContext.getHostContext().getPrivBroker().runPriv()) and have
 *     full flexibility over interaction with the privileged
 *     command(s).  In this case the subclass is responsible for
 *     posting any busy dialogs and for calling taskSucceeded() or
 *     taskFailed().  If you need to wait on an event, you should be able to
 *     use waitForEvent() for this; see the notes above on waiting for
 *     events.</P> 
 * </OL>
 *    If the Task is successful, TaskDoneListeners may wish to display
 *    result information.  These clients will call getResultViewPanel() to 
 *    request a ResultViewPanel that they can display as a dialog or
 *    inside some part of their UI.  getResultViewPanel will call
 *    Subclass.createResultViewPanel after first checking that
 *    Task.succeeded() was called.
 * </UL>
 * Specific tasks must have an associated properties file that resides
 * in the same package as <I>{taskname}.class</I> named
 * <I>{taskname}P.properties</I>.  The package containing the task may
 * also define a properties file named <I>PackageP.properties</I> to provide
 * fallback values for all tasks in the package.
 *
 * @see java.lang.Process
 * @see com.sgi.sysadm.manager.RunTask
 * @see com.sgi.sysadm.ui.Form
 * @see com.sgi.sysadm.ui.Guide
 * @see com.sgi.sysadm.ui.TaskContext
 * @see com.sgi.sysadm.ui.taskData.TaskDataVerifier
 */
public abstract class Task extends JPanel {
    private TaskContext _taskContext;
    private ResourceStack _rs;
    private TaskData _taskData;
    private JPanel _taskUIPanel;
    private Form _form;
    private Guide _guide;
    private TaskControlPanel _controlPanel;
    private Hashtable _publicDataKeys;
    private int _preferredUI;
    private Vector _doneListeners = new Vector();
    private static final int NO_MODE = 0;
    private static final int FORM_MODE = 1;
    private static final int GUIDE_MODE = 2;
    private int _mode = NO_MODE;
    private static final String _class =
        ResourceStack.getClassName(Task.class);
    private PrivBroker _privBroker;
    private boolean _taskInProgress = false;
    private static boolean _noRunPrivFlag;
    private static boolean _noPrereqsFlag;

    /**
     * _taskSucceeded is set to true by Task.taskSucceeded() when runPriv()
     * receives an exit code of zero from the privileged command it runs.
     * getResultViewPanel() checks the value of _taskSucceeded
     * to determine if the task was successful and a result panel should
     * be created.
     */ 
    private boolean _taskSucceeded = false;

    /**
     * The property set <I>Task.privList&lt;n&gt;</I> describes the set of
     * privileges needed to perform a task.  Each privilege must be given its
     * own property name, with <I>Task.privList</I> as the base.  For example,
     * a task that has two privileges would describe them in the
     * <I>{taskname}P.properties</I> file as follows:
     * <PRE>
     *       Task.privList0 = firstPrivilege
     *       Task.privList1 = secondPrivilege
     * </PRE>
     */
    public static final String PRIV_LIST = "Task.privList";

    /**
     * The property set <I>Task.publicData&lt;n&gt;</I> describes the set of
     * TaskData attributes that can be set by clients of the task.  The TaskData
     * attributes are identified by their key.  Any TaskData attribute not
     * included in this set will be inaccessible to task clients.  Each public
     * data attribute must be given its own property name, with
     * <I>Task.publicData</I> as the base.  For example, the Create PPP
     * Connection Task would describe its public data attributes "loginName"
     * and "modemName" as follows:
     * <PRE>
     *       Task.publicData0 = loginName
     *       Task.publicData1 = modemName
     * </PRE>
     */
    public static final String PUBLIC_DATA = "Task.publicData";
    
    /**
     * The property <I>Task.shortName</I> is a String containing the short name
     * for the task.  This property is intended for the initial title of a
     * frame containing the task, the title of dialogs posted by the task, and
     * the task name in lists of tasks.  For example, "Create PPP
     * Connection" would be an appropriate short name.
     */
    public static final String SHORT_NAME = "Task.shortName";
    
    /**
     * The property <I>Task.longName</I> is a String describing the purpose
     * of the task. This property is intended to be the default Form page
     * title, so should only be a single line.  For example, "Create a
     * New PPP Connection" would be an appropriate long name.
     */
    public static final String LONG_NAME = "Task.longName";
    
    /**
     * <I>Task.itemTester</I> is a String containing the
     * <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH
     * relative</A> name of a class that determines whether the task
     * is relevant to a particular Item.  TaskLoader.getItemTester()
     * uses this property to determine what ItemTester class to load
     * and return.
     *
     * @see com.sgi.sysadm.ui.TaskLoader 
     */
    public static final String ITEM_TESTER = "Task.itemTester";
    
    /**
     * <I>Task.taskShelfIcon</I> is a String containing the
     * <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH
     * relative</A> name of the <A HREF="glossary.html#IconImageFile">icon 
     * image file</A> of an icon to display along with the short name
     * of the task in a TaskShelf.
     */
    public static final String TASK_SHELF_ICON = "Task.taskShelfIcon";
    
    /**
     * The property <I>Task.preferredUI</I> is a string that determines which
     * user interface, Form or Guide, will be displayed at startup if a task
     * has both interfaces.  Valid values are <I>form</I> and <I>guide</I>.
     * The default value is defined in com.sgi.sysadm.ui.TaskContextP.properties.
     * This property may be overridden by clients of Task by calling
     * Task.setPreferredUI().
     */
    public static final String PREFERRED_UI = "Task.preferredUI";
    
    /**
     * The property <I>Task.widthInPoints</I> is an integer that sets
     * the width of the task interface (both Form and Guide), in
     * <A HREF="glossary.html#Point">points</A>.  If the
     * corresponding property specifying the height of the task interface
     * is missing, the height calculated from the width using the
     * <A HREF="glossary.html#GoldenRatio">golden ratio</A>.
     * <P>
     * The default value of this property is defined in
     * com.sgi.sysadm.ui.TaskContextP.properties.  Overriding this property
     * in {package}.PackageP.properties or {package}.{taskname}P.properties
     * is discouraged because of the resizing that will have to occur in
     * containers that are used to display multiple tasks in sequence.
     */
    public static final String WIDTH_IN_POINTS = "Task.widthInPoints";

    /**
     * The property <I>Task.heightInPoints</I> is an integer that sets
     * the height of the task interface (both Form and Guide), in
     * <A HREF="glossary.html#Point">points</A>.
     */
    public static final String HEIGHT_IN_POINTS = "Task.heightInPoints";
    
    /**
     * String intended as an argument to Task.setPreferredUI() to set the
     * Form interface as the preferred interface.
     */ 
    public static final String FORM_UI = "form";
    
    /**
     * String intended as an argument to Task.setPreferredUI() to set the
     * Guide interface as the preferred interface.
     */
    public static final String GUIDE_UI = "guide";
    
    /**
     * The property <I>Task.topInset</I> is an integer that defines the inset,
     * in points,  between the top of the task container and the task interface.
     * <P>
     * The default value for this property is provided in
     * com.sgi.sysadm.ui.TaskContextP.properties and it may be overridden in
     * either {package}.PackageP.properties or {package}.{taskname}P.properties.
     */
    public static final String TOP_INSET = "Task.topInset";
    
    /**
     * The property <I>Task.leftInset</I> is an integer that defines the inset,
     * in points,  between the left side of the task container and the task
     * interface.
     * <P>
     * The default value for this property is provided in
     * com.sgi.sysadm.ui.TaskContextP.properties and it may be overridden in
     * either {package}.PackageP.properties or {package}.{taskname}P.properties.
     */
    public static final String LEFT_INSET = "Task.leftInset";
    
    /**
     * The property <I>Task.bottomInset</I> is an integer that defines the inset,
     * in points, between the bottom of the task container and the task
     * interface.
     * <P>
     * The default value for this property is provided in
     * com.sgi.sysadm.ui.TaskContextP.properties and it may be overridden in
     * either {package}.PackageP.properties or {package}.{taskname}P.properties.
     */
    public static final String BOTTOM_INSET = "Task.bottomInset";
    
    /**
     * The property <I>Task.rightInset</I> is an integer that defines
     * the inset, in points, between the right side of the task
     * container and the task interface.
     * <P>
     * The default value for this property is provided in
     * com.sgi.sysadm.ui.TaskContextP.properties and it may be overridden in
     * either {package}.PackageP.properties or
     * {package}.{taskname}P.properties.
     */
    public static final String RIGHT_INSET = "Task.rightInset";
    
    /**
     * The property <i>Task.titleFormat</i> is a format string that includes
     * two arguments: {0} for the task name and {1} for the server name.  The
     * format string is used to format the Task window title string.
     * The title string is used as-is when the Form interface is
     * displayed and is incorporated into the Guide.TITLE_FORMAT
     * string when the Guide interface is displayed.
     * <P>
     * Task subclasses that wish to add dynamic information to the
     * Task window title (as opposed to rearranging or reordering the
     * format string) may call TaskContext.setTaskTitle() anytime
     * after the Task constructor has completed.
     * <P>
     * A default value for this property is provided in TaskContextP.properties
     * and it may be overridden in either {package}.PackageP.properties or
     * {package}.{taskname}P.properties.
     */
    public static final String TITLE_FORMAT = "Task.titleFormat";
    
    /** 
     * The property set <I>Task.ProductAttributes&lt;n&gt;</I> is a
     * string array that specifies the products from which to request
     * <A HREF="glossary.html#ProductAttributes">product attributes</A>.  
     * Any product attributes set will be placed into the TaskData
     * before setOperands() or verifyPrereqs() are called.
     */
    public static final String PRODUCT_ATTRIBUTES = "Task.ProductAttributes";

    /**
     * The property set <I>Task.runOnceAttributes&lt;n&gt;</I> is a
     * String array that specifies the TaskData attributes that should
     * be used for controlling run once behavior of a task.  After
     * <A HREF="glossary.html#ProductAttributes">product attributes</A> 
     * have been set, the launch code in TaskFrame checks
     * to see if a TaskFrame instance for a Task of the same class
     * already exists with identical TaskData values for the keys in
     * <B>RUN_ONCE_ATTRIBUTES</B>.  If such an instance already
     * exists, it is used instead of creating a new one.
     */
    public static final String RUN_ONCE_ATTRIBUTES = "Task.runOnceAttributes";

    /**
     * The property <I>Task.operandTypeAccepted</I> is a String that
     * specifies what type of operand the Task accepts.  It is
     * used by the infrastructure to determine whether or not to pass
     * operands.  For example, if an ItemView passes its Item to a
     * TaskShelfPanel, the TaskShelfPanel will only pass that item to
     * tasks that accept operands of that item type.
     */
    public static final String OPERAND_TYPE_ACCEPTED =
	"Task.operandTypeAccepted";

    /**
     * The String <I>Task.allOperandTypes</I> is the value a Task
     * should use for the property <I>Task.operandTypeAccepted</I> if
     * the Task will accept an operand of any type.
     */
    public static final String ALL_OPERAND_TYPES =
	"Task.allOperandTypes";

    /**
     * The property <I>Task.keywords</I> is a String containing a
     * whitespace separated list of words the user may use when
     * searching for this task.
     * <P>
     * There is no default value provided for this property.
     */
    public static final String KEYWORDS = "Task.keywords";

    /**
     * The property <I>Task.helpKey</I> is a String containing the
     * help tag within HELP_BOOK of the help text for this task.
     */
    public static final String HELP_KEY = "Task.helpKey";

    /**
     * The property set <I>Task.sharedProperties&lt;n&gt;</I>
     * describes the set of properties files that should be pushed
     * onto the Task ResourceStack, in order, before the
     * {taskname}P.properties file.  These shared properties files are
     * useful for properties that will be shared by a group of Tasks
     * in a particular product.  Properties that are shared by all of
     * the Tasks in a product should use the PackageP.properties file,
     * which will be pushed onto the ResourceStack automatically.
     */
    public static final String SHARED_PROPERTIES = "Task.sharedProperties";

    
    /**
     * A system property that controls whether runPriv actually makes a
     * call to the server.  If this property is defined,
     * Task.runPriv() will always call taskSucceeded() without
     * attempting to contact the server.
     */
    public static final String DEMO_NO_RUNPRIV = "Demo.noRunPriv";
    

    /**
     * A system property that controls whether Task verifies prereqs
     * before showing the UI.  If set, no prereqs will be checked.
     * This can be used to see a UI that would not normally be visible
     * due to prereqs failing.  <b>Note:</B> Verifing prereqs often
     * has side effects in the Task (e.g. setting TaskData), so the
     * Task may not work as expected.
     */
    public static final String DEMO_NO_PREREQS = "Demo.noPrereqs";

    /**
     * The property <I>Task.showCancelButton</I> is a boolean that
     * controls whether the Cancel button is shown on the Task.  It
     * should be "true" or "false".
     */
    public static final String SHOW_CANCEL_BUTTON =
        "Task.showCancelButton";

    /**
     * If a Task will wait for an event on an Item after its privcmd
     * succeeds before returning the success to the user, the property
     * <I>Task.eventTimeout</I> determines how many seconds it will wait for
     * the event before giving up.
     */
    public static final String EVENT_TIMEOUT = 
        "Task.eventTimeout";

    /**
     * If a Task will wait for an event on an Item after its privcmd
     * succeeds before returning the success to the user, the property
     * <I>Task.eventTimeoutReason</I> is the message which will be displayed
     * if the event fails to arrive before the Task times out.  If your task
     * waits for an event, you may want to provide a less-generic message for
     * this property.
     */
    public static final String EVENT_TIMEOUT_REASON = 
        "Task.eventTimeoutReason";

    /**
     * The flag passed to runPriv() and waitForEvent() to indicate that they
     * shouldn't wait for an event.
     */
    protected static final int EVENT_TYPE_NONE = 0;

    /**
     * The flag passed to runPriv() and waitForEvent() to indicate that they
     * should wait for an add event on the desired Item.
     */
    protected static final int EVENT_TYPE_ADD = 1;

    /**
     * The flag passed to runPriv() and waitForEvent() to indicate that they
     * should wait for a modify event on the desired Item.
     */
    protected static final int EVENT_TYPE_CHANGE = 2;

    /**
     * The flag passed to runPriv() and waitForEvent() to indicate that they
     * should wait for a remove event on the desired Item.
     */
    protected static final int EVENT_TYPE_REMOVE = 4;

    static {
	if (SysUtil.getSystemProperty(DEMO_NO_RUNPRIV) != null) {
	    _noRunPrivFlag = true;
	    Log.warning(_class, DEMO_NO_RUNPRIV + " flag is set");
	} else {
	    _noRunPrivFlag = false;
	}
	if (SysUtil.getSystemProperty(DEMO_NO_PREREQS) != null) {
	    _noPrereqsFlag = true;
	    Log.warning(_class, DEMO_NO_PREREQS + " flag is set");
	} else {
	    _noPrereqsFlag = false;
	}
    }
    
   /**
     * Constructor.
     *
     * @param taskContext The context for this task.
     */
    public Task(TaskContext taskContext) {
	_taskContext = taskContext;
	_rs = _taskContext.getResourceStack();
	_taskData = _taskContext.getTaskData();
	
	try {
	    setPreferredUI(_rs.getString(Task.PREFERRED_UI));
	} catch (MissingResourceException exception) {
	    // Task.PREFERRED_UI is optional
	}
	
	addAncestorListener(new AncestorListener() {
	    public void ancestorAdded(AncestorEvent event) {
		if (_taskUIPanel == null) {
		    setupInterface();
		    removeAncestorListener(this);
		}
	    }
	    
	    public void ancestorMoved(AncestorEvent event) {
	    }
	    
	    public void ancestorRemoved(AncestorEvent event) {
	    }
	});
    }
    
    /**
     * Get the UIContext for this task.  Package visibility only.
     *
     * @return UIContext for this task.
     */
    /* package */ UIContext getUIContext() {
	return _taskContext;
    }
    
    /**
     * Register interest in task completion.
     *
     * @param listener Listener interested in task completion.
     */
    public void addTaskDoneListener(TaskDoneListener listener) {
	_doneListeners.addElement(listener);
    }
    
    /**
     * Unregister interest in task completion.
     *
     * @param listener Listener no longer interested in task completion.
     */
    public void removeTaskDoneListener(TaskDoneListener listener) {
	_doneListeners.removeElement(listener);
    }
    
    /**
     * Notify TaskDoneListeners that the task has completed successfully or
     * has been cancelled.
     *
     * @param result A TaskResult describing whether the Task succeeded or
     *               was cancelled.
     */    
    public void taskDone(TaskResult result) {
	Enumeration enum = _doneListeners.elements();
	while (enum.hasMoreElements()) {
	    TaskDoneListener listener = (TaskDoneListener)enum.nextElement();
	    listener.taskDone(result);
	}
    }

    /**
     * Notify the Task base class that the Task operation succeeded
     * after a call to subclass.ok().  The base class will clear the
     * busy cursor set when OK was pressed and call taskDone().
     *
     * @param event A ResultEvent that contains privcmd output text
     *              as the result and privcmd error text as the
     *              reason.  May be null.
     */
    protected void taskSucceeded(ResultEvent event) {
	_taskSucceeded = true;
	TaskResult result =
	    new TaskResult(Task.this, TaskResult.SUCCEEDED);
	if (event != null) {
	    result.setOutputText((String)event.getResult());
	    result.setErrorText(event.getReason());
	}
	taskDone(result);
	_taskContext.notBusy();
	_taskInProgress = false;
    }


    /**
     * Notify the Task base class that the Task operation failed.  The base
     * class will post an error if <TT>event</TT> is non-null,
     * and then clear the busy cursor set when the OK button was
     * pressed.  The Task UI will remain open (Task.taskDone() will
     * not be called) to allow the User to make a change and try again.
     *
     * @param event A ResultEvent that contains privcmd output text as
     * 		    the result and privcmd error text as the reason.
     * 		    That text will be displayed in an error dialog.
     * 		    Pass null if no error dialog should be displayed.
     */
    protected void taskFailed(ResultEvent event) {
	_taskSucceeded = false;
	if (event != null) {
	    _taskContext.postError(
		MessageFormat.format(
		    _rs.getString("Task.Error.taskFailed"),
		    new Object[] { (String)event.getResult(),
				       event.getReason() }),
		new ActionListener() {
		    public void actionPerformed(ActionEvent event) {
			_taskContext.notBusy();
			_taskInProgress = false;
		    }
	    });
	} else {
	    _taskContext.notBusy();
	    _taskInProgress = false;
	}
    }
    
    //
    // The following methods must be implemented by subclasses of Task
    //
    
    /**
     * Called by the Task client to pass operands to the task.  The
     * subclass must override this method if the task accepts
     * operands, as determined by the property Task.OPERAND_TYPE_ACCEPTED.
     * Any verification that requires privileges or contact with the server
     * must be postponed until verifyPrereqsBeforeCheckPrivs() or
     * verifyPrereqsAfterCheckPrivs() is called because this is a
     * synchronous method.
     * <P>
     * If Task.OPERAND_TYPE_ACCEPTED is not defined, then the subclass
     * need not override this method and the base class implementation will
     * confirm that <TT>operands</TT> is null or empty.  If the Task
     * client is attempting to pass one or more operands,
     * TaskInitFailedException will be thrown with the generic error
     * message Task.Error.noOperandsAccepted.
     *
     * @param operands A (possibly null or empty) vector of task operands.
     *
     * @exception com.sgi.sysadm.ui.TaskInitFailedException Thrown with error
     *            code INVALID_OPERANDS if the type or number of operands
     *            is invalid.
     */
    public void setOperands(Vector operands) throws TaskInitFailedException {
	try {
	    _rs.getString(OPERAND_TYPE_ACCEPTED);
	    Log.fatal("The property Task.OPERAND_TYPE_ACCEPTED is defined " +
		     "but the Task subclass has not overridden the " +
		     "setOperands() method.");
	} catch (MissingResourceException exception) {
	    // This Task does not accept operands
	}
	
	if (operands != null && operands.size() > 0) {
	    throw new
		TaskInitFailedException(
		    _rs.getString("Task.Error.noOperandsAccepted"),
		    TaskInitFailedException.INVALID_OPERANDS);
	}
    }
    
    /**
     * Verify the TaskData attributes, Task operands, and state of the
     * system being administered to make sure it is reasonable to 
     * go ahead with this Task.
     * <P>
     * Task.verifyPrereqs() is called by Task clients (such as RunTask) after
     * they have set TaskData attributes and Task operands.  verifyPrereqs()
     * first calls verifyPrereqsBeforeCheckPrivs(), then it attempts to obtain
     * privileges, and finally it calls verifyPrereqsAfterCheckPrivs().  A
     * failure at any of these three stages aborts the operation and
     * <TT>clientListener</TT>.failed is called.
     * <P>
     * Task implements trivial versions of verifyPrereqsBeforeCheckPrivs() and
     * verifyPrereqsAfterCheckPrivs() that simply call
     * ResultListener.succeeded().  Subclasses should override these methods
     * as needed.  If verification consists of multiple asynchronous calls that
     * must be chained together, the subclass may use the version of
     * TaskContext.dataOK() that takes a Vector of TaskDataVerifiers to
     * simplify the chaining logic.
     *
     * @param clientListener A ResultListener to notify of successful or
     *                       failed prereq verification.
     */
    public void verifyPrereqs(ResultListener clientListener) {
        if (_noPrereqsFlag) {
            Log.warning(_class, "Skipping all prereq checking because " +
                        "Demo.noPrereqs was set");
            clientListener.succeeded(new ResultEvent(this));
            return;
        }
	Vector prereqList = new Vector();
	prereqList.addElement(new TaskDataVerifier() {
	    public void dataOK(int browseFlag, Object context,
			       ResultListener listener) {
		verifyPrereqsBeforeCheckPrivs(listener);
	    }
	});

	prereqList.addElement(new TaskDataVerifier() {
	    public void dataOK(int browseFlag, Object context,
			       ResultListener listener) {
		checkPrivs(listener);
	    }
	});

	prereqList.addElement(new TaskDataVerifier() {
	    public void dataOK(int browseFlag, Object context,
			       ResultListener listener) {
		verifyPrereqsAfterCheckPrivs(listener);
	    }
	});

	_taskContext.dataOK(prereqList, TaskDataVerifier.MAY_BE_EMPTY,
			    null, clientListener);
    }

    /**
     * A trivial verification that always calls listener.succeeded().
     * Subclasses should override this method to do whatever prerequitiste
     * verification is possible before privileges have been obtained.  For
     * example, the subclass may be able to determine that the server is in an
     * invalid state without privileges, but must wait until privileges are
     * obtained to confirm the existence of an Item.
     *
     * @param listener A Result Listener to notify of a successful or failed
     *                 prereq verification.
     */
    protected void verifyPrereqsBeforeCheckPrivs(ResultListener listener) {
	listener.succeeded(new ResultEvent(this));
    }

    /**
     * A trivial verification that always calls listener.succeeded().
     * Subclasses should override this method to do prerequisite verification
     * that requires privileges.
     *
     * @param listener A Result Listener to notify of a successful or failed
     *                 prereq verification.
     */
    protected void verifyPrereqsAfterCheckPrivs(ResultListener listener) {
	listener.succeeded(new ResultEvent(this));
    }
    
    /**
      * registerInterfaces() is called by the base class to request
      * registration of the Task user interface.  registerUI() should create a
      * Form and/or Guide, and must register these interfaces by calling
      * Task.setForm() and Task.setGuide() respectively.
      */
    public abstract void registerInterfaces();
    
    /**
     * After the User presses the OK button, the base class will
     * verify TaskData and then call subclass.ok() to initiate
     * performance of the task operation. The subclass must call one of
     * taskSucceeded() or taskFailed() when processing of the OK
     * request has been completed (some versions of runPriv() do this
     * automatically).   Otherwise input to the Task will
     * remain blocked and the cursor will remain busy.
     */
    public abstract void ok();
    
    /**
     * Called by clients of the task when the task has been completed
     * successfully.  This method checks to make sure that
     * Task.taskSucceeded() has been called and then calls
     * Subclass.createResultViewPanel to obtain a ResultViewPanel
     * containing task-specific information about the result of the
     * task.  If Task.taskSucceeded() has not been called, this method
     * will call Log.fatal().
     */
    public final ResultViewPanel getResultViewPanel() {
	Log.assert(_taskSucceeded, "getResultViewPanel() was called " +
		   "without a previous call to taskSucceeded()");
	return createResultViewPanel();
    }
    
    /**
     * Called by the base class to obtain a ResultViewPanel from the
     * subclass.
     * <P>
     * If it's really not possible to provide a coherent message about
     * the results of your Task, this method may return null.
     */
    protected abstract ResultViewPanel createResultViewPanel();

    /**
     * Register interest in changes to the Task title.
     *
     * @param listener Object interested in changes to the Task title.
     */
    public void addTitleListener(TitleListener listener) {
	_taskContext.addTitleListener(listener);
    }
    
    /**
     * Unregister interest in changes to the Task title.
     *
     * @param listener Object no longer interested in changes to the Task title.
     */
    public void removeTitleListener(TitleListener listener) {
	_taskContext.removeTitleListener(listener);
    }

    /**
     * Get the current title of the Task
     *
     * @return The Task's title
     */
    public String getTaskTitle() {
        return _taskContext.getTitleString();
    }
    
    
    /**
     * Set the value of a specific public data item by key and value.
     *
     * @param key Name of public TaskData attribute to set.
     * @param value New value of the public TaskData attribute.
     *
     * @exception com.sgi.sysadm.ui.TaskInitFailedException Thrown with error
     *            code INVALID_DATA_ATTR if <TT>key</TT> is not a valid
     *            public data attribute of this task.
     */
    public void setTaskDataAttr(String key, String value) 
	    throws TaskInitFailedException{
	if (_publicDataKeys == null) {
	    getPublicDataKeys();
	}
	String type = (String)_publicDataKeys.get(key);
	
	if (type == null) {
	    throw new TaskInitFailedException(
		MessageFormat.format(
		    _rs.getString("Task.Error.invalidDataAttr"),
		    new Object[] { key }),
		TaskInitFailedException.INVALID_DATA_ATTR);
	}	
	
	_taskData.setAttr(new Attribute(key, type, value));
    }

    /**
     * Set the value of a specific public data attribute.
     *
     * @param attr Attribute to replace.
     *
     * @exception com.sgi.sysadm.ui.TaskInitFailedException Thrown with error
     *            code INVALID_DATA_ATTR if <TT>attr</TT> is not a valid
     *            public data attribute of this task.
     */
    public void setTaskDataAttr(Attribute attr) 
	    throws TaskInitFailedException{
	setTaskDataAttr(attr.getKey(), (String)attr.getValue());
    }
    
    /**
     * Get a specific public TaskData attribute.
     *
     * @param key Name of the public TaskData attribute to get.
     * @return An Attribute, if "key" describes a public TaskData attribute,
     *         null otherwise.
     */
    public Attribute getTaskDataAttr(String key) {
	if (_publicDataKeys == null) {
	    getPublicDataKeys();
	}
	
	if (!_publicDataKeys.containsKey(key)) {
	    return null;
	}
	
	return _taskData.getAttr(key);
    }
    
    /**
     * Get the TaskContext associated with this Task.
     * 
     * @return TaskContext associated with this Task.
     */
    /*package*/ TaskContext getTaskContext() {
	return _taskContext;
    }

    /**
     * Set the preferred user interface. If a Task has both Form and Guide
     * interfaces, this preference will be used to decide which interface to
     * use when the Task is first displayed.  Calling this method overrides
     * the property Task.PREFERRED_UI.
     *
     * @param preferredUI Task.FORM_UI if Form is the preferred interface,
     *                    Task.GUIDE_UI if Guide is the preferred interface.
     */
    public void setPreferredUI(String preferredUI) {
	if (preferredUI.equalsIgnoreCase(FORM_UI)) {
	    _preferredUI = FORM_MODE;
	} else if (preferredUI.equalsIgnoreCase(GUIDE_UI)) {
	    _preferredUI = GUIDE_MODE;
	} else {
	    Log.fatal("Property " + PREFERRED_UI + " is set to '" +
		     preferredUI + "' but should be one of '" +
		     GUIDE_UI + "' or '" + FORM_UI + "'.");
	}
    }
    
    /**
     * checkPrivs() is a service provided by the base class for
     * checking and obtaining the privileges needed to perform the Task.
     *
     * @param listener Gets notified of the result of the check.
     */
    protected void checkPrivs(final ResultListener listener) {
	String[] privList = null;
	try {
	    privList = _rs.getStringArray(PRIV_LIST);
	} catch (MissingResourceException exception) {
	    // There are no privileges for this task.  Print an info
	    // message and succeed.
	    Log.info(_class, "Task has no privilege list");
	    listener.succeeded(new ResultEvent(this));
	    return;
	}
	checkPrivsOnServer(privList, listener);
    }
    
    private void checkPrivsOnServer(String[] privList,
				    final ResultListener listener) {
	
	_taskContext.getPrivs(privList, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		listener.succeeded(event);
		if (_privBroker == null) {
		    _privBroker = _taskContext.getHostContext()
			.getPrivBroker();
		}	    
	    }
	    
	    public void failed(ResultEvent event) {
		ResultEvent resEvent = new ResultEvent(this);
		resEvent.setReason(event.getReason());
		listener.failed(resEvent);
	    }
	});
    }
    
    /**
     * Register a Form interface to use for this Task.  An
     * assertion will fail
     * if this method is called more than once per Task instance.
     *
     * @param form Form interface to register for this Task.
     */
    protected void setForm(Form form) {
	Log.assert(_form == null, "Duplicate call to Task.setForm()");
	
	_form = form;
	_taskUIPanel.add(_form, FORM_UI);
    }
    
    /**
     * Register a Guide interface to use for this Task.  An
     * assertion will fail
     * if this method is called more than once per Task instance.
     *
     * @param guide Guide interface to register for this Task.
     */
    protected void setGuide(Guide guide) {
	Log.assert(_guide == null, "Duplicate call to Task.setGuide()");
	
	_guide = guide;
	_taskUIPanel.add(_guide, GUIDE_UI);
    }

    // A little something to avoid tripicate code.  Pass the args to the
    // priv command in whatever form they come in this class, and break
    // out the differences right when calling the privBroker.
    private final class RunPrivArgs {
	public TaskData _taskData;
	public String _argString;
	public String _argVector[];

	public RunPrivArgs (TaskData taskData) {
	    _taskData = taskData;
	    _argString = null;
	    _argVector = null;
	}

	public RunPrivArgs (String argString) {
	    _taskData = null;
	    _argString = argString;
	    _argVector = null;
	}

	public RunPrivArgs (String argVector[]) {
	    _taskData = null;
	    _argString = null;
	    _argVector = argVector;
	}
    }

    /**
     * Subclasses should call this version of runPriv() if the task can
     * be performed by running a single privcmd, the privcmd does not accept
     * any private data (such as passwords) via stdin, it is reasonable to 
     * send all TaskData attributes to the privcmd, and task failure can be
     * handled by displaying the privcmd output and error text in a standard
     * error dialog.
     * <P>
     * runPriv() will post a busy dialog while the privileged command
     * Process  is running.  When the privileged command exits,
     * runPriv() will call taskSucceeded() if the exit code is zero,
     * or it will call taskFailed().
     *
     * @param privcmdName Name of the privileged command to run.
     */
    protected void runPriv(String privcmdName) {
	OutputStream stream = runPriv(privcmdName, new RunPrivArgs(_taskData),
				      (Category)null, (ItemTester)null,
				      EVENT_TYPE_NONE);
	try {
	    stream.close();
	} catch (IOException ex) {
	    Log.error(_class,
		      "unexpected exception when closing OutputStream");
	}
    }
    
    /**
     * Subclasses should call this version of runPriv() if the task can
     * be performed by running a single privcmd, the privcmd accepts some
     * private data (such as passwords) via stdin, and task failure can be
     * handled by displaying the privcmd output and error text in a standard
     * error dialog.
     * <P>
     * runPriv() will post a busy dialog while the privileged command
     * Process is running.  When the privileged command exits,
     * runPriv() will call taskSucceeded() if the exit code is zero,
     * or it will call taskFailed().
     *
     * @param privcmdName Name of the privileged command to run.
     * @param taskData TaskData attributes to be sent to the privileged command
     *                 via the command line.
     *
     * @return An OutputStream that clients can use to send private data or
     *         control data to the privileged command.  The caller is
     *         responsible for closing the OutputStream.
     */
    protected OutputStream runPriv(String privcmdName, TaskData taskData) {
	return runPriv(privcmdName, new RunPrivArgs(taskData), (Category)null,
		       (ItemTester)null, EVENT_TYPE_NONE);
    }

    /**
     * This version of runPriv() is just like runPriv(String, TaskData), but if
     * its ItemTester argument is non-null and its eventType isn't
     * EVENT_TYPE_NONE, it delays success until the specified event type  has
     * occurred in the Category on an Item matching the ItemTester.  If this
     * does not happen within <i>Task.eventTimeout</i> seconds, it calls
     * taskFailed().
     * <P>
     * You might want to do this when the Task is launched from a metatask, and
     * subsequent steps cannot be performed until after the changes caused by
     * the Task's privcmd have been propagated back to the client. In general,
     * it's probably better to <em>not</em> use this feature unless absolutely
     * necessary, as it means the user spends longer staring at an hourglass.
     * <P>
     * If the eventType is EVENT_TYPE_CHANGE, the ItemFinder should match the
     * Item in its new (modified) state, not its state before modification.
     * For other information on how this method determines whether the expected
     * event has been received, see the <CODE>waitForEvent</CODE> method.
     *
     * @param privcmdName Name of the privileged command to run.
     * @param taskData TaskData attributes to be sent to the privileged command
     *                 via the command line.
     * @param category Category to be monitored for the event.  If itemTester
     *                 is null or the eventType is EVENT_TYPE_NONE, this is
     *                 ignored.
     * @param itemTester ItemTester used to determine whether Category events
     *                   are for the Item of interest.  If this is null, no
     *                   event will be waited for.
     * @param eventType The type of event expected on the item.  This must be
     *                  EVENT_TYPE_NONE, EVENT_TYPE_ADD, EVENT_TYPE_CHANGE, or
     *                  EVENT_TYPE_REMOVE.  If this is EVENT_TYPE_NONE, no
     *                  event will be waited for.
     *
     * @return An OutputStream that clients can use to send private data or
     *         control data to the privileged command.  The caller is
     *         responsible for closing the OutputStream.
     *
     * @see #waitForEvent(ResultListener, Category, ItemTester, int)
     */
    protected OutputStream runPriv(String privcmdName, TaskData taskData,
    				     Category category, ItemTester itemTester,
				     int eventType) {
	return runPriv(privcmdName, new RunPrivArgs(taskData), category,
		       itemTester, eventType);
    }

    /**
     * Subclasses should call this version of runPriv() if the multiple
     * privcmds are needed to accomplish the task or the subclass wants to
     * regain control after the privcmd completes.
     * <P>
     * The caller is responsible for posting a busy dialog while the
     * privileged command(s) run(s), then clearing that dialog and calling
     * taskSucceeded() or taskFailed() as appropriate when the
     * privileged command(s) succeed(s) or fail(s).
     * <P>
     * If the exit code of the privileged command Process is zero,
     * runPriv will call listener.succeeded, otherwise runPriv will call
     * listener.failed.  The ResultEvent passed to the listener
     * will contain the privileged command Process output text as the
     * result and the privcmd error text as the reason.
     *
     * @param privcmdName Name of the privileged command to run.
     * @param taskData TaskData attributes to be sent to the privileged command
     *                 via the command line.
     * @param listener ResultListener to be notified when the privileged
     *                 command completes.
     *
     * @return An OutputStream that clients can use to send private data or
     *         control data to the privileged command.  The caller is
     *         responsible for closing the OutputStream.
     */
    protected OutputStream runPriv(String privcmdName, TaskData taskData,
				   final ResultListener listener) {
	return runPriv(privcmdName, new RunPrivArgs(taskData), listener,
		       (Category)null, (ItemTester)null, EVENT_TYPE_NONE);
    }
    
    /**
     * Subclasses should call this version of runPriv() if the task needs
     * full control over privileged command input and output and needs to
     * regain control when the privileged command exits.  For example, a task
     * that runs multiple privileged commands or displays privileged command
     * output as it arrives would use this version of runPriv().
     * <P>
     * The subclass is responsible for displaying a busy dialog, if any,
     * while the privileged command is running, as well as for responding
     * to privileged command exit (via ProcessListener.processExited())
     * by calling Task.taskSucceeded() for successful completion or
     * Task.taskFailed() on failure.
     *
     * @param privcmdName Name of the privileged command to run.
     * @param taskData TaskData attributes to be sent to the privileged command
     *                 via the command line.
     * @param listener A ProcessListener for monitoring privilege command
     *                 output and exit status.  May not be null.
     *
     * @return An OutputStream that clients can use to send private data or
     *         control data to the privileged command.  The caller is
     *         responsible for closing the OutputStream.
     */
    protected OutputStream runPriv(String privcmdName, TaskData taskData,
				   ProcessListener listener) {
	return runPriv(privcmdName, new RunPrivArgs(taskData), listener);
    }

    /**
     * Subclasses should call this version of runPriv() if the task can
     * be performed by running a single privcmd, the privcmd accepts some
     * private data (such as passwords) via stdin, and task failure can be
     * handled by displaying the privcmd output and error text in a standard
     * error dialog.
     * <P>
     * runPriv() will post a busy dialog while the privileged command
     * Process is running.  When the privileged command exits,
     * runPriv() will call taskSucceeded() if the exit code is zero,
     * or it will call taskFailed().
     *
     * @param privcmdName Name of the privileged command to run.
     * @param argString A String containing the command line arguments
     *			for privileged command.
     *
     * @return An OutputStream that clients can use to send private data or
     *         control data to the privileged command.  The caller is
     *         responsible for closing the OutputStream.
     */
    protected OutputStream runPriv(String privcmdName, String argString) {
	return runPriv(privcmdName, new RunPrivArgs(argString), (Category)null,
		       (ItemTester)null, EVENT_TYPE_NONE);
    }

    /**
     * Subclasses should call this version of runPriv() if the multiple
     * privcmds are needed to accomplish the task or the subclass wants to
     * regain control after the privcmd completes.
     * <P>
     * The caller is responsible for posting a busy dialog while the
     * privileged command(s) run(s), then clearing that dialog and calling
     * taskSucceeded() or taskFailed() as appropriate when the
     * privileged command(s) succeed(s) or fail(s).
     * <P>
     * If the exit code of the privileged command Process is zero,
     * runPriv will call listener.succeeded, otherwise runPriv will call
     * listener.failed.  The ResultEvent passed to the listener
     * will contain the privileged command Process output text as the
     * result and the privcmd error text as the reason.
     *
     * @param privcmdName Name of the privileged command to run.
     * @param argString A String containing the command line arguments
     *			for privileged command.
     * @param listener ResultListener to be notified when the privileged
     *                 command completes.
     *
     * @return An OutputStream that clients can use to send private data or
     *         control data to the privileged command.  The caller is
     *         responsible for closing the OutputStream.
     */
    protected OutputStream runPriv(String privcmdName, String argString,
				   final ResultListener listener) {
	return runPriv(privcmdName, new RunPrivArgs(argString), listener,
		       (Category)null, (ItemTester)null, EVENT_TYPE_NONE);
    }
    
    /**
     * Subclasses should call this version of runPriv() if the task needs
     * full control over privileged command input and output and needs to
     * regain control when the privileged command exits.  For example, a task
     * that runs multiple privileged commands or displays privileged command
     * output as it arrives would use this version of runPriv().
     * <P>
     * The subclass is responsible for displaying a busy dialog, if any,
     * while the privileged command is running, as well as for responding
     * to privileged command exit (via ProcessListener.processExited())
     * by calling Task.taskSucceeded() for successful completion or
     * Task.taskFailed() on failure.
     *
     * @param privcmdName Name of the privileged command to run.
     * @param argString A String containing the command line arguments
     *			for privileged command.
     * @param listener A ProcessListener for monitoring privilege command
     *                 output and exit status.  May not be null.
     *
     * @return An OutputStream that clients can use to send private data or
     *         control data to the privileged command.  The caller is
     *         responsible for closing the OutputStream.
     */
    protected OutputStream runPriv(String privcmdName, String argString,
				   ProcessListener listener) {
	return runPriv(privcmdName, new RunPrivArgs(argString), listener);
    }
    
    /**
     * Subclasses should call this version of runPriv() if the task can
     * be performed by running a single privcmd, the privcmd accepts some
     * private data (such as passwords) via stdin, and task failure can be
     * handled by displaying the privcmd output and error text in a standard
     * error dialog.
     * <P>
     * runPriv() will post a busy dialog while the privileged command
     * Process is running.  When the privileged command exits,
     * runPriv() will call taskSucceeded() if the exit code is zero,
     * or it will call taskFailed().
     *
     * @param privcmdName Name of the privileged command to run.
     * @param argVector A String array containing the command line arguments
     *			for privileged command.
     *
     * @return An OutputStream that clients can use to send private data or
     *         control data to the privileged command.  The caller is
     *         responsible for closing the OutputStream.
     */
    protected OutputStream runPriv(String privcmdName, String argVector[]) {
	return runPriv(privcmdName, new RunPrivArgs(argVector), (Category)null,
		       (ItemTester)null, EVENT_TYPE_NONE);
    }

    /**
     * Subclasses should call this version of runPriv() if the multiple
     * privcmds are needed to accomplish the task or the subclass wants to
     * regain control after the privcmd completes.
     * <P>
     * The caller is responsible for posting a busy dialog while the
     * privileged command(s) run(s), then clearing that dialog and calling
     * taskSucceeded() or taskFailed() as appropriate when the
     * privileged command(s) succeed(s) or fail(s).
     * <P>
     * If the exit code of the privileged command Process is zero,
     * runPriv will call listener.succeeded, otherwise runPriv will call
     * listener.failed.  The ResultEvent passed to the listener
     * will contain the privileged command Process output text as the
     * result and the privcmd error text as the reason.
     *
     * @param privcmdName Name of the privileged command to run.
     * @param argVector A String array containing the command line arguments
     *			for privileged command.
     * @param listener ResultListener to be notified when the privileged
     *                 command completes.
     *
     * @return An OutputStream that clients can use to send private data or
     *         control data to the privileged command.  The caller is
     *         responsible for closing the OutputStream.
     */
    protected OutputStream runPriv(String privcmdName, String argVector[],
				   final ResultListener listener) {
	return runPriv(privcmdName, new RunPrivArgs(argVector), listener,
		       (Category)null, (ItemTester)null, EVENT_TYPE_NONE);
    }
    
    /**
     * Subclasses should call this version of runPriv() if the task needs
     * full control over privileged command input and output and needs to
     * regain control when the privileged command exits.  For example, a task
     * that runs multiple privileged commands or displays privileged command
     * output as it arrives would use this version of runPriv().
     * <P>
     * The subclass is responsible for displaying a busy dialog, if any,
     * while the privileged command is running, as well as for responding
     * to privileged command exit (via ProcessListener.processExited())
     * by calling Task.taskSucceeded() for successful completion or
     * Task.taskFailed() on failure.
     *
     * @param privcmdName Name of the privileged command to run.
     * @param argVector A String array containing the command line arguments
     *			for privileged command.
     * @param listener A ProcessListener for monitoring privilege command
     *                 output and exit status.  May not be null.
     *
     * @return An OutputStream that clients can use to send private data or
     *         control data to the privileged command.  The caller is
     *         responsible for closing the OutputStream.
     */
    protected OutputStream runPriv(String privcmdName, String argVector[],
				   ProcessListener listener) {
	return runPriv(privcmdName, new RunPrivArgs(argVector), listener);
    }


    private OutputStream runPriv(String privcmdName, RunPrivArgs privcmdArgs,
				   Category category, ItemTester itemTester,
				   int eventType) {
	// XXX Don't allow cancel for now.
	_taskContext.busy(_rs.getString("Task.Busy.performingTask"));
	return runPriv(privcmdName, privcmdArgs, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		_taskContext.notBusy();
		taskSucceeded(event);
	    }

	    public void failed(ResultEvent event) {
		_taskContext.notBusy();
		taskFailed(event);
	    }
	}, category, itemTester, eventType);
    }

    private OutputStream runPriv(String privcmdName, RunPrivArgs privcmdArgs,
				 final ResultListener listener,
				 final Category category,
				 final ItemTester itemTester,
				 final int eventType) {
	return runPriv(privcmdName, privcmdArgs, new ProcessListener() {
	    private StringBuffer _outputText = new StringBuffer("");
	    private StringBuffer _errorText = new StringBuffer("");
	    
	    // Handle privileged command output to stdout.
	    public void processOutputData(ProcessEvent event) {
		_outputText.append(new String(event.getBytes()));
	    }
	    
	    // Handle privileged command output to stderr.
	    public void processErrorData(ProcessEvent event) {
		_errorText.append(new String(event.getBytes()));
	    }
	    
	    public void processExited(ProcessEvent event) {
		ResultEvent result =
		    new ResultEvent(Task.this, _outputText.toString().trim());
		result.setReason(_errorText.toString().trim());

		if (event.getExitCode() == 0) {
		    // The privcmd succeeded.  If this Task wants to wait for
		    // an Item to be affected by its privcmd, do that here.
		    if (itemTester == null || eventType == EVENT_TYPE_NONE) {
			listener.succeeded(result);
		    } else {
			waitForEvent(listener, category, itemTester, eventType);
		    }
		} else {
		    listener.failed(result);
		}
	    }
	});
    }

    private OutputStream runPriv(String privcmdName, RunPrivArgs privcmdArgs,
				   ProcessListener listener) {
	Log.assert(listener != null, "You must supply a ProcessListener to " +
		   "handle processExited(), including calling taskDone() on " +
		   "successful completion of the privileged command.");
		   
	if (_noRunPrivFlag) {
	    Log.warning(_class, "runPriv being skipped because " +
			DEMO_NO_RUNPRIV + " flag set.");
	    listener.processExited(new ProcessEvent(
		new NullProcess(), ProcessEvent.EXITED));
	    return new NullOutputStream();
	}
	_taskSucceeded = false;
	
	// Start the privileged command.
	Log.assert(_privBroker != null,
		   "You must call checkPrivs() before calling runPriv()."
		   + "\nTo do this, you can define the Task.privList resource set"
		   + "\nin your task's properties file.");
	Process process = null;
	if (privcmdArgs._taskData != null) {
	    process = _privBroker.runPriv(privcmdName, privcmdArgs._taskData);
	} else if (privcmdArgs._argString != null) {
	    process = _privBroker.runPriv(privcmdName, privcmdArgs._argString);
	} else if (privcmdArgs._argVector != null) {
	    process = _privBroker.runPriv(privcmdName, privcmdArgs._argVector);
	} else {
	    Log.fatal("Missing privcmd args");
	}

	// Create a watcher for the privileged command.
	ProcessWatcher watcher = new ProcessWatcher(process);
	
	// Initiate privileged command monitoring.
	watcher.addProcessListener(listener);

	watcher.startWatching();
	
	return process.getOutputStream();
    }

    /**
     * Wait for an event on a specific Item.  The Category &amp; ItemTester
     * identify the Item, and the <CODE>eventType</CODE> indicates
     * whether to wait for an add, a change, or a removal.
     * <P>
     * If the expected event is EVENT_TYPE_ADD, listener.succeeded() is called
     * when an Item matching the ItemTester is added, or if it's already in the
     * Category.
     * <P>
     * If the expected event is EVENT_TYPE_CHANGE, listener.succeeded() is
     * called when an Item is changed to match the ItemTester, or if an item
     * matching the ItemTester is already in the Category.
     * <P>
     * If the expected event is EVENT_TYPE_REMOVE, listener.succeeded() is
     * called when an Item matching the ItemTester is removed, or if the item
     * isn't already in the Category when monitoring begins.
     * <P>
     * If the expected event is EVENT_TYPE_NONE, this method will assert out.
     *
     * @param listener The ResultListener to call succeeded() on when the
     *                 expected event is received, or failed() on if the event
     *                 isn't received within <i>Task.eventTimeout</i>
     *                 seconds.  Must not be null.
     * @param category The Category to listen on for events.  May not be null.
     * @param tester The ItemTester which will identify the Item on which we're
     *               waiting for the event on.  Must not be null.
     * @param eventType The type of event we're waiting for.  Must be 
     *                  EVENT_TYPE_ADD, EVENT_TYPE_CHANGE, or EVENT_TYPE_REMOVE.
     */
    protected void waitForEvent(ResultListener listener, Category category,
				 ItemTester tester, int eventType) {
	new ItemEventWaiter(listener, category, tester, eventType);
    }

    //
    // Task private methods
    //

    /**
     * Set up the user interface for this Task.
     */
    private void setupInterface() {
	setLayout(new BorderLayout());
	
	// Create a panel for the Form or Guide UI
	_taskUIPanel = new JPanel(new CardLayout());
	add(_taskUIPanel, BorderLayout.CENTER);

	// Create a panel for a separator and the TaskControlPanel
	JPanel lowerPanel = new JPanel(new BorderLayout());
	lowerPanel.add(new JSeparator(), BorderLayout.NORTH);
	
	// Create the Task control panel
	_controlPanel = new TaskControlPanel(_rs);
	lowerPanel.add(_controlPanel, BorderLayout.SOUTH);
	_taskContext.setTaskControlPanel(_controlPanel);
	_controlPanel.addControlListener(new TaskControlAdapter() {
	    /**
	     * The Form/Guide toggle button has been pressed.  Toggle between
	     * the Form and Guide interfaces.
	     */
	    public void taskToggleInterface() {
		if (_mode == FORM_MODE) {
		    setGuideMode();
		} else {
		    setFormMode();
		}
	    }
	    
	    public synchronized void taskCancel() {
		if (_taskInProgress) {
		    // Don't allow the Task to be cancelled twice.
		    Toolkit.getDefaultToolkit().beep();
		    return;
		}
		_taskInProgress = true;

		// Cancel the task
		taskDone(new TaskResult(Task.this, TaskResult.CANCELLED));
		_taskInProgress = false;
	    }
	    
	    public synchronized void taskOK() {
		if (_taskInProgress) {

		    // Don't allow the Task to be executed twice.
		    Toolkit.getDefaultToolkit().beep();
		    return;
		}
		_taskInProgress = true;

		// Prevent a second press of the OK button
		_taskContext.busy();

		// If we're in Guide mode, first verify that the data on the
		// last page is valid.  Then verify that all of the
		// TaskData is valid.
		if (_guide != null && _mode == GUIDE_MODE) {
		    _guide.getCurrentPage().okToAdvance(new
							ResultListener() {
			public void succeeded(ResultEvent event) {
			    verifyTaskData();
			}

			public void failed(ResultEvent event) {
			    verificationFailed(event);
			}
		    });
		} else {
		    verifyTaskData();
		}
	    }
	    
	    public void taskHelp() {
		try {
		    HostContext.help(_rs.getString(HostContext.HELP_SET),
                                     _rs.getString(HELP_KEY));
		} catch (MissingResourceException ex) {
		    _taskContext.postInfo(
			_rs.getString("Task.Info.noHelpAvailable"));
		}
	    }
	});
    
	add(lowerPanel, BorderLayout.SOUTH);
	
	// Tell the subclass to register the Form and/or Guide interfaces.
	registerInterfaces();
		
	// Make sure registerInterfaces() called setForm() and/or setGuide().
	Log.assert(_form != null || _guide != null,
		   "Missing call to Task.setForm() or Task.setGuide()");
	
	// If only one interface is registered, disable the interface
	// toggle components in the TaskControlPanel.
	if (_form == null || _guide == null) {
	    _controlPanel.setInterfaceToggleEnabled(false);
	}
		   
	//Enable the cancel button based on a resource
	_controlPanel.setCancelButtonEnabled(
	    _rs.getBoolean(SHOW_CANCEL_BUTTON));

	// Set the initial interface to display.
	if ((_preferredUI == Task.FORM_MODE && _form != null) |
	     _guide == null) {
	    setFormMode();
	} else {
	    setGuideMode();
	}
    }
    
    
    /**
     * Display the Form interface.
     */
    private void setFormMode() {
	Log.assert(_form != null,
	    "Call to Task.setFormMode() before call to Task.setForm()");
	
	_form.showForm();
	_controlPanel.setFormMode();
	
	((CardLayout)_taskUIPanel.getLayout()).show(_taskUIPanel, FORM_UI);
	_mode = FORM_MODE;
    }
    
    /**
     * Display the Guide interface.
     */
    private void setGuideMode() {
	Log.assert(_guide != null,
	    "Call to Task.setGuideMode() before call to Task.setGuide()");
	
	_guide.showGuide();
	_controlPanel.setGuideMode();
	
	((CardLayout)_taskUIPanel.getLayout()).show(_taskUIPanel, GUIDE_UI);
	_mode = GUIDE_MODE;
    }
    
    /**
     * Record which TaskData attributes are public, if any.
     */
    private void getPublicDataKeys() {
	Log.assert(_publicDataKeys == null,
		   "Duplicate call to getPublicDataKeys()");
		   
	_publicDataKeys = new Hashtable();
	String[] publicData = null;
	
	try {
	    publicData = _rs.getStringArray(Task.PUBLIC_DATA);

	    for (int ii = 0; ii < publicData.length; ii++) {
		Attribute attr = _taskData.getAttr(publicData[ii]);
		Log.assert(attr != null,
		    "Attempt to make non-existent TaskData attribute '" +
		    publicData[ii] + "' public.  Check your properties file.");
		_publicDataKeys.put(publicData[ii], attr.getTypeString());
	    }
	    
	} catch (MissingResourceException exception) {
	}
    }

    private void verifyTaskData() {
	// Verify the data for this task.
	_taskContext.allDataOK(TaskDataVerifier.MUST_BE_SET,
			       null, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		// Tell the subclass to perform the operation.
		ok();
	    }
		    
	    public void failed(ResultEvent event) {
		verificationFailed(event);
	    }
	});
    }

    private void verificationFailed(ResultEvent event) {
	// If the ResultEvent contains a reason for the
	// failure, post that in an error dialog.
	String reason = event.getReason();
	if (reason != null) {
	    _taskContext.postError(reason,
				   new ActionListener() {
		public void actionPerformed(ActionEvent ev) {
		    _taskContext.notBusy();
		    _taskInProgress = false;
		}
	    });
	} else {
	    _taskContext.notBusy();
	    _taskInProgress = false;
	}
    }

    /**
     * Checks all privileges needed for several tasks.  It gets all of
     * the privileges out of all of the ResourceFiles and checks them
     * all in one batch.  If any privilege is not granted,
     * listener.failed() will be called.
     *
     * @param tasks An array of task loaders
     * @param listener The listener to notify of success or failure.
     */
    protected void checkPrivs(TaskLoader[] tasks, ResultListener listener) {
	// XXX Use a Set instead of a Hashtable when it becomes available.
	Hashtable privs = new Hashtable();
	Object dummy = new Object();
	for (int ii = 0; ii < tasks.length; ii++) {
	    try {
		String[] privlist = 
		    tasks[ii].getResourceStack().getStringArray(PRIV_LIST);
		for (int jj = 0; jj < privlist.length; jj++) {
		    privs.put(privlist[jj], dummy);
		}
	    } catch (MissingResourceException mre) {
		// do nothing.  privs are optional.
	    }
	}
	    
	String[] privList = new String[privs.size()];
	Enumeration enum = privs.keys();
	int ii = 0;
	while (enum.hasMoreElements()) {
	    privList[ii++] = (String)enum.nextElement();
	}
	
	checkPrivsOnServer(privList, listener);
    }


    /**
     * A class to use if we're in Demo.noRunPriv Mode
     */
    private class NullOutputStream extends OutputStream {
	public void write(int b) {
	}
    } 
    
    /**
     * A class to use if we're in Demo.noRunPriv Mode
     */
    private class NullProcess extends Process{
	public void destroy() {
	}
	public int exitValue() {
	    return 0;
	}
	public InputStream getErrorStream() {
	    return null;
	}
	public InputStream getInputStream() {
	    return null;
	}
	public OutputStream getOutputStream() {
	    return null;
	}
	public int waitFor() {
	    return 0;
	}
    }

    /**
     * A class which waits for a specific type of event on an Item.  See the
     * comment for waitForEvent().
     */
    private class ItemEventWaiter implements ActionListener, CategoryListener {

	private Category       category;
	private ResultListener listener;
	private ItemTester     tester;
	private int            eventType;
	private boolean        itemExists = false;
	private Timer          timer;

	public ItemEventWaiter(ResultListener listener, Category category, 
				 ItemTester tester, int eventType) {
	    Log.assert(eventType != EVENT_TYPE_NONE,
        	       "Waiting for EVENT_TYPE_NONE?");
	    Log.assert((eventType == EVENT_TYPE_ADD ||
			eventType == EVENT_TYPE_CHANGE ||
			eventType == EVENT_TYPE_REMOVE), "Invalid eventType \""
		       + eventType + "\" passed to ItemEventWaiter");
	    int timeout = _rs.getInt(EVENT_TIMEOUT);
	    Log.debug("ItemEventWaiter", "Waiting " + timeout + " seconds for "
		      + (eventType == EVENT_TYPE_ADD ? "ADD" :
		      (eventType == EVENT_TYPE_CHANGE ? "CHANGE" : "REMOVE"))
		      + " on category \"" + category.getSelector() + "\"");
	    this.category = category;
	    this.listener = listener;
	    this.tester = tester;
	    this.eventType = eventType;
	    //  Woo hoo hoo!  Upon adding the category listener, you can get
	    //  itemAdded() events before the constructor completes, meaning
	    //  it's possible to hit succeed() while timer is still null.
	    //  (I didn't think that could happen, but it does.)
	    category.addCategoryListener(this);
	    timer = new Timer(timeout * 1000, this);
	    timer.setRepeats(false);
	    //  To indicate that we're already done & shouldn't start the
	    //  timer, succeed() sets listener = null.
	    if (this.listener != null) timer.start();
	}

	/**
	 * If we're waiting for an add, and this is our item, we're done.
	 * If we're waiting for a modify, and this item matches the item we
	 * expect to wind up with, we're done.
	 * If we're waiting for a delete, and this is our item, we know we're
	 * going to have to keep waiting...
	 */
	public void itemAdded(Item item) {
	    Log.trace("ItemEventWaiter", category.getSelector() +
			" itemAdded: " + item.getSelector());
	    if ((eventType == EVENT_TYPE_ADD) ||
		(eventType == EVENT_TYPE_CHANGE)) {
		if (tester.testItem(item).passed()) {
		    succeed();
		}
	    }
	    else if (eventType == EVENT_TYPE_REMOVE) {
		if (tester.testItem(item).passed()) {
		    itemExists = true;
		}
	    }
	}

	/**
	 * If we're waiting for a modify, and newItem matches what we're
	 * waiting for the Item to look like, we're done.  And, heck, as
	 * long as we're here, we might as well succeed if we're waiting
	 * for an add.
	 */
	public void itemChanged(Item oldItem, Item newItem) {
	    Log.trace("ItemEventWaiter", category.getSelector() +
			" itemChanged: " + newItem.getSelector());
	    if (((eventType == EVENT_TYPE_CHANGE) ||
	         (eventType == EVENT_TYPE_ADD)) &&
		tester.testItem(newItem).passed()) {
		succeed();
	    }
	}

	/**
	 * If we're waiting for a delete, and this is our item, we're done.
	 */
	public void itemRemoved(Item item) {
	    Log.trace("ItemEventWaiter", category.getSelector() +
			" itemRemoved: " + item.getSelector());
	    if (eventType == EVENT_TYPE_REMOVE) {
		if (tester.testItem(item).passed()) {
		    succeed();
		}
	    }
	}

	/**
	 * If we're waiting for a delete, and our item doesn't exist, we're
	 * done.
	 */
	public void endExists() {
	    Log.trace("ItemEventWaiter", category.getSelector() + " endExists");
	    if ((eventType == EVENT_TYPE_REMOVE) && (!itemExists)) {
		succeed();
	    }
	}

	/**
	 * We received the event we were waiting for.  Clean things up and
	 * notify the listener.
	 */
	private void succeed() {
	    //  In case we get here before the constructor completes, indicate
	    //  that the timer shouldn't be started.
	    ResultListener raceConditionKlugeMeister = listener;
	    listener = null;

	    Log.debug("ItemEventWaiter",
			"Received expected event on category \"" +
			category.getSelector() + "\"");

	    if (timer != null) timer.stop();
	    category.removeCategoryListener(this);
	    if (raceConditionKlugeMeister != null) {
		raceConditionKlugeMeister.succeeded(new ResultEvent(Task.this));
	    }
	}

	/**
	 * Called by our Timer when we're done waiting for the event to arrive
	 * for our item.  This means we've timed out before it's arrived, so we
	 * clean things up and call listener.failed().
	 */
	public void actionPerformed(ActionEvent event) {
	    Log.info("ItemEventWaiter", "Timed out waiting for " +
		       (eventType == EVENT_TYPE_ADD ? "ADD" :
			 (eventType == EVENT_TYPE_CHANGE ? "CHANGE" : "REMOVE"))
		       + " on category \"" + category.getSelector() + "\"");
	    timer.stop();  // unnecessary because of timer.setRepeats(false)?
	    category.removeCategoryListener(this);
	    if (listener != null) {
		ResultEvent result =
		    new ResultEvent(Task.this, "");
		result.setReason(_rs.getString(EVENT_TIMEOUT_REASON));
		listener.failed(result);
	    }
	}

	public void beginBlockChanges() {}
	public void endBlockChanges() {}

	//  AttrListener methods
	public void attrAdded(AttrEvent event) {}
	public void attrChanged(AttrEvent event) {}
	public void attrRemoved(AttrEvent event) {}
    }
}
