//
// TaskContext.java
//
//	A UIContext that provides services to Tasks such as access
//	to TaskData and maintaining the Task title.
// 
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;
import java.awt.*;
import java.text.MessageFormat;
import java.util.*;

/**
 * TaskContext is a UIContext that provides additional services to
 * Tasks, their subclasses, and components such as creating a Task specific
 * ResourceStack, accessing TaskData, verifying TaskData, and
 * maintaining the Task title.
 * <p>
 * TaskContext creates and maintains a ResourceStack to encourage sharing
 * of properties across multiple Tasks and to allow Tasks to fall back on
 * default properties defined by the client package or by the infrastructure.
 * The creator of TaskContext, typically TaskLoader, must push the (optional)
 * Package ResourceBundle and (required) Task ResourceBundle onto the
 * TaskContext ResourceStack.
 * <P>
 * TaskContext supports access to TaskData and verification of
 * TaskData.  Task subclasses create TaskDataVerifiers and add them to
 * the list of verifiers maintained by TaskContext so that the
 * verifiers may be shared among the many classes that make up a
 * Task.  For example, if a Task has both a Form and Guide interface,
 * they can share TaskDataVerifiers rather than each defining their
 * own (duplicate) verifiers.
 * <P>
 * TaskData verification frequently requires that multiple
 * asynchronous calls be made in sequence, one call for each piece of
 * TaskData.  For example, the Create PPP Connection Task might need
 * to confirm with the server that the proposed User Name exists and
 * then that the proposed Connection name does not exist.  If any
 * verification in the sequence fails, no further calls in the
 * sequence are made and a notification of failure is sent to the
 * caller.  This sequential, asynchronous verification process is also
 * known as <I>chained</I> verification.
 * <P>
 * TaskContext provides support for chaining by allowing clients to
 * create an ordered list of TaskDataVerifiers by calls to
 * TaskContext.appendTaskDataVerifier() or
 * TaskContext.insertTaskDataVerifier().  Clients can then trigger
 * individual TaskDataVerifiers by calling TaskContext.dataOK(), or
 * they can trigger the entire chained verification by calling
 * TaskContext.allDataOK().
 * <P>
 * See the <A HREF="../tutorials/HowToWriteATask.html">How to Write a
 * Task</A> tutorial for details and examples on verifying TaskData.
 * 
 * @see com.sgi.sysadm.util.ResourceStack
 * @see com.sgi.sysadm.ui.taskData.TaskDataVerifier
 */
public class TaskContext extends UIContext {
    private Task _task;
    private TaskData _taskData = new TaskData();
    private TaskControlPanel _controlPanel;
    private String _taskTitle; 
    private Vector _titleListeners = new Vector();
    private Vector _verifiers = new Vector();
    private Hashtable _verifierLookup = new Hashtable();
    private String _titleString;
    
    /**
     * When the user presses the OK button, the Task will call
     * TaskContext.allDataOK(TaskDataVerifiers.MUST_BE_SET).  If
     * a verifier named TaskContext.ALL_DATA_VERIFIER has been added,
     * that verifier will be called to determine if the TaskData is valid.
     * Otherwise each TaskDataVerifier that has been added will be called
     * in order.
     */
    public static final String ALL_DATA_VERIFIER =
				    "TaskContext.allDataVerifier";
    
    
    /**
     * Constructor.
     * <p>
     * Only package visibility is allowed to prevent Task clients from
     * attempting to instantiate this class directly.
     *
     * @param hostContext The HostContext associated with this TaskContext.
     */
    /* package */ TaskContext(HostContext hostContext) {
	super();
	setHostContext(hostContext);
	// Add the TaskContext ResourceBundle to the ResourceStack
	getResourceStack().pushBundle(
	    getClass().getName() + ResourceStack.BUNDLE_SUFFIX);
    }
    
     /**
     * Constructor that takes a ResourceStack
     * <p>
     * This should be used only if the ResourceStack that you pass is
     * guaranteed to have SysadmUIP and TaskContextP on it.
     * Only package visibility is allowed to prevent Task clients from
     * attempting to instantiate this class directly.
     *
     * @param hostContext The HostContext associated with this TaskContext.
     * @param rs The ResourceStack to use with this TaskContext.
     */
    /* package */ TaskContext(HostContext hostContext, ResourceStack rs) {
	super(rs);
	setHostContext(hostContext);
    }
    
    
    /**
     * Identify the task that will be associated with this context so that
     * we can respond to task completion.  The dialog parent of this
     * TaskContext will be set to the task, and the initial dialog title
     * will be set to the property Task.SHORT_NAME.
     * <p>
     * This method may only be called once per instance of TaskContext, and
     * should only be called by the owner of the instance.
     *
     * @param task Task associated with instance of TaskContext.
     */
    /* package */ void setTask(Task task) {
	Log.assert(_task == null, "Duplicate call to TaskContext.setTask()");
	
	_task = task;
	String shortName = getResourceStack().getString(Task.SHORT_NAME);
	setDialogTitle(shortName);
	setTitleString(
	    MessageFormat.format(getResourceStack().getString(Task.TITLE_FORMAT), 
				 new Object[] { shortName,
						getHostContext().getHostName()
					       }));
    }
    
    /**
     * Get the Task that is associated with this TaskContext.
     *
     * @return the Task associated with this context,
     *	       <TT>null</TT> if TaskContext().setTask() has not been
     *         called yet.
     */
    public Task getTask() {
	return _task;
    }
    

    /**
     * Get the TaskData object created for the TaskContext.  There will be
     * no TaskData attributes until TaskContext.setTask() has been called.
     *
     * @return The TaskData object created for this TaskContext.
     */
    public TaskData getTaskData() {
	return _taskData;
    }
    
    
    /**
     * Set the TaskControlPanel associated with this TaskContext.
     * 
     * @param controlPanel TaskControlPanel associated with this TaskContext.
     */
    /* package */ void setTaskControlPanel(TaskControlPanel controlPanel) {
	_controlPanel = controlPanel;
    }
    
    /**
     * Get the TaskControlPanel associated with this TaskContext.
     *
     * @return The TaskControlPanel associated with this context,
     *         null if TaskContext.setTask() has not been called yet.
     */
    public TaskControlPanel getTaskControlPanel() {
	return _controlPanel;
    }
    
    /**
     * Get the name of the server associated with this TaskContext.
     * This is a convenience routine and returns the same value as
     * found in the HostContext for this Task.
     *
     * @return Name of the server associate with this TaskContext.
     * 
     * @see com.sgi.sysadm.ui.HostContext
     */
    public String getServerName() {
	return getHostContext().getHostName();
    }

    /**
     * Set the string to be used as the Task title.  This will be the
     * exact title string when the Form interface is displayed.  When
     * the Guide interface is displayed, additional information about
     * the current page number and total number of pages in the guide
     * will be appended to the title.
     *
     * If this method is not called by the Task subclass, then the
     * title will be formed using the property Task.TITLE_FORMAT,
     * which includes the hostname and Task.shortName.
     *
     * @param titleString The string to be used as the Task title.
     */
    public void setTitleString(String titleString) {
	_titleString = titleString;
    }

    /**
     * Get the string to be used as the Task title when the Form
     * interface is displayed.
     *
     * @return The String to be used as the Form title.
     */
    public String getTitleString() {
	return _titleString;
    }
    
    /**
     * Add a listener to the list of those interested in changes to
     * the Task title.  The listener will be notified immediately if
     * the Task title has already been set.
     *
     * @param listener The TitleListener interested in changes to
     *                 the Task title.
     */
    public void addTitleListener(TitleListener listener) {
	if (_taskTitle != null) {
	    listener.titleChanged(new TitleEvent(_task, _taskTitle));
	}
	_titleListeners.addElement(listener);
    }
    
    /**
     * Remove a listener from the list of those interested in changes to
     * the Task title.
     *
     * @param listener The TitleListener no longer interested in changes
     *                 to the Task title.
     */
    public void removeTitleListener(TitleListener listener) {
	_titleListeners.removeElement(listener);
    }
    
    /**
     * Set the title of the Task.  Notifies all TaskTitleListeners.
     *
     * @param taskTitle Localized Task title.
     */
    public void setTaskTitle(String taskTitle) {
	Log.assert(_task != null,
		   "call to setTaskTitle before call to setTask");
	
	_taskTitle = taskTitle;
    	
	// Notify listeners (if any) of the title change.
	TitleEvent titleEvent = new TitleEvent(_task, _taskTitle);
	Enumeration enum = _titleListeners.elements();
	while (enum.hasMoreElements()) {
	    TitleListener listener =
		(TitleListener)enum.nextElement();
	    listener.titleChanged(titleEvent);
	}
    }

    /**
     * Append a TaskDataVerifier onto the list of verifiers.
     *
     * @param verifierName The name that will be used to refer to the added
     *        verifier, usually the name of the TaskData attribute being
     *        verified or TaskContext.ALL_DATA_VERIFIER.
     * @param verifier The verifier to add.
     */
    public void appendTaskDataVerifier(String verifierName,
				       TaskDataVerifier verifier) {
	_verifiers.addElement(verifier);
	_verifierLookup.put(verifierName, verifier);
    }
    
    /**
     * Insert a TaskDataVerifier into the list of verifiers.
     *
     * @param previousName The name of the verifier after which the new
     *        verifier should be inserted.  Set to null if you want
     *        the new verifier to be first on the list.
     * @param verifierName The name that will be used to refer to the
     *        inserted verifier, usually the name of the TaskData attribute
     *        being verified or TaskContext.ALL_DATA_VERIFIER.
     * @param verifier The verifier to insert.
     */
    public void insertTaskDataVerifier(String previousName,
				       String verifierName,
				       TaskDataVerifier verifier) {
	int index = 0;
	if (previousName != null) {
	    // Put this verifier after 'previousName'
	    index = _verifiers.indexOf(_verifierLookup.get(previousName));
	}
	
	_verifiers.insertElementAt(verifier, index);
	_verifierLookup.put(verifierName, verifier);
    }
    
    /**
     * Remove a TaskDataVerifier from the list of verifiers.
     *
     * @param verifierName The name of the verifier to remove.
     */
    public void removeTaskDataVerifier(String verifierName) {
	_verifiers.removeElement(_verifierLookup.remove(verifierName));
    }
    
    /**
     * Call a specific TaskDataVerifier that has been added via
     * TaskContext.appendTaskDataVerifier() or
     * TaskContext.insertTaskDataVerifier().
     *
     * @param verifierName The name of the verifier to call.
     * @param browseFlag TaskDataVerifier.MAY_BE_EMPTY if null strings or zero
     *                   values are valid, TaskDataVerifier.MUST_BE_SET if a
     *                   valid value must be entered.
     * @param context An object the verifier can use to behave differently
     *                in different contexts.
     * @param listener A ResultListener to be notified when the verification
     *                 succeeds or fails.
     */
    public void dataOK(String verifierName, int browseFlag, Object context,
		       ResultListener listener) {
	((TaskDataVerifier)_verifierLookup.get(verifierName))
				.dataOK(browseFlag, context, listener);
    }
    
    /**
     * Invoke a sequence of TaskDataVerifiers that have been added via
     * TaskContext.appendTaskDataVerifier() or
     * TaskContext.insertTaskDataVerifier().  The dataOK() method of each
     * verifier will be called in turn.  If a verifier fails,
     * listener.failed() will be called and no additional verifiers
     * will be invoked.  If all of the verifiers succeed, then
     * listener.succeeded() will be called.
     * 
     * @param verifierList An non-<TT>null</TT> array of verifier
     *                     names to call, in order.  The
     *                     names refer to verifiers that have been added via
     *                     TaskContext.appendTaskDataVerifier() or
     *                     TaskContext.insertTaskDataVerifier().
     * @param browseFlag TaskDataVerifier.MAY_BE_EMPTY if null strings or zero
     *                   values are valid, TaskDataVerifier.MUST_BE_SET if a
     *                   valid value must be entered.
     * @param context An object the verifiers can use to behave differently
     *                in different contexts.
     * @param listener A ResultListener to be notified when the verification
     *                 succeeds or fails.
     */
    public void dataOK(String[] verifierList, int browseFlag, Object context,
		       ResultListener listener) {
	Vector v = new Vector(verifierList.length);
	for (int ii = 0; ii < verifierList.length; ii++) {
	    v.addElement(_verifierLookup.get(verifierList[ii]));
	}
	callNextVerifier(v, 0, browseFlag, context, listener);
    }

    /**
     * Invoke a sequence of TaskDataVerifiers.  The dataOK() method of each
     * verifier will be called in turn.  If a verifier fails,
     * listener.failed() will be called and no additional verifiers
     * will be invoked.  If all of the verifiers succeed, then
     * listener.succeeded() will be called.
     * 
     * @param verifierList A non-<TT>null</TT> Vector of verifiers to
     *                     call, in order.
     * @param browseFlag TaskDataVerifier.MAY_BE_EMPTY if null strings or zero
     *                   values are valid, TaskDataVerifier.MUST_BE_SET if a
     *                   valid value must be entered.
     * @param context An object the verifiers can use to behave differently
     *                in different contexts.
     * @param listener A ResultListener to be notified when the verification
     *                 succeeds or fails.
     */
    public void dataOK(Vector verifierList, int browseFlag,
                       Object context, ResultListener listener) {
	callNextVerifier(verifierList, 0, browseFlag, context, listener);
    }

    /**
     * Verify that all TaskData is valid.  If a verifier named
     * TaskContext.ALL_DATA_VERIFIER has been added, then that verifier
     * will be used to verify TaskData.  Otherwise all TaskDataVerifiers
     * that have been added via TaskContext.appendTaskDataVerifier()
     * or TaskContext.insertTaskDataVerifier() will be called, in order,
     * to verify TaskData.
     *
     * @param browseFlag TaskDataVerifier.MAY_BE_EMPTY if null strings or zero
     *                   values are valid, TaskDataVerifier.MUST_BE_SET if a
     *                   valid value must be entered.
     * @param context An object the verifiers can use to behave differently
     *                in different contexts.
     * @param listener A ResultListener to be notified when the verification
     *                 succeeds or fails.
     */
    public void allDataOK(int browseFlag, Object context,
			  ResultListener listener) {
	TaskDataVerifier allDataVerifier =
	    (TaskDataVerifier)_verifierLookup.get(ALL_DATA_VERIFIER);
	
	if (allDataVerifier == null) {
	    // Call each of the TaskDataVerifiers, in order.
	    callNextVerifier(_verifiers, 0, browseFlag, context, listener);
	} else {
	    // Call the ALL_DATA_VERIFIER
	    allDataVerifier.dataOK(browseFlag, context, listener);
	}
    }
    
    /**
     * Call the next verifier in a vector of TaskDataVerifiers.  On success,
     * this method will be invoked recursively to call the next verifier in
     * the list. Otherwise listener.failed() will be called.  If there
     * are no more verifiers in the list, then listener.succeeded()
     * will be called.
     *
     * @param verifierList A vector of TaskDataVerifiers to call, in order.
     * @param verifierNumber The verifier number to call.  If this verifier
     *                       number is past the end of the verifier list, it
     *                       will be assumed that all previous verifiers
     *                       succeeded.
     * @param browseFlag TaskDataVerifier.MAY_BE_EMPTY if null strings or zero
     *                   values are valid, TaskDataVerifier.MUST_BE_SET if a
     *                   valid value must be entered.
     * @param context An object the verifiers can use to behave differently
     *                in different contexts.
     * @param listener A ResultListener to be notified when the verification
     *                 succeeds or fails.
     */
    private void callNextVerifier(final Vector verifierList,
                                  final int verifierNumber,
                                  final int browseFlag, final Object context,
				  final ResultListener listener) {
	if (verifierNumber >= verifierList.size()) {
	    // All of the verifiers succeeded.  Inform the listener.
	    listener.succeeded(new ResultEvent(this));
	    return;
	}
	
	((TaskDataVerifier)verifierList.elementAt(verifierNumber))
	    .dataOK(browseFlag, context, new ResultListener() {
		public void succeeded(ResultEvent event) {
		    callNextVerifier(verifierList, verifierNumber + 1,
				     browseFlag, context, listener);
		}
		
		public void failed(ResultEvent event) {
			listener.failed(event);
		    }
	});
    }
}
