//
// TaskControlPanel.java
//
//	A user interface for driving a Task.
//	
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import javax.swing.border.*;
import java.awt.event.*;
import java.awt.*;
import java.util.*;

/**
 * TaskControlPanel is a user interface for driving a Task.  It allows users
 * to initiate a Task operation, cancel a Task, request help, toggle
 * between the Form and Guide interface (if both are present), and navigate
 * through the pages of the Guide interface (when it is present).  It
 * communicates with the Task via the TaskControlListener interface.
 */
public class TaskControlPanel extends JPanel {
    private ResourceStack _rs;
    private Vector _controlListeners = new Vector();
    private JButton _prevButton;
    private JButton _formGuideButton;
    private String _guideButtonText;
    private String _formButtonText;
    private JButton _okButton;
    private JButton _cancelButton;
    private JButton _helpButton;
    private JButton _nextButton;
    private int _minButtonSpacing;
    private int _buttonArrowSpacing;
    
    /**
     * The property <I>TaskControlPanel.guideButtonLabel</I> is the String that
     * is displayed on the interface toggle button when the button
     * will toggle the interface from Form mode to Guide mode (in
     * other words, when the Form is the current interface).  The
     * interface toggle button will only be displayed the Task has
     * both a Form and a Guide.
     */
    public static final String GUIDE_BUTTON_LABEL =
			    "TaskControlPanel.guideButtonLabel";

    /**
     * The property <I>TaskControlPanel.formButtonLabel</I> is a String that
     * is displayed on the interface toggle button when the button will
     * toggle the interface from Guide mode to Form mode (in other
     * words, when the Guide is the current interface).  The
     * interface toggle button will only be displayed the Task has
     * both a Form and a Guide.
     */
    public static final String FORM_BUTTON_LABEL = 
			    "TaskControlPanel.formButtonLabel";
			    
    /**
     * The property <I>TaskControlPanel.prevButtonLabel</I> is a String that
     * is displayed on the prev button when the Guide is displayed.
     */
    public static final String PREV_BUTTON_LABEL =
			    "TaskControlPanel.prevButtonLabel";
			    
    /**
     * The property <I>TaskControlPanel.nextButtonLabel</I> is a String that
     * is displayed on the next button when the Guide is displayed.
     */
    public static final String NEXT_BUTTON_LABEL =
			    "TaskControlPanel.nextButtonLabel";
			    
    /**
     * The property <I>TaskControlPanel.okButtonLabel</I> is a String that
     * is displayed on the OK button.
     */
    public static final String OK_BUTTON_LABEL =
			    "TaskControlPanel.okButtonLabel";

    /**
     * The property <I>TaskControlPanel.cancelButtonLabel</I> is a String that
     * is displayed on the cancel button.
     */
    public static final String CANCEL_BUTTON_LABEL =
			    "TaskControlPanel.cancelButtonLabel";

    /**
     * The property <I>TaskControlPanel.helpButtonLabel</I> is a String that
     * is displayed on the help button.
     */
    public static final String HELP_BUTTON_LABEL =
			    "TaskControlPanel.helpButtonLabel";

    /**
     * The property <I>TaskControlPanel.minButtonSpacing</I> is the
     * minimum distance, in points, between any two buttons in the
     * TaskControlPanel.
     */
    public static final String MIN_BUTTON_SPACING =
	"TaskControlPanel.minButtonSpacing";

    /**
     * The property <I>TaskControlPanel.buttonArrowSpacing</I> is the
     * spacing in points between the arrow icon and the text in the
     * "Next" and "Prev" buttons.
     */
    public static final String BUTTON_ARROW_SPACING =
        "TaskControlPanel.buttonArrowSpacing";

    /**
     * The property <I>TaskControlPanel.marginTop</I> is the distance,
     * in points, between the top of the control panel and its
     * contents.
     */
    public static final String MARGIN_TOP =
	"TaskControlPanel.marginTop";

    /**
     * The property <I>TaskControlPanel.marginLeft</I> is the distance,
     * in points, between the left side of the control panel and its
     * contents.
     */
    public static final String MARGIN_LEFT =
	"TaskControlPanel.marginLeft";

    /**
     * The property <I>TaskControlPanel.marginBottom</I> is the distance,
     * in points, between the bottom of the control panel and its
     * contents.
     */
    public static final String MARGIN_BOTTOM =
	"TaskControlPanel.marginBottom";

    /**
     * The property <I>TaskControlPanel.marginRight</I> is the distance,
     * in points, between the right side of the control panel and its
     * contents.
     */
    public static final String MARGIN_RIGHT =
	"TaskControlPanel.marginRight";

    /**
     * The property <I>TaskControlPanel.prevButtonKeyCode</I> is the
     * key code of the accelerator for the "Prev" button.
     */
    public static final String PREV_BUTTON_KEY_CODE =
        "TaskControlPanel.prevButtonKeyCode";

    /**
     * The property <I>TaskControlPanel.okButtonKeyCode</I> is the
     * key code of the accelerator for the "OK" button.
     */
    public static final String OK_BUTTON_KEY_CODE =
        "TaskControlPanel.okButtonKeyCode";

    /**
     * The property <I>TaskControlPanel.cancelButtonKeyCode</I> is the
     * key code of the accelerator for the "Cancel" button.
     */
    public static final String CANCEL_BUTTON_KEY_CODE =
        "TaskControlPanel.cancelButtonKeyCode";

    /**
     * The property <I>TaskControlPanel.nextButtonKeyCode</I> is the
     * key code of the accelerator for the "Next" button.
     */
    public static final String NEXT_BUTTON_KEY_CODE =
        "TaskControlPanel.nextButtonKeyCode";

    // Class used to notify a TaskControlListener of an event.
    private interface TaskControlNotifier {
	public void notify(TaskControlListener listener);
    }

    /**
     * Layout for buttons.
     */
    private class ButtonLayout implements LayoutManager {
	public void addLayoutComponent(String name, Component comp) {
	}
	public Dimension minimumLayoutSize(Container parent) {
	    Insets insets = parent.getInsets();
	    return new Dimension(insets.left + insets.right,
				 insets.top + insets.bottom);
	}
	public Dimension preferredLayoutSize(Container parent) {
	    Insets insets = parent.getInsets();
	    Dimension buttonSize = getButtonSize(parent);
	
	    int numButtons = _formGuideButton.isVisible() ? 6 : 5;

	    return new Dimension(insets.left + insets.right +
				 numButtons * (buttonSize.width +
					       _minButtonSpacing) -
				 _minButtonSpacing,
				 insets.top +
				 insets.bottom + buttonSize.height);
	}
	public void removeLayoutComponent(Component comp) {
	}
	public void layoutContainer(Container parent) {
	    Dimension parentSize = parent.getSize();
	    if (parentSize.width <= 0) {
		// Don't bother doing the layout if the parent isn't
		// a reasonable size yet.
		return;
	    }
	    Insets insets = parent.getInsets();
	    Dimension buttonSize = getButtonSize(parent);
	    _prevButton.setBounds(insets.left, + insets.top,
				  buttonSize.width, buttonSize.height);
	    _nextButton.setBounds(parentSize.width - insets.right -
				  buttonSize.width, insets.top,
				  buttonSize.width, buttonSize.height);
	    Component buttons[];
	    if (_formGuideButton.isVisible()) {
		buttons = new Component[] { _okButton, _cancelButton,
					    _helpButton, _formGuideButton };
	    } else {
		buttons = new Component[] { _okButton, _cancelButton,
					    _helpButton };
	    }

	    int buttonLeft = (parentSize.width
		- buttons.length * (buttonSize.width + _minButtonSpacing) 
		- _minButtonSpacing) / 2;
	    
	    for (int ii = 0; ii < buttons.length; ii++) {
		buttons[ii].setBounds(buttonLeft, insets.top ,
				      buttonSize.width, buttonSize.height);
		buttonLeft += buttonSize.width + _minButtonSpacing;
	    }
	}
    };
    
    /**
     * ButtonIcon is the base class for NextIcon and PrevIcon, which
     * draw an arrow and some text for the "Next" and "Prev" buttons.
     * This represents shared code between the "Next" and "Prev"
     * button rendering.
     */
    private abstract class ButtonIcon implements Icon {
	String _text;
	ArrowIcon _icon;
	Dimension _size = null;
	int _textWidth;
	int _textYOff;
	int _iconYOff;
	ButtonIcon(String text, ArrowIcon icon) {
	    _text = text;
	    _icon = icon;
	}
	void setSize(FontMetrics fm) {
	    int height = fm.getAscent();
	    _icon.setIconHeight(height);
	    _icon.setIconWidth(height);
	    _textWidth = fm.stringWidth(_text);
	    _textYOff = fm.getAscent();
	    _size = new Dimension(_textWidth + height + _buttonArrowSpacing,
				  fm.getHeight());
	    _iconYOff = (_size.height - height + 1) / 2;
	}
	public int getIconWidth() {
	    return _size.width;
	}
	public int getIconHeight() {
	    return _size.height;
	}
	public void paintIcon(Component c, Graphics g, int x, int y) {
	    Color color = c.isEnabled() ? c.getForeground()
		: UIManager.getColor("Button.disabledText");
	    g.setColor(color);
	    drawIt(c, g, x, y);
	}
	protected abstract void drawIt(Component c, Graphics g,
				       int x, int y);
    }

    /**
     * PrevIcon draws an arrow pointing left and a label, for use on
     * the "Prev" button.
     */
    private class PrevIcon extends ButtonIcon {
	PrevIcon(ResourceStack rs) {
	    super(rs.getString(PREV_BUTTON_LABEL),
		  new ArrowIcon(rs, ArrowIcon.LEFT));
	}
	public void drawIt(Component c, Graphics g, int x, int y) {
	    g.drawString(_text,
			 x + _icon.getIconWidth() + _buttonArrowSpacing,
			 y + _textYOff);
	    _icon.paintIcon(c, g, x, y + _iconYOff);
	}
    }

    /**
     * NextIcon draws some text and an arrow pointing right, for use
     * on the "Next" button.
     */
    private class NextIcon extends ButtonIcon {
	NextIcon(ResourceStack rs) {
	    super(rs.getString(NEXT_BUTTON_LABEL),
		  new ArrowIcon(rs, ArrowIcon.RIGHT));
	}
	public void drawIt(Component c, Graphics g, int x, int y) {
	    g.drawString(_text, x, y + _textYOff);
	    _icon.paintIcon(c, g, x + _textWidth + _buttonArrowSpacing,
			    y + _iconYOff);
	}
    }

    /**
     * ArrowButton knows how to properly deal with a ButtonIcon to
     * implement the "Next" and "Prev" buttons.
     */
    private class ArrowButton extends JButton {
	ButtonIcon _icon;
	ArrowButton(ButtonIcon icon) {
	    super(icon);
	    _icon = icon;
	}
	public Dimension getPreferredSize() {
	    _icon.setSize(getGraphics().getFontMetrics(getFont()));
	    return super.getPreferredSize();
	}
    }
    /**
     * Constructor.
     * <p>
     * All components are created here.  TaskControlPanel is
     * visible by default.
     *
     * @param rs ResourceStack that defines TaskControlPanel properties.
     */
    public TaskControlPanel(ResourceStack rs) {
	_rs = rs;
	
	setLayout(new ButtonLayout());
	setBorder(new EmptyBorder(_rs.getPixels(MARGIN_TOP),
				  _rs.getPixels(MARGIN_LEFT),
				  _rs.getPixels(MARGIN_BOTTOM),
				  _rs.getPixels(MARGIN_RIGHT)));
	
	_minButtonSpacing = _rs.getPixels(MIN_BUTTON_SPACING);
	_buttonArrowSpacing = _rs.getPixels(BUTTON_ARROW_SPACING);
	_guideButtonText = _rs.getString(GUIDE_BUTTON_LABEL);
	_formButtonText = _rs.getString(FORM_BUTTON_LABEL);
	
	_prevButton = new ArrowButton(new PrevIcon(_rs));
	_prevButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		notifyControlListeners(new TaskControlNotifier() {
		    public void notify(TaskControlListener listener) {
			listener.taskGoToPage(new GuideNavigationEvent(
				TaskControlPanel.this,
				GuideNavigationEvent.PREV_PAGE));
		    }
		});
	    }
	});
	addKeyCodeShortCut(_prevButton, PREV_BUTTON_KEY_CODE);
	_prevButton.setEnabled(false);
	
	_okButton =
	    new JButton(_rs.getString(OK_BUTTON_LABEL));
	_okButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		notifyControlListeners(new TaskControlNotifier() {
		    public void notify(TaskControlListener listener) {
			listener.taskOK();
		    }
		});
	    }
	});
	addKeyCodeShortCut(_okButton, OK_BUTTON_KEY_CODE);
	
	_cancelButton =
	    new JButton(_rs.getString(CANCEL_BUTTON_LABEL));
	_cancelButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		notifyControlListeners(new TaskControlNotifier() {
		    public void notify(TaskControlListener listener) {
			listener.taskCancel();
		    }
		});
	    }
	});
	addKeyCodeShortCut(_cancelButton, CANCEL_BUTTON_KEY_CODE);

	_helpButton =
	    new JButton(_rs.getString(HELP_BUTTON_LABEL));
	_helpButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		notifyControlListeners(new TaskControlNotifier() {
		    public void notify(TaskControlListener listener) {
			listener.taskHelp();
		    }
		});
	    }
	});
	
	_formGuideButton = new JButton("");
	_formGuideButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		notifyControlListeners(new TaskControlNotifier() {
		    public void notify(TaskControlListener listener) {
			listener.taskToggleInterface();
		    }
		});
	    }
	});
	
	_nextButton = new ArrowButton(new NextIcon(_rs));
	_nextButton.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent event) {
		notifyControlListeners(new TaskControlNotifier() {
		    public void notify(TaskControlListener listener) {
			listener.taskGoToPage(new GuideNavigationEvent(
				TaskControlPanel.this,
				GuideNavigationEvent.NEXT_PAGE));
		    }
		});
	    }
	});
	addKeyCodeShortCut(_nextButton, NEXT_BUTTON_KEY_CODE);
	
	add(_prevButton, null);
	add(_okButton, null);
	add(_cancelButton, null);
	add(_helpButton, null);
	add(_formGuideButton, null);
	add(_nextButton, null);
    }
    
    /**
     * Register interest in TaskControlPanel events.
     *
     * @param listener Object interested in receiving events.
     */
    public void addControlListener(TaskControlListener listener) {
	_controlListeners.addElement(listener);
    }
    
    /**
     * Unregister interest in TaskControlPanel events.
     *
     * @param listener Object no longer interested in receiving events.
     */
    public void removeControlListener(TaskControlListener listener) {
	_controlListeners.removeElement(listener);
    }
    
    /**
     * Tell the control panel that it should go into Form control mode.
     * Guide navigation features will be hidden.
     * <p>
     * If the Task has only a Form, the Task base class will call 
     * TaskControlPanel.setInterfaceToggleEnabled(false) to disable
     * the Form/Guide toggle components.  Those components are enabled
     * by default.
     */
    public void setFormMode() {
	_prevButton.setVisible(false);
	_nextButton.setVisible(false);
	_formGuideButton.setText(_guideButtonText);
	validate();
    }

    /**
     * Tell the control panel to go into Guide control mode.
     * Guide navigation features will be displayed.
     * <p>
     * If the Task has only a Guide, the Task base class will call 
     * TaskControlPanel.setInterfaceToggleEnabled(false) to disable
     * the Form/Guide toggle components.  Those components are enabled
     * by default.
     */
    public void setGuideMode() {
	_prevButton.setVisible(true);
	_nextButton.setVisible(true);
	_formGuideButton.setText(_formButtonText);
	validate();
    }
    
    /**
     * Enable or disable the Prev components.
     *
     * @param enable true to enable the Prev components,
     *               false to disable them.
     */ 
    public void setPrevEnabled(boolean enable) {
	_prevButton.setEnabled(enable);
    }
    
    /**
     * Enable or disable the Next components.
     *
     * @param enable true to enable the Next components,
     *		     false to disable them.
     */
    public void setNextEnabled(boolean enable) {
	_nextButton.setEnabled(enable);
    }
    
    /**
     * Enable or disable the OK components.
     *
     * @param enable true to enable the OK components,
     *		     false to disable them.
     */
    public void setOKEnabled(boolean enable) {
	_okButton.setEnabled(enable);
    }
    
    /**
     * Enable or disable the Form/Guide toggle components
     *
     * @param enable true to enable the Form/Guide toggle components,
     *               false to disable them.
     */
    public void setInterfaceToggleEnabled(boolean enable) {
	_formGuideButton.setVisible(enable);
	_formGuideButton.setEnabled(enable);
    }

    /**
     * Make the cancel button visible and enabled or invisible and disabled.
     *
     * The cancel button defaults to being visible and enabled.
     * 
     * @param enable true to show the cancel button and enable it,
     *               false to hide the cancel button and disable it.
     *  
     */
    public void setCancelButtonEnabled(boolean enable) {
	_cancelButton.setVisible(enable);
	_cancelButton.setEnabled(enable);
    }
    

    /**
     * Notify all of the TaskControlListeners of a particular event.
     *
     * @param notifier Notifier that knows what event has occurred.
     */
    private void notifyControlListeners(TaskControlNotifier notifier) {
	Enumeration enum = _controlListeners.elements();
	while (enum.hasMoreElements()) {
	    TaskControlListener listener =
		(TaskControlListener)enum.nextElement();
	    notifier.notify(listener);
	}
    }

    /**
     * Compute the maximum width and height of buttons in this
     * panel.
     * 
     * @param parent panel to get components from.
     * 
     * @return maximum width & height of buttons in this panel.
     */
    private Dimension getButtonSize(Container parent) {
	Component kids[] = parent.getComponents();
	Dimension size = new Dimension(0, 0);
	for (int ii = 0; ii < kids.length; ii++) {
	    Dimension prefSize = kids[ii].getPreferredSize();
	    if (prefSize.width > size.width) {
		size.width = prefSize.width;
	    }
	    if (prefSize.height > size.height) {
		size.height = prefSize.height;
	    }
	}
	return size;
    }

    /**
     * Add an accelerator to a button.
     * 
     * @param button Button to add accelerator to.
     * @param keyCodeResource Specifies a keyCode resource.
     */
    private void addKeyCodeShortCut(JButton button, String keyCodeResource) {
	try {
	    UIContext.addKeyCodeShortCut(button, _rs.getInt(keyCodeResource));
	} catch (MissingResourceException ex) {
	}
    }
}
