//
// TaskFrame.java
//
//	Container for displaying tasks.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import java.awt.event.*;
import java.text.*;
import java.util.*;

/**
 * TaskFrame is a container for displaying tasks.  TaskFrame is
 * intended for Task clients that want to a task in a separate,
 * standalone window.  The TaskFrame may be reused to show a set of
 * Tasks in sequence.
 * <P>
 * By default, TaskFrame will post a result panel when the Task completes
 * successfully.  This behavior can be modified by calling
 * TaskFrame.setResultHandling(). If result handling is set to
 * TaskFrame.POST_RESULT_PANEL (the default), TaskFrame clients will
 * be notified of successful task completion after the posted result
 * panel has been dismissed by the user.
 * <P>
 * TaskFrame clients should not call Task.addTaskDoneListener(), but should
 * instead call TaskFrame.addTaskDoneListener() so they can be certain that
 * the TaskFrame has a chance to respond to Task completion.
 */
public class TaskFrame extends RFrame
                       implements TitleListener, TaskDoneListener {

    private Task _task;
    private UIContext _uic;
    private int _resultHandling = POST_RESULT_PANEL;
    private Vector _doneListeners = new Vector();

    /**
     * If the <TT>resultHandling</TT> parameter to setResultHandling() is set
     * to TaskFrame.POST_RESULT_PANEL, then the TaskFrame will display a result
     * panel if the task succeeds.
     */
    public static final int POST_RESULT_PANEL = 0;

    /**
     * If the <TT>resultHandling</TT> parameter to setResultHandling() is set
     * to TaskFrame.POST_RESULT_PANEL, then the TaskFrame will not display
     * a result panel if the task succeeds.
     */
    public static final int DONT_POST_RESULT_PANEL = 1;

    /**
     * Constructor.
     */
    public TaskFrame() {
	setDefaultCloseOperation(DISPOSE_ON_CLOSE);
	// Listen for frame being closed by the window manager.
	addWindowListener(new WindowAdapter() {
	    public void windowClosing(WindowEvent event) {
		Log.assert(_task != null, "Task is null");
		_task.taskDone(new TaskResult(_task, TaskResult.CANCELLED));
	    }
	});
    }

    /**
     * Constructor that includes task to display.
     *
     * @param task The task to display and listen to.
     */
    public TaskFrame(Task task) {
	this();
	setTask(task);
    }

    /**
     * Set or replace the Task displayed in this frame.
     *
     * @param task Task to display in this frame.
     */
    public void setTask(Task task) {
	if (_task != null) {
	    _task.removeTitleListener(this);
	    _task.removeTaskDoneListener(this);
	}

	_task = task;
	_uic = task.getUIContext();

	setContentPane(_task);

	// Keep the TaskFrame title in sync with the Task title
	_task.addTitleListener(this);

	// Listen for Task completion
	_task.addTaskDoneListener(this);
    }

    /**
     * Set the result handling method for the TaskFrame.
     *
     * @param resultHandling Set to POST_RESULT_PANEL if you want
     *                       the TaskFrame to handle task completion, otherwise
     *                       set to DONT_POST_RESULT_PANEL.
     *
     * @see #POST_RESULT_PANEL
     * @see #DONT_POST_RESULT_PANEL
     */
    public void setResultHandling(int resultHandling) {
	_resultHandling = resultHandling;
    }

    /**
     * Register interest in task completion.
     *
     * @param listener Listener interested in task completion.
     */
    public void addTaskDoneListener(TaskDoneListener listener) {
	_doneListeners.addElement(listener);
    }

    /**
     * Unregister interest in task completion.
     *
     * @param listener Listener no longer interested in task completion.
     */
    public void removeTaskDoneListener(TaskDoneListener listener) {
	_doneListeners.removeElement(listener);
    }

    /**
     * Notify TaskDoneListeners of task completion.  Derived classes which
     * override taskDone() should call this method.
     *
     * @param result Result of task completion.
     */
    protected void notifyDoneListeners(TaskResult result) {
	Enumeration enum = _doneListeners.elements();
	while (enum.hasMoreElements()) {
	    TaskDoneListener listener = (TaskDoneListener)enum.nextElement();
	    listener.taskDone(result);
	}
    }

// -------------- TitleListener methods --------------------

    /**
     * Notify TitleListeners that the title changed.
     *
     * @param event A TitleEvent containing the new title.
     *
     * @see com.sgi.sysadm.ui.event.TitleListener#titleChanged(TitleEvent)
     */
    public void titleChanged(TitleEvent event) {
	setTitle(event.getTitle());
    }

// -------------- TaskDoneListener methods --------------------

    /**
     * Notify TaskDoneListeners that the Task completed successfully
     * or was cancelled by the User.  If the Task was successful, and
     * the TaskFrame's result handling is POST_RESULT_PANEL, and
     * Task.getResultViewPanel() returns a non-null ResultViewPanel,
     * this will pass the ResultViewPanel to
     * ResultViewFrame.launchResultViewFrame().
     *
     * @param result A TaskResult containing information about Task
     *               completion.
     *
     * @see com.sgi.sysadm.ui.event.TaskDoneListener#taskDone(TaskResult)
     */
    public void taskDone(final TaskResult result) {
	ResourceStack rs = _uic.getResourceStack();

	switch (result.getResultType()) {
	case TaskResult.SUCCEEDED:
	    ResultViewPanel rv;
	    if ((_resultHandling == POST_RESULT_PANEL) &&
                ((rv = _task.getResultViewPanel()) != null)) {
		ResultViewFrame.launchResultViewFrame(
		    rv,
		    _task.getTaskContext().getTitleString(),
		    rs,
		    _uic);
		notifyDoneListeners(result);
		return;
	    }

	    // else fall through

	default:
	    notifyDoneListeners(result);
	    break;
	}
    }

    /**
     * A static method to launch a Task in a new TaskFrame.
     *
     * @param tlre The TaskLauchRequestEvent that specifies the Task
     *              to launch
     * @param uic The UIContext to use when displaying the Task
     * @param resultHandling pass POST_RESULT_PANEL, or
     *                       DONT_POST_RESULT_PANEL, as appropriate.
     * @param listener A ResultListener that will be notified of the
     *                 Task when it is created.  Cast the result in
     *                 ResultEvent to a Task.  If the Task fails to
     *                 launch, listener.failed() will be called.
     *
     * @see #POST_RESULT_PANEL
     * @see #DONT_POST_RESULT_PANEL
     */
    public static void launchTaskFrame(final TaskLaunchRequestEvent tlre,
				       final UIContext uic,
				       final int resultHandling,
				       final ResultListener listener) {
	final TaskLoader loader = tlre.getTaskLoader();
	loader.setDialogParent(uic.getDialogParent());
	uic.busy();
	loader.loadTask(tlre.getTaskData(), tlre.getOperands(),
			new ResultListener() {
	    public void succeeded(ResultEvent event) {
		final Task task = (Task)event.getResult();
		if (listener != null) {
		    listener.succeeded(new ResultEvent(new Object(), task));
		}

		// Run-once logic: We create a key based on the task
		// name and the values of Attributes with keys in the
		// Task.RUN_ONCE_ATTRIBUTES property.  If HostContext
		// already knows about an instance with this key, use
		// it.
		TaskContext tc = task.getTaskContext();
		TaskData td = tc.getTaskData();
		StringBuffer keyBuf = new StringBuffer(
		    loader.getFullTaskName());

		try {
		    String attrs[] = tc.getResourceStack()
			.getStringArray(Task.RUN_ONCE_ATTRIBUTES);
		    if (attrs != null) {
			for (int ii = 0; ii < attrs.length; ii++) {
			    keyBuf.append("." + td.getAttr(attrs[ii])
					  .getValueString());
			}
		    }
		} catch (MissingResourceException ex) {
		}

		String key = keyBuf.toString();

		HostContext hc = loader.getHostContext();
		Vector vec = hc.getClients(key);
		if (vec.size() > 0) {
		    ((HostContext.Client)
		        vec.elementAt(0)).getFrame().toFront();
		    uic.notBusy();
		    return;
		}
		final TaskFrame tf = new TaskFrame(task);
		tf.setResultHandling(resultHandling);
		final HostContext.Client taskClient =
		    new HostContext.Client(tf, task.getUIContext());
		hc.registerClient(taskClient, key);
		tf.addTaskDoneListener(new TaskDoneListener() {
		    public void taskDone(TaskResult result) {
			tf.dispose();
			task.getUIContext().getHostContext().
			    unregisterClient(taskClient);
		    }
		});
		tf.pack();
		tf.setVisible(true);
		uic.notBusy();
	    }

	    public void failed(ResultEvent event) {
		uic.notBusy();
		if (listener != null) {
		    listener.failed(event);
		}
	    }
	});
    }

    /**
     * A static method to launch a Task in a new TaskFrame.
     *
     * @param event The TaskLauchRequestEvent that specifies the Task
     *              to launch
     * @param uic The UIContext to use when displaying the Task
     * @param resultHandling pass POST_RESULT_PANEL, or
     *                       DONT_POST_RESULT_PANEL, as appropriate.
     * @see #POST_RESULT_PANEL
     * @see #DONT_POST_RESULT_PANEL
     */
    public static void launchTaskFrame(TaskLaunchRequestEvent event,
				       final UIContext uic,
				       final int resultHandling) {
	launchTaskFrame(event, uic, resultHandling, new ResultListener() {
	    public void succeeded(ResultEvent event) {
		// No response is needed.
	    }

	    public void failed(ResultEvent event) {
		String reason = event.getReason();
		// If reason is null, it means we don't have to post
		// the message.  This happens when the user presses
		// "Cancel" on the privilege dialog.
		if (reason != null) {
		    uic.postError(event.getReason());
		}
	    }
	});
    }
}
