//
// TaskLaunchComponent.java
//
//      A component that can launch another task.
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import javax.swing.*;
import javax.swing.border.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.util.*;
import java.util.*;
import java.awt.event.*;
import java.awt.*;

/**
 * A component containing a Task link that the user can click to launch
 * another Task.  These components can be combined into a "metatask"
 * or "taskset."  A metatask is a set of Tasks that can be executed in
 * a programmer-controlled way, such as in a particular order (ex.,
 * Task <I>A</I> must be completed before Tasks <I>B</I> and <I>C</I>
 * become launchable, or Item <I>Foo</I> must exist before Task
 * <I>Bar</I> becomes launchable).  Metatasks can be implemented as
 * just another extension of the Task class.
 * <p>
 * The TaskLaunchComponent appears visually as three lines of text.
 * The first
 * line, the "description" line, describes the Task to be
 * launched as a human readable string.  The text comes from the
 * ResourceStack that is passed to
 * the constructor.  This line is a RichTextArea, so it can contain
 * glossary links.  The name of the RichTextArea that is created is
 * <tt>&lt;name&gt;.description</tt>, where &lt;name&gt is the
 * TaskLaunchComponent's <tt>name</tt>
 * that is passed to the constructor.  To set the text that is
 * displayed for the description, you would provide a resource with the
 * name <i>&lt;name&gt;.description.text</i>.  See the
 * RichTextArea about other resources that effect the look of the
 * RichTextArea.
 * <p>
 * The second line contains a Task Icon and a link to launch the Task.
 * The icon and the name of the Task are determined automatically from
 * the TaskLoader that is passed to the constructor.
 * <p>
 * The third line is the status of the Task.   The TaskLaunchComponent
 * automatically updates the status when the component is first shown,
 * when the Task is launched, when the Task is canceled, and when the
 * Task is completed.  The strings that are displayed come from the
 * ResourceStack passed to the constructor, and are described in
 * NEVER_LAUNCHED, LAUNCHED, CANCELED, and COMPLETED.  The status line can also
 * be set programatically with <tt>setStatus</tt>.  You can disable the
 * automatic setting of the status field with
 * the <tt>enableAutoStatus</tt> method.  The status text is
 * preceded by a string that says something like "Status: ".  The exact
 * text displayed is determined by the STATUS_PREFIX resource. The
 * name of the RichTextComponent that is created is described by
 * STATUS, and you can use that name to affect
 * the way the the RichTextComponent appears (see RichTextComponent
 * for more information).
 * <p>
 * When a user clicks on the Task link or icon, the
 * TaskLaunchComponent automatically creates a new TaskFrame window for the
 * Task and displays it.  When the task completes, the Task's usual
 * ResultView is
 * not shown.  Instead, a dialog with the text described in the
 * SUCCESS resource is displayed.  If
 * SUCCESS is not defined, then a more general resource,
 * TASK_COMPLETED_MESSAGE, is used.
 * <p>
 * There is a border inside the TaskLaunchComponent. Its size can be
 * adjusted with the TOP, BOTTOM, LEFT, and RIGHT resources.
 * <p>
 * The TaskLaunchComponent looks different when it is enabled versus
 * when it is disabled.  When enabled, the Task link becomes active,
 * and a special colored border is placed around the component.  The border's
 * size is controlled via the ENABLED_WIDTH resource, and the color
 * comes from a UIDefault called "TaskLaunchComponent.highlight".  See
 * the documention for RApp on how to set UIDefaults.
 *
 * @see Task
 * @see RApp
 * @see RichTextComponent
 * @see RichTextArea
 */
public class TaskLaunchComponent extends OneColumnPanel {

    /**
     * A resource <i>TaskLaunchComponent.neverLaunched</i> is the
     * string to display for the status if the Task has not yet been
     * launched.
     */
    public static final String NEVER_LAUNCHED =
        "TaskLaunchComponent.neverLaunched";
    
    /**
     * A resource <i>TaskLaunchComponent.launched</i> is the
     * string to display for the status if the Task is currently
     * launched.
     */
    public static final String LAUNCHED =
        "TaskLaunchComponent.launched";

    /**
     * A resource <i>TaskLaunchComponent.canceled</i> is the
     * string to display for the status if the Task has been canceled.
     */
    public static final String CANCELED =
        "TaskLaunchComponent.canceled";

    /**
     * A resource <i>TaskLaunchComponent.completed</i> is the
     * string to display for the status if the Task has been
     * successfully completed.
     */
    public static final String COMPLETED =
        "TaskLaunchComponent.completed";

    /**
     * A resource <I>TaskLaunchComponent.statusPrefix</i> is the
     * String to prepend to the status field.
     */
    public static final String STATUS_PREFIX = 
        "TaskLaunchComponent.statusPrefix";

    /**
     * The resource <I>TaskLaunchComponent.taskCompletedMessage</i> is
     * a String that is displayed in a dialog whenever a task
     * successfully completes, if the SUCCESS resource is not found.
     * 
     * @see SUCCESS
     */
    public static final String TASK_COMPLETED_MESSAGE =
        "TaskLaunchComponent.taskCompletedMessage";

    /**
     * The resource <i>&lt;name&gt;.success</i> is message that will
     * be displayed in the status line when the task sucessfully
     * completes.  If this resource is not defined, then the
     * TASK_COMPLETED_MESSAGE resource will be used instead.
     *
     * @see TASK_COMPLETED_MESSAGE
     */
    public static final String SUCCESS = ".success";

    /**
     * The string <tt>&lt;name&gt;.description</tt> is used to name
     * the RichTextArea used to display the description.  This is not
     * a resource key, but the <b>name</b> of the RichTextArea. You can use
     * this name to set properties that control the RichTextArea, as
     * described in RichTextArea.
     *
     * @see RichTextArea
     */
    public static final String DESCRIPTION = ".description";

    /**
     * The string <tt>&lt;name&gt;.status</tt> is used to name
     * the RichTextComponent used to display the status.   This is not
     * a resource key, but the <b>name</b> of the
     * RichTextComponent. You can use
     * this name to set properties that control the RichTextComponent, as
     * described in RichTextComponent.
     *
     * @see RichTextComponent
     */
    public static final String STATUS = ".status";
    
    /**
     * The resource <I>TaskLaunchComponent.margin.top</I> gives the
     * number of points to use for the top of the border inside the
     * component.
     */
    public static final String TOP = 
        "TaskLaunchComponent.margin.top";
    
     /**
     * The resource <I>TaskLaunchComponent.margin.bottom</I> gives the
     * number of points to use for the bottom of the margin inside the
     * component.
     */
    public static final String BOTTOM =
        "TaskLaunchComponent.margin.bottom";
    
    /**
     * The resource <I>TaskLaunchComponent.margin.left</I> gives the
     * number of points to use for the left of the margin inside the
     * component.
     */
    public static final String LEFT =
        "TaskLaunchComponent.margin.left";
    
    /**
     * The resource <I>TaskLaunchComponent.margin.right</I> gives the
     * number of points to use for the right of the margin inside the
     * component.
     */
    public static final String RIGHT = 
        "TaskLaunchComponent.margin.right";
    /**
     * The resource <I>TaskLaunchComponent.border.enabledWidth</i> gives
     * the width (in points) to use for the colored margin that highlights the
     * component when it is enabled.
     */
    public static final String ENABLED_WIDTH = 
        "TaskLaunchComponent.border.enabledWidth";
    private TaskLoader _loader;
    private ResourceStack _rs;
    private UIContext _uic;
    private Vector _inTD = new Vector();
    private Vector _outTD = new Vector();
    private TaskData _td;
    private static final int PARENT = 0;
    private static final int CHILD = 1;
    private Vector _listeners = new Vector();
    private TaskLaunchComponentState _state = 
        new TaskLaunchComponentState(TaskLaunchComponentState.NEVER_LAUNCHED);
    private RichTextComponent _status;
    private boolean _autoStatus = true;
    private RichTextArea _text;
    private RichTextComponent _link;
    private String _enabledLink;
    private String _disabledLink;
    private String _neverLaunchedString;
    private String _launchedString;
    private String _canceledString;
    private String _completedString;
    private String _statusPrefix;
    private String _successString;
    private Border _enabledBorder;
    private Border _disabledBorder;
    private static String CLASS_NAME = "TaskLaunchComponent";
    private Component _glassPane;
    private Task _currentTask;
    
    /**
     * Create a TaskLaunchComponent.
     *
     * @param uic The UIContext.  This component will call blockInput() on
     *            the UIContext while a task is launched, and will use
     *            the context to post error messages.
     * @param loader The task loader of the task to launch.
     * @param parentTD The TaskData of the parent Task (metatask).  Will 
     *                 be used to move TaskData Attributes between the
     *                 parent and child Tasks.  Can be null if there
     *                 is no parent Task, or if no TaskData Attributes
     *                 will be moved  between parent and child via the
     *                 <tt>addInTaskDataBinding</tt> or
     *                 <tt>addOutTaskDataBinding</tt> methods.
     * @param name The name of this TaskLaunchComponent.  Will be used
     *             <ul>
     *             <li>for naming the RichTextArea used for the
     *             description line (see DESCRIPTION)
     *             <li> for naming the RichTextComponent used for
     *             the status line (see STATUS)
     *             <li>for creating the property
     *             name to lookup when the task completed (see SUCCESS).
     *             </ul>
     * @param rs The ResourceStack to use to look up resources. 
     *
     * @see DESCRIPTION
     * @see STATUS
     * @see SUCCESS
     * @see addInTaskDataBinding
     * @see addOutTaskDataBinding
     */
    public TaskLaunchComponent(UIContext uic,
			       TaskLoader loader, 
			       TaskData parentTD,
			       String name, 
			       ResourceStack rs) {
				   
	super(rs, "TaskLaunchComponent");
	_uic = uic;
	_td = parentTD;
	_rs = rs;
	_loader = loader;
	_disabledBorder = 
	    BorderFactory.createCompoundBorder(
		BorderFactory.createCompoundBorder(
		    BorderFactory.createEtchedBorder(),
		    // The diasabled border must be the same size as
		    // the enabled border so
		    // that the components will not change size when
		    // they're enabled vs disabled.  The following
		    // empty border makes them the same.
		    BorderFactory.createEmptyBorder(2,2,2,2)),
		BorderFactory.createEmptyBorder( 
		    rs.getPixels(TOP),
		    rs.getPixels(LEFT),
		    rs.getPixels(BOTTOM),
		    rs.getPixels(RIGHT)));
	
	_enabledBorder = 
	    BorderFactory.createCompoundBorder(
		BorderFactory.createCompoundBorder(
		    BorderFactory.createBevelBorder(BevelBorder.RAISED),
		    BorderFactory.createLineBorder(
			UIManager.getLookAndFeelDefaults().getColor(
			    "TaskLaunchComponent.highlight"), 
			rs.getPixels(ENABLED_WIDTH))),
		BorderFactory.createEmptyBorder( 
		    rs.getPixels(TOP),
		    rs.getPixels(LEFT),
		    rs.getPixels(BOTTOM),
		    rs.getPixels(RIGHT)));

	_successString = rs.getString(new String[] 
				      { name + SUCCESS,
					TASK_COMPLETED_MESSAGE
				      });
	_statusPrefix = rs.getString(STATUS_PREFIX);
	_enabledLink = "<A HREF=>" + _loader.getLongTaskName() + "</A>";
	_disabledLink =   _loader.getLongTaskName();
	_text = new RichTextArea(name + DESCRIPTION, rs);
	_text.setMargins(new Insets(0,0,0,0));
	addComponent(_text);
	_link = new RichTextComponent();
	_link.setMargins(new Insets(0,0,0,0));
	_link.setText(_enabledLink);
	final JLabel iconLabel = new JLabel(
	    _loader.getResourceStack().getIcon(Task.TASK_SHELF_ICON)); 
	iconLabel.addMouseListener(new MouseAdapter() {
	    int _currentLinkState;
	    
	    Cursor _cursor = iconLabel.getCursor();

	    public void mouseEntered(MouseEvent event) {
		if (isEnabled()) {
		    _cursor = iconLabel.getCursor();
		    iconLabel.setCursor(
			Cursor.getPredefinedCursor(Cursor.HAND_CURSOR));
		    _currentLinkState = _link.getLinkState(0);
		}
	    }

	    public void mouseExited(MouseEvent event) {
		if (isEnabled()) {
		    iconLabel.setCursor(_cursor);
		    _link.setLinkState(
			0, _currentLinkState);
		    _link.repaint();
		}
	    }
	    	
	    public void mousePressed(MouseEvent event) {
		if (isEnabled()) {
		    _link.setLinkState(
			0, RichTextComponent.LINK_STATE_ACTIVE);
		    _link.repaint();
		}
	    }
	    
	    public void mouseClicked(MouseEvent event) {
		if (isEnabled()) {
		    _link.setLinkState(
			0, RichTextComponent.LINK_STATE_VISITED);
		    _currentLinkState =
			RichTextComponent.LINK_STATE_VISITED;
		    _link.repaint();
		    launchTask();
		}
	    }
	    
	});
	JPanel panel = new JPanel();
	panel.add(iconLabel);
	panel.add(_link);
	addComponent(panel);
	_status = new RichTextComponent(name + ".status", rs);
	_status.setMargins(new Insets(0,0,0,0));
	addComponent(_status);
	handleStateChange();
	_link.addLinkListener(new LinkListener() {
	    public void linkActivated(LinkEvent e) {
		launchTask();
	    }
	});
	setEnabled(true); 	// to set the border
    }
    
    private void launchTask() {
	setGlassPane();
	_uic.blockInput(true);
	TaskData newTD = null;
	int numInAttrs = _inTD.size();
	if (numInAttrs != 0) {
	    newTD = new TaskData();
	    for (int ii = 0; ii < numInAttrs; ii++) {
		newTD.setAttr(new Attribute(
		    ((String[])_inTD.elementAt(ii))[CHILD],
		    _td.getAttr(
			((String[])_inTD.elementAt(ii))[PARENT]).
		    getValue()
		    ));
	    }
	}
	TaskLaunchRequestEvent event = 
	    new TaskLaunchRequestEvent(this, _loader, null, newTD);
	TaskFrame.launchTaskFrame(
	    event,
	    _uic,
	    TaskFrame.DONT_POST_RESULT_PANEL,
	    new ResultListener() {
	    public void succeeded(ResultEvent event) {
		final Task task = (Task)event.getResult();
		if (alreadyRunning(_loader, task)) {
		    _uic.blockInput(false);
		    _uic.postError(_rs.getString(
			"TaskLaunchComponent.Error.alreadyRunning"));
		} else {
		    _currentTask = task;
		    task.addTaskDoneListener(new myTaskDoneListener());
		    _state = new TaskLaunchComponentState(
			TaskLaunchComponentState.LAUNCHED);
		    handleStateChange();
		}
	    }
	    public void failed(ResultEvent event) {
		String reason = event.getReason();
		if (reason != null) {
		    _uic.blockInput(false);
		    _uic.postError(reason);
		}
	    }
	});
    }

    /**
     * Connects some parent's (ex., metatask) TaskData to the child's
     * TaskData. The parent TaskData is defined by the
     * <tt>parentTD</tt> that was passed to the constructor.  The
     * child's TaskData is the TaskData in the Task that this
     * TaskLaunchComponent creates.When the user launches the Task, the TaskData
     * specified by this call will be copied from the parent TaskData and
     * placed in the child. 
     * <p>
     * This method allows you to specify both the name the
     * TaskData Attribute has in the parent, as well as what name the
     * TaskData Attribute will have in the child.  This is useful if
     * you need to use a different name for the TaskData Attributes in
     * the two TaskDatas.  In many cases, though, both names
     * will be the same.
     * <p>
     * This method can be called as many times as necessary.
     *
     * @param parent The name of the TaskData Attribute to copy from the parent.
     * @param child The name to give the TaskData Attribute in the child.
     */
    public void addInTaskDataBinding(String parent, String child) {
	Log.assert(_td != null, "Must pass non-null TaskData " +
		   "to the constructor of TaskLaunchComponent if you want " +
		   "to call addInTaskDataBinding");
	String[] array = new String[2];
	array[PARENT] = parent;
	array[CHILD] = child;
	_inTD.addElement(array);
    }
    
    /**
     * Connects some child's TaskData to the parent's (ex., metatask)
     * TaskData.  The parent TaskData is
     * defined by the <tt>parentTD</tt> that was passed to the
     * constructor.  The child's TaskData is the TaskData in the Task
     * that this TaskLaunchComponent creates.
     * When the Task successfully completes, the TaskData
     * specifed by this call will be copied from the child's TaskData
     * and placed in the parent's TaskData.  
     * <p>
     * This method allows you to specify both the name the
     * TaskData Attribute has in the child, as well as what name the
     * TaskData Attribute will have in the parent.  This is useful if
     * you need to use a different name for the TaskData Attributes in
     * the two TaskDatas.  In many cases, though, both names
     * will  be the same.
     * <p>
     * This method can be called as many times as necessary.
     *
     * @param child The name of the task data to copy from the child.
     * @param parent The name to give the task data in the parent.
     */
    public void addOutTaskDataBinding(String child, String parent) {
	Log.assert(_td != null, "Must pass non-null TaskData " +
		   "to the constructor of TaskLaunchComponent if you want " +
		   "to call addOutTaskDataBinding");
	String[] array = new String[2];
	array[PARENT] = parent;
	array[CHILD] = child;
	_outTD.addElement(array);
    }

    /**
     * Adds a listener to the list of listeners that will be notified
     * when this TaskLaunchComponent changes state.
     *
     * @param listener The TaskLaunchComponentListener to add
     */
    public void addTaskLaunchComponentListener(
	TaskLaunchComponentListener listener) {
	_listeners.addElement(listener);
    }
    
    /**
     * Removes a listener from the list of listeners that will be notified
     * when this TaskLaunchComponent changes state.
     *
     * @param listener The TaskLaunchComponentListener to remove
     */
    public void removeTaskLaunchComponentListener(
	TaskLaunchComponentListener listener) {
	_listeners.removeElement(listener);
    }
    
    /**
     * Sets the status field of the component programatically.  The
     * component automatically shows some status: "Not launched",
     * "Launched", "Canceled", and "Finished" (in localized versions),
     * unless that feature has been disabled via the
     * <tt>enableAutoStatus</tt> method.
     * If auto status is disabled, then the <tt>status</tt> String that
     * is passed will be used until another call to
     * <tt>setStatus</tt>.  If auto status is enabled, then the
     * <tt>status</tt> String that is passed will be used until the
     * component changes status, when the TaskLaunchComponent will
     * automatically set the status to the new value.
     * <p>
     * It is often desirable to set the status of the
     * TaskLaunchComponent in response to being notified that the
     * state of the component has changed (via a
     * TaskLaunchComponentListener).  If auto status is enabled, then
     * the status field is set automatically before callbacks to
     * TaskLaunchComponentListeners occur, so listeners can call
     * setStatus without fear that the auto status will overwrite the
     * status they just passed to setStatus.
     * 
     * @param status The String to show in the status field.
     */
    public void setStatus(String status) {
	_status.setText(_statusPrefix + status);
	_status.invalidate();
	Component parent, grandparent;
	if ((parent = _status.getParent()) != null &&
	    (grandparent = parent.getParent()) != null) {
	    grandparent.validate();
	} else {
	    if (parent == null) {
		Log.error(
		    CLASS_NAME,
		    "Got a request to update status, but the status area\n" +
		    "is no longer in the TaskLaunchComponent container.\n" +
		    "This means that the component hierarchy has been\n" +
		    "destroyed, possibly by the RFrame.dispose method.\n" +
		    "Check for problems with a method\n" +
		    "calling setStatus at inappropriate times, such as\n" +
                    "after the RFrame that contained the\n" +
		    "TaskLaunchComponent was destroyed.\n" +
		    SysUtil.stackTraceToString(new Throwable()));
	    }
	}
    }

    /**
     * Sets this component to be enabled or disabled.
     * If the component is disabled, then the Task can't be launched.
     * If the component is enabled, then the Task can be launched, and
     * a border is drawn around the component.
     *
     * @param enabled True to enable, false to disable.
     *
     */
    public void setEnabled(boolean enabled) {
	_link.setText(enabled ? _enabledLink : _disabledLink);
	_link.setEnabled(enabled);
	_text.setEnabled(enabled);
	if (enabled) {
	    setBorder(_enabledBorder);
	} else {
	    setBorder(_disabledBorder);
	}
	super.setEnabled(enabled);
	validate();
	repaint();
    }

    /**
     * Handle a state change
     */
    private void handleStateChange() {
	
	TaskLaunchComponentEvent event = 
	    new TaskLaunchComponentEvent(this, _state);

	if (_status != null && _autoStatus) {
	    switch (_state.getTaskState()) {
	    case TaskLaunchComponentState.NEVER_LAUNCHED:
		if (_neverLaunchedString == null) {
		    _neverLaunchedString =
			_rs.getString(NEVER_LAUNCHED);
		}
		setStatus(_neverLaunchedString);
		break;
	    case TaskLaunchComponentState.LAUNCHED:
		if (_launchedString == null) {
		    _launchedString =
			_rs.getString(LAUNCHED);
		}
		setStatus(_launchedString);
		break;
	    case TaskLaunchComponentState.CANCELED:
		if (_canceledString == null) {
		    _canceledString =
			_rs.getString(CANCELED);
		}
		setStatus(_canceledString);
		break;
	    case TaskLaunchComponentState.FINISHED:
		if (_completedString == null) {
		    _completedString =
			_rs.getString(COMPLETED);
		}
		setStatus(_completedString);
		break;
	    }
	}
	if  (_state.getTaskState() == TaskLaunchComponentState.FINISHED) {
	    _uic.postInfo(_successString);
	}

	int n = _listeners.size();
	for (int ii = 0; ii < n; ii++) {
	    ((TaskLaunchComponentListener)_listeners.elementAt(ii)).
		taskLaunchComponentStateChanged(event);
	
	}
    }
    
    /**
     * Return the current state of the TaskLaunchComponent.
     *
     * @return a TaskLaunchComponentState object representing the
     *         current state.
     */
    public TaskLaunchComponentState getState() {
	return _state;
    }
    
    
    /**
     * Determines whether or not this component will automatically set
     * the status.  Defaults to true.  If auto status is enabled, then
     * the TaskLaunchComponent will set the status field to one of
     * four values (descibed in NEVER_LAUNCHED, LAUNCHED, CANCELED,
     * and COMPLETED).  If disabled, the TaskLaunchComponent will not
     * automatically set the state, and the <tt>setStatus</tt> method
     * should be used to set the status.
     *
     * @param enable Pass true to automatically set status, false to
     * not.
     * @see #NEVER_LAUNCHED
     * @see #LAUNCHED
     * @see #CANCELED
     * @see #COMPLETED
     */
    public void enableAutoStatus(boolean enable) {
	_autoStatus = enable;
    }
    
    /**
     * Sets the HTML that will be displayed for the description line.
     * If this method is not called, the TaskLaunchComponent will use
     * a default value that comes from the DESCRIPTION resource.
     *
     * @param description The HTML text to display as the
     * "description"
     */
    public void setDescription(String description) {
	_text.setText(description);
    }
    

    /**
     * The class that knows how to deal with a task completion.  Could
     * be an inner class, but this makes it more legible.
     */
    private class myTaskDoneListener implements TaskDoneListener {

	public void taskDone(TaskResult result) {
	    _currentTask = null;
	    _uic.blockInput(false);
	    if (result.getResultType() == TaskResult.SUCCEEDED) {
		int numOutAttrs = _outTD.size();
		for (int ii = 0; ii < numOutAttrs; ii++) {
		    String childKey =
			((String[])_outTD.elementAt(ii))[CHILD];
		    Attribute childAttr =
			result.getTask().getTaskDataAttr(childKey);
		    if (childAttr == null) {
			Log.fatal(
			    "Tried to get non-public task data attribute: " +
			    childKey);
		    }
		    _td.setAttr(new Attribute(
			((String[])_outTD.elementAt(ii))[PARENT],
			childAttr.getValue()));
		}
		
		_state = new TaskLaunchComponentState(
		    TaskLaunchComponentState.FINISHED);
		handleStateChange();
		
	    } else {
		_state = new TaskLaunchComponentState(
		    TaskLaunchComponentState.CANCELED);
		handleStateChange();
	    }
	}
    }
    
    /**
     * The first time this method is called, it sets up a listener on
     * our glass pane to pop our task forward when we're clicked on.
     *
     * This is here to workaround the fact that
     * TaskFrame.launchTaskFrame doesn't tell you which Frame is being
     * used; what we'd really like to be doing here is calling the
     * UIContext.blockInput() method that takes a topWindow.
     *
     * See bug 672153.
     */
    private void setGlassPane() {
	if (_glassPane != null) {
	    // Already been here.
	    return;
	}
	JRootPane root = getRootPane();
	if (root == null) {
	    return;
	}
	_glassPane = root.getGlassPane();
	if (_glassPane == null) {
	    return;
	}
	_glassPane.addMouseListener(new MouseAdapter() {
	    public void mouseClicked(MouseEvent event) {
		try {
		    // Only beep if we can bring the task to the
		    // front; otherwise, assume that UIContext beeping
		    // is going to handle everything for us.
		    ((Window)_currentTask.getTopLevelAncestor()).toFront();
		    Toolkit.getDefaultToolkit().beep();
		} catch (ClassCastException ex) {
		} catch (NullPointerException ex) {
		}
	    }
	});
    }

    /**
     * XXX Hack!!  See bug 672153.  This is a duplication of the
     * run-once logic in TaskFrame.
     * 
     * @param loader Task loader that loaded the task
     * @param task Task to check for run-once
     *
     * @return true if task is already running, false otherwise.
     */
    private boolean alreadyRunning(TaskLoader loader, Task task) {
	// Run-once logic: We create a key based on the task
	// name and the values of Attributes with keys in the
	// Task.RUN_ONCE_ATTRIBUTES property.  If HostContext
	// already knows about an instance with this key, use
	// it.
	TaskContext tc = task.getTaskContext();
	TaskData td = tc.getTaskData();
	StringBuffer keyBuf = new StringBuffer(
	    loader.getFullTaskName());

	try {
	    String attrs[] = tc.getResourceStack()
		.getStringArray(Task.RUN_ONCE_ATTRIBUTES);
	    if (attrs != null) {
		for (int ii = 0; ii < attrs.length; ii++) {
		    keyBuf.append("." + td.getAttr(attrs[ii])
				  .getValueString());
		}
	    }
	} catch (MissingResourceException ex) {
	}

	String key = keyBuf.toString();

	HostContext hc = loader.getHostContext();
	Vector vec = hc.getClients(key);
	if (vec.size() > 0) {
	    ((HostContext.Client)
	     vec.elementAt(0)).getFrame().toFront();
	    return true;
	}

	return false;
    }
}
