//
// TaskLoader.java
//
//	Loads Task properties and instantiates Tasks.
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import java.awt.Component;
import java.lang.reflect.*;
import java.text.*;
import java.util.*;
import javax.swing.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.ui.*;
import com.sgi.sysadm.ui.taskData.*;
import com.sgi.sysadm.util.*;

/**
 * TaskLoader is intended to be the primary mechanism for clients to query
 * and load Tasks.
 * <p>
 * Tasks are loaded in a two-step process to allow the Task to be queried
 * for static information, such as its privileges, without having to
 * instantiate the Task.
 * <p>
 * To support the two-step loading mechanism, Tasks need to define a
 * ResourceBundle named "{taskname}P.properties" and place it in the
 * same package that contains "{taskname}.class".
 * 
 * @see com.sgi.sysadm.ui.Task
 */
public class TaskLoader {

    private String _taskName;
    private HostContext _hc;
    private ResourceStack _rs;
    private String _testerClassName;
    private Constructor _testerConstructor;
    private JComponent _dialogParent;
    
    private static ResourceStack _rsProto;
    
    /**
     * Constructor.  Load the task properties.
     * 
     * @param hostContext The HostContext representing the server to
     *                    load the task from.
     * @param taskName The <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH 
     *                 relative</A> name of the task to load.
     *
     * @exception java.util.MissingResourceException Thrown if the task
     *            ResourceBundle is not found.
     */
    public TaskLoader(HostContext hostContext, String taskName) 
	    throws MissingResourceException {
	if (_rsProto == null) {
	    _rsProto = (new TaskContext(hostContext)).getResourceStack();
	}
	_taskName = taskName;
	_hc = hostContext;
	_rs = (ResourceStack)_rsProto.clone();
	loadTaskProperties();
    }

    /**
     * Get the HostContext associated with this TaskLoader.
     * 
     * @return The HostContext associated with this TaskLoader.
     */
    public HostContext getHostContext() {
	return _hc;
    }

    /**
     * Get the <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH
     * relative</A> name of the task associated with this loader.
     *
     * @return the <A HREF="glossary.html#CLASSPATHRelative">CLASSPATH
     *         relative</A> name of the task associated with this loader.
     */
    public String getFullTaskName() {
	return _taskName;
    }

    /**
     * Gets the long name (Task.LONG_NAME) of the task.
     * 
     * @return A String containing the long name of the task.
     */
    public String getLongTaskName() {
	return _rs.getString(Task.LONG_NAME);
    }

    /**
     * Get the ResourceStack associated with this task.
     *
     * @return the ResourceStack associated with this task.
     */
    public ResourceStack getResourceStack() {
	return _rs;
    }
    
    /**
     * Get an ItemTester for the task to be loaded.  This method first
     * searches for the property Task.ITEM_TESTER that describes the
     * ItemTester class to load.  If that property is not found, we
     * look for an ItemTester class called {taskname}Tester.class.  If
     * that class is not found, we return an ItemTester that always
     * returns true.
     *
     * @exception com.sgi.sysadm.ui.TaskLoaderException if the Task.ITEM_TESTER
     *            property is defined and specifies a non-existent class
     *            or if the tester class cannot be loaded or instantiated
     *            for some reason.
     *
     * @return An ItemTester for the task to be loaded.
     */
    public ItemTester getItemTester() throws TaskLoaderException {
	if (_testerConstructor != null) {
	    return instantiateItemTester();
	}
	
	String testerClassName = null;
	
	// First look for the Task.ITEM_TESTER property.
	try {
	    testerClassName = _rs.getString(Task.ITEM_TESTER);
	    
	    try {
		loadItemTester(testerClassName);
	    } catch (ClassNotFoundException exception) {
		// The property specified a non-existent class.
		throw new TaskLoaderException(MessageFormat.format(
			    _rs.getString("TaskLoader.Error.missingTesterClass"),
			    new Object[] { testerClassName }));
	    }
	    return instantiateItemTester();
	} catch (MissingResourceException exception) {
	    // The Task.ITEM_TESTER property is optional
	}
	
	// Now look for an ItemTester class called {taskname}Tester.class
	try {
	    loadItemTester(_taskName + "Tester");
	    return instantiateItemTester();
	} catch (ClassNotFoundException exception) {
	    // The {taskname}Tester class is optional
	}
	
	// Since no task-specific ItemTester exists, return an ItemTester
	// that always returns true
	return new ItemTester() {
	    public ItemTestResult testItem(Item item) {
		return ItemTestResult.TEST_PASSED;
	    }
	};
    }

    /**
     * Determine if the Task accepts the given operand type.
     *
     * @param operandType An Item selector representing the operand
     *                    type to check.
     *
     * @return true if <TT>operandType</TT> is accepted, false otherwise.
     */
    public boolean acceptsOperandType(String operandType) {
	String typeAccepted =
	    _rs.getString(Task.OPERAND_TYPE_ACCEPTED, null);
	return (Task.ALL_OPERAND_TYPES.equals(typeAccepted) ||
		operandType.equals(typeAccepted));
    }

    /**
     * Set the dialog parent for the next call to <tt>loadTask</tt>.
     * The privilege dialog will be posted over the dialog parent.
     * Error dialogs will be posted over the dialog parent if the Task
     * has not yet been displayed.
     * 
     * @param dialogParent Dialog parent for next call to
     *                     <tt>loadTask</tt>.
     */
    public void setDialogParent(JComponent dialogParent) {
	_dialogParent = dialogParent;
    }

    /**
     * Load the Task class, instantiate it, set the product attributes (if any)
     * and verify prerequisites.  Use this method if no operands will be
     * passed to the Task and no TaskData attributes will be set on the Task.
     * <P>
     * <TT>listener</TT>.succeeded() will be called if the Task loading process
     * is successful.  The <TT>listener</TT> may use ResultEvent.getResult()
     * to get a reference to the Task class that was loaded.
     * <P>
     * <TT>listener</TT>.failed() will be called if there is an error at any
     * stage of the Task loading process.  The <TT>listener</TT> may use
     * ResultEvent.getReason() to get a localized error message String suitable
     * for display to the user.
     *
     * @param listener The ResultListener that will be notified when the
     *                 loading, instantiation, and prerequisite verification
     *                 of the Task has succeeded or failed.
     */
    public void loadTask(ResultListener listener) {
	loadTask(null, null, listener);
    }

    /**
     * Load the Task class, instantiate it, set TaskData attributes, pass
     * operands, set the product attributes (if any) and verify prerequisites.
     * <P>
     * <TT>listener</TT>.succeeded() will be called if the Task loading process
     * is successful.  The <TT>listener</TT> may use ResultEvent.getResult()
     * to get a reference to the Task class that was loaded.
     * <P>
     * <TT>listener</TT>.failed() will be called if there is an error at any
     * stage of the Task loading process.  The <TT>listener</TT> may use
     * ResultEvent.getReason() to get a localized error message String suitable
     * for display to the user.
     *
     * @param taskData The set of TaskData attributes to set on the Task.  May
     *                 be empty or null if there are no attributes to set.
     * @param operands A Vector of Object to pass to the Task as operands.  May
     *                 be empty or null if there are no operands to pass.
     * @param listener The ResultListener that will be notified when the
     *                 loading, instantiation, and prerequisite verification
     *                 of the Task has succeeded or failed.
     */
    public void loadTask(final TaskData taskData, final Vector operands,
			 final ResultListener listener) {
	// Create a new TaskContext for the task to be loaded.
	final TaskContext taskContext = new TaskContext(_hc, _rs);
	if (_dialogParent != null) {
	    taskContext.setDialogParent(_dialogParent,
					new ResultListener() {
		public void succeeded(ResultEvent e) {
		    continueLoadingTask(taskData, operands, 
					listener, taskContext);
		}
		
		public void failed(ResultEvent e) {
		    // Won't get here
		}
	    });
	    _dialogParent = null;
	} else {
	    continueLoadingTask(taskData, operands, 
				listener, taskContext);
	}
    }
    
    private void continueLoadingTask(final TaskData taskData, 
				     final Vector operands,
				     final ResultListener listener,
				     final TaskContext taskContext) {
	// If any error messages are generated, fill in the task name.
	Object[] fillins =
	{
	    _taskName,
	};

	ResultEvent result = new ResultEvent(this);
	
	// Locate the task class.
	Class taskClass;
	try {
	    taskClass = SysUtil.forName(_taskName);
	} catch (ClassNotFoundException exception) {
	    result.setReason(MessageFormat.format(
		_rs.getString("TaskLoader.Error.noSuchTaskClass"), fillins));
	    listener.failed(result);
	    return;
	}
	
	// Get the class constructor
	Constructor taskConstructor = null;
	try {
	    taskConstructor =
		taskClass.getConstructor(new Class[] { TaskContext.class });
	} catch (NoSuchMethodException exception) {
	    result.setReason(MessageFormat.format(
		_rs.getString("TaskLoader.Error.missingTaskConstructor"),
		fillins));
	} catch (SecurityException exception) {
	    result.setReason(MessageFormat.format(
		_rs.getString("TaskLoader.Error.prohibitedTaskConstructor"),
		fillins));
	}
	
	if (taskConstructor == null) {
	    listener.failed(result);
	    return;
	}
	Task task = null;
	// Attempt to instantiate the task.
	try {
	    task = (Task)
		(taskConstructor.newInstance(new Object[] { taskContext }));
	} catch (InstantiationException exception) { 
	    result.setReason(MessageFormat.format(
		_rs.getString("TaskLoader.Error.cantInstantiateTask"),
		fillins));
	} catch (IllegalAccessException exception) {
	    result.setReason(MessageFormat.format(
		_rs.getString("TaskLoader.Error.illegalTaskConstructorAccess"),
		 fillins));
	} catch (IllegalArgumentException exception) {
	    result.setReason(MessageFormat.format(
		_rs.getString("TaskLoader.Error.illegalTaskArgument"),
		fillins));
	} catch (InvocationTargetException exception) {
	    exception.getTargetException().printStackTrace();
	    result.setReason(MessageFormat.format(
		_rs.getString("TaskLoader.Error.unableToInvokeTaskOnTarget"),
		fillins));
	}
	
	if (task == null) {
	    listener.failed(result);
	    return;
	}
	final Task taskCopy = task;
	// Tell the task context about this specific task
	taskContext.setTask(task);

	// Get the list of products for which we need attributes, if any.
	String[] products =
	    _rs.getStringArray(Task.PRODUCT_ATTRIBUTES, null);
	
	// Load the product attributes.
	loadProductAttributes(products, 0, taskContext.getTaskData(),
			      task, taskContext,
			      new ResultListener() {
			      	  public void succeeded(ResultEvent event) {
				      initTask(taskCopy, taskData, 
					       operands, listener);
				  }

				  public void failed(ResultEvent event) {
				      listener.failed(event);
				  }
	});
    }
    

    /**
     * Loads the productAttributes for the products specified
     *
     * @param products The list of products for which to load the
     *                 product Attributes
     * @param numProduct The index into the products array where this
     *                   method should start looking.  Pass 0
     * @param taskData The TaskData in which to place the product
     *                 attributes.
     * @param task The task to place in the successfull result
     * @param tc The UIContext to pass to hostContext.getProductAttributes
     * @param listener The listener to notify about the success of the
     *                 getProductAttributes() calls.
     */
    private void loadProductAttributes(final String[] products,
				       final int numProduct,
				       final TaskData taskData,
				       final Task task,
				       final TaskContext tc,
				       final ResultListener listener) {
	if (products == null || products.length == 0) {
	    listener.succeeded(new ResultEvent(this));
	    return;
	}

	_hc.getProductAttributes(
	    products[numProduct], taskData, tc,
	    HostContext.DONT_REFRESH_ATTRIBUTES,
	    null,
	    new ResultListener() {
	    public void succeeded(ResultEvent event) {
		if (numProduct == products.length-1) {
		    listener.succeeded(new ResultEvent(this));
		    return;
		} else {
		    loadProductAttributes(products, numProduct+1, taskData,
					  task, tc, listener);
		}
	    }

	    public void failed(ResultEvent event) {
		listener.failed(event);
	    }
	});
    }
    
    /**
     * Initialize the Task after product attributes have been loaded and
     * then verify the Task prerequisites.
     *
     * @param listener ResultListener to notify of successful or failed
     *                 initialization.
     */
    private void initTask(final Task task, TaskData taskData, Vector operands,
			  final ResultListener listener) {
	final ResultEvent result = new ResultEvent(this);

	try {
	    // Set the TaskData attributes, if any.
	    if (taskData != null) {
		Enumeration taskDataAttrs = taskData.getAttrEnum();
		while (taskDataAttrs.hasMoreElements()) {
		    task.setTaskDataAttr(
			(Attribute)taskDataAttrs.nextElement());
		}
	    }

	    // Pass operands to the Task, if any.
	    if (operands != null) {
		task.setOperands(operands);
	    }
	} catch (TaskInitFailedException exception) {
	    result.setReason(exception.getMessage());
	    listener.failed(result);
	    return;
	}

	// Verify the prerequisites of the task.
	task.verifyPrereqs(new ResultListener() {
	    public void succeeded(ResultEvent event) {
		task.getTaskContext().setDialogParent(task);
		result.setResult(task);
		listener.succeeded(result);
	    }

	    public void failed(ResultEvent event) {
		listener.failed(event);
	    }
	});
    }    

    /**
     * Attempt the first stage of task loading: reading the package
     * and task ResourceBundles.
     *
     * @exception java.util.MissingResourceException Thrown if the package
     *            or task ResourceBundles are not found.
     */
    private void loadTaskProperties() throws MissingResourceException {
	// Push the set of PackageP files onto the Task ResourceStack
	_rs.pushPackageBundles(ResourceStack.getPackageName(_taskName));

	// Attempt to load the properties for the named task, which
	// are required to exist.
	// XXX This assumes the task has been installed on the client
	// XXX We may need an alternative method for retrieving it
	// XXX from the server.
	ResourceStack rs = new ResourceStack();
	String taskBundleName;
	try {
	    taskBundleName = _taskName + ResourceStack.BUNDLE_SUFFIX;
	    rs.pushBundle(taskBundleName);
	} catch (MissingResourceException exception) {
	    throw new MissingResourceException(
		MessageFormat.format(
		    _rs.getString("TaskLoader.Error.noTaskProperties"),
		    new Object[] { _taskName }),
		exception.getClassName(),
		exception.getKey());
	}

	// Allow the Task properties to specify a list of additional
	// properties files that should be pushed onto the
	// ResourceStack ahead of the Task ResourceBundle.  This is
	// useful for Task subclasses that share properties with a
	// group of other Tasks, but not necessarily with all other
	// Tasks.
	String[] sharedBundles =
	    rs.getStringArray(Task.SHARED_PROPERTIES, null);

	// Load the shared bundle(s) (if any) onto the Task ResourceStack.
	// Throw a MissingResourceException if the Task specified
	// a non-existent shared bundle.
	if (sharedBundles != null) {
	    for (int ii = 0; ii < sharedBundles.length; ii++) {
		_rs.pushBundle(sharedBundles[ii]);
	    }
	}
	
	// Push the task ResourceBundle onto the task ResourceStack.
	_rs.pushBundle(taskBundleName);
    }
    
    /**
     * Load (but do not instantiate) the named ItemTester class.
     *
     * @param testerClassName Name of the ItemTester class to load.
     *
     * @exception java.lang.ClassNotFoundException if <TT>testerClassName</TT>
     *            does not exist.
     * @exception com.sgi.sysadm.ui.TaskLoaderException if the tester class
     *            was found but could not be loaded for some reason.
     */
    private void loadItemTester(String testerClassName)
	    throws ClassNotFoundException, TaskLoaderException {
	// Only allow one ItemTester per task for now
	Log.assert(_testerConstructor == null,
	           "Attempted to create multiple ItemTesters for task");
		   
	// Locate the tester class.  This will throw ClassNotFoundException
	// if the class does not exist.
	Class taskClass = SysUtil.forName(testerClassName);
	
	_testerClassName = testerClassName;
	
	// Get the class constructor
	String errMsg = null;
	try {
	    _testerConstructor = taskClass.getConstructor(null);
	} catch (NoSuchMethodException exception) {
	    errMsg =
		MessageFormat.format(
		    _rs.getString("TaskLoader.Error.missingTesterConstructor"),
		    new Object[] { _testerClassName });
	} catch (SecurityException exception) {
	    errMsg = MessageFormat.format(
		_rs.getString("TaskLoader.Error.prohibitedTesterConstructor"),
		new Object[] { _testerClassName });
	}
	
	if (_testerConstructor == null) {
	    throw new TaskLoaderException(errMsg);
	}
    }
    
    /**
     * Get an instance of the ItemTester class for the task to be loaded.
     *
     * @exception com.sgi.sysadm.ui.TaskLoaderException if the ItemTester
     *            could not be instantiated for some reason.
     *
     * @return An instance of the ItemTester class for the task to be loaded.
     */
    private ItemTester instantiateItemTester() throws TaskLoaderException {
	Log.assert(_testerConstructor != null,
		   "loadItemTester not yet called.");
	
	ItemTester tester = null;
	
	// Attempt to instantiate the tester.
	String errMsg = null;
	try {
	    tester = (ItemTester)(_testerConstructor.newInstance(null));
	} catch (InstantiationException exception) { 
	    errMsg = MessageFormat.format(
			_rs.getString("TaskLoader.Error.cantInstantiateTester"),
			new Object[] { _testerClassName });
	} catch (IllegalAccessException exception) {
	    errMsg = MessageFormat.format(
		_rs.getString("TaskLoader.Error.illegalTesterConstructorAccess"),
		new Object[] { _testerClassName });
	} catch (IllegalArgumentException exception) {
	    errMsg = MessageFormat.format(
			_rs.getString("TaskLoader.Error.illegalTesterArgument"),
			new Object[] { _testerClassName });
	} catch (InvocationTargetException exception) {
	    errMsg = MessageFormat.format(
		_rs.getString("TaskLoader.Error.unableToInvokeTesterOnTarget"),
		new Object[] { _testerClassName });
	} catch (ClassCastException exception) {
	    errMsg = MessageFormat.format(
			_rs.getString("TaskLoader.Error.unableToCastClass"),
			new Object [] { _testerClassName });
	}
	
	if (tester == null) {
	    throw new TaskLoaderException(errMsg);
	}
	
	return tester;
    }
    
    /**
     * Test if this TaskLoader is equivalent to <TT>other</TT>.
     *
     * @param other A TaskLoader to compare with this one for
     *              equality.
     */
    public boolean equals(TaskLoader other) {
	return other._taskName.equals(_taskName);
    }
}
