//
// TaskPage.java
//
//      A Panel for laying out Form and Guide pages
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import java.awt.*;
import javax.swing.*;
import java.util.*;
import com.sgi.sysadm.util.*;
import com.sgi.sysadm.ui.richText.*;

/**
 * A panel that contains a page for a Task.  The overall look is as follows:
 * The upper-left corner of the panel contains an icon.  To the right
 * of the Icon is the title of the Task.  Below the the title is a
 * RichTextArea known as the IntroText.  Below the IntroText are the
 * task components that make up the functional part of the TaskPage.
 * <p>
 * When adding the task components to the TaskPage, you can choose to
 * add them as normal components or as optional components.  Optional
 * components use an OptionalLinkLabel for their label to distinguish
 * them from the non-optional components that use LinkLabels.
 * <p>
 * The task components are arranged in a LabelComponentPanel.
 * Several of the methods to add task components take a <tt>name</tt>
 * parameter.  This name is used to name the component when it is
 * added to the LabelComponentPanel.  There are many resources you can
 * set based on the component's name to control the way that the task
 * components and their labels are laid out in the
 * LabelComponentPanel.  See the LabelComponentPanel documentation
 * for information about the resources.
 * <p>
 * In the case of the <tt>addTaskComponent</tt> and
 * <tt>addOptionalTaskComponent</tt> methods
 * that takes a <tt>component</tt> and a <tt>name</tt>, the <tt>name</tt> is
 * also used as the <tt>name</tt> argument to the LinkLabel or
 * OptionalLinkLabel that is
 * created for use as a label.  See the LinkLabel documentation on how
 * to use that name to set resources to control the text displayed in
 * the LinkLable and the glossary entry that the LinkLabel will
 * launch.
 *
 * @see RichTextArea
 * @see OptionalLinkLabel
 * @see LinkLabel
 * @see LabelComponentPanel
 */
public class TaskPage extends RPanel {
   
    private static String _class = "TaskPage";
    private Component _icon = null;
    private Component _title = null;
    private Component _intro = null;
    private LabelComponentPanel _fields;
    private ResourceStack _rs;
    
    /**
     * Construct a TaskPage.  <tt>resourceStack</tt> specifies
     * resources for the TaskPage.
     * 
     * @param resourceStack Specifies resources for the TaskPage.
     */
    public TaskPage(ResourceStack resourceStack) {
	_rs = resourceStack;
	setLayout( new TaskPageLayoutManager(_rs));
	setBorder(
	    BorderFactory.createEmptyBorder(_rs.getPixels(Task.TOP_INSET),
					    _rs.getPixels(Task.LEFT_INSET),
					    _rs.getPixels(Task.BOTTOM_INSET),

					    _rs.getPixels(Task.RIGHT_INSET)));
	_fields = new LabelComponentPanel(_rs);
	add(_fields,TaskPageLayoutManager.FIELDS);
    }  

    /** 
     * Adds an Component to the TaskPage in the icon's spot.
     *
     * @param icon The component to place in the icon's spot.  Pass a
     *             null to remove the current icon.
     */
    public void setIcon(Component icon) {
	if (_icon != null) {
	    remove(_icon);
	}
	_icon = icon;
	if (_icon != null) {
	    add(_icon,TaskPageLayoutManager.ICON);
	}
    }

    /** 
     * Adds an Icon the the TaskPage in the icon's spot
     * @param icon The Icon to place in the icon's spot.
     */
    public void setIcon(Icon icon) {
	setIcon(new JLabel(icon));
    }

    /**
     * Adds an Icon to the TaskPage, based on the name of a resource
     * describing the icon.  If the resource is not found, a
     * MissingResourceException will be thrown.
     *
     * @param iconRes The resource naming the FtrIcon or ImageIcon to add.
     */
    public void setIcon(String iconRes) {
	setIcon(_rs.getIcon(iconRes));
    }

    /** 
     * Adds a title to the TaskPage
     * @param title The title for the page.  Pass the empty string to
     *              remove the title.
     */
    public void setTitle(String title) {
	if (title.equals("")) {
	    setTitle((Component)null);
	} else {
	    Component newTitle = new RLabel(title, _rs, "TaskPage.title");
	    setTitle(newTitle);
	}	
    }

    /** 
     * Adds a component to the title's spot on the TaskPage.
     * @param title The title for the page.  Pass a null to remove
     *              the title.
     */
    public void setTitle(Component title) {
	if (_title != null) {
	    remove(_title);
	}
	_title = title;
	if (_title != null) {
	    add(_title,TaskPageLayoutManager.TITLE);
	}
    }
    
    /**
     * Adds the introText to the TaskPage
     * @param intro The IntroText
     */
    public void setIntroText(Component intro) {
	if (_intro != null) {
	    remove(_intro);
	}
	_intro = intro;
	if (_intro != null) {
	    add(_intro, TaskPageLayoutManager.INTRO);
	}
    }

    /**
     * Adds a component but no label to the TaskPage
     * @param component The component to add
     */
    public void addTaskComponent(Component component) {
	_fields.addComponent(component, null, null);
    }
    
    /** 
     * Adds a component and a label to the TaskPage.  A LinkLabel will
     * be created to display the label.
     * 
     * @param component The component to add
     * @param name The name of component.  This will be passed as the
     *             <tt>name</tt> argument to both LinkLabel and
     *             LabelComponentPanel.
     *
     * @return A RichTextComponent that is the label that was created
     *
     * @see LinkLabel
     * @see LabelComponentPanel
     */
    public RichTextComponent addTaskComponent(Component component, 
					      String name) {
	LinkLabel label =  new LinkLabel (_rs, name);
	addTaskComponent(component, name, label);
	return label;
    }
    
    /** 
     * Adds a component and a label to the TaskPage. Since no name is
     * passed, the LabelComponentPanel will use a default layout when
     * adding the component and label to the layout.
     *
     * @param component The component to add
     * @param label The label that describes this component
     */
    public void addTaskComponent(Component component, Component label) {
	_fields.addComponent(component, null, label);
    }  

    /** 
     * Adds a component and a label to the TaskPage. 
     * 
     * @param component The component to add
     * @param name  The name of  this component.  This will be passed as the
     *              <tt>name</tt> argument to LabelComponentPanel, and
     *              can be used to create resources that control the
     *              layout of the component and its label.
     * @param label The label that describes this component
     * 
     * @see LabelComponentPanel
     */
    public void addTaskComponent(Component component, 
				 String name, 
				 Component label) {
	_fields.addComponent(component, name, label);
    }

    /**
     * Adds an optional component but no label to the TaskPage
     * @param component The component to add
     */
    public void addOptionalTaskComponent(Component component) {
	addTaskComponent(component, null, null);
    }
    
    /** 
     * Adds a component and a label to the TaskPage.  An OptionalLinkLabel will
     * be created to display the label.
     * @param component The component to add
     * @param name  The name of  this component.  This will be passed as the
     *              <tt>name</tt> argument to both OptionalLinkLabel and
     *              LabelComponentPanel.
     *
     * @return A RichTextComponent that is the label that was created
     */
    public RichTextComponent addOptionalTaskComponent(Component component, 
						      String name) {
	OptionalLinkLabel label =  new OptionalLinkLabel (_rs, name);
	addTaskComponent(component, name, label);
	return label;
    }
    
    /** 
     * Adds an optional component and a label to the TaskPage. Since no
     * name is passed, the LabelComponentPanel will use a default
     * layout when adding the component and label to the layout.
     * 
     * @param component The component to add
     * @param label The label that describes this component
     */
    public void addOptionalTaskComponent(Component component, Component label) {
	addTaskComponent(component, null, label);
    }
    
    /** 
     * Adds an optional component and a label to the TaskPage. 
     *
     * @param component The component to add
     * @param name  The name of this component.  This will be passed as the
     *              <tt>name</tt> argument to LabelComponentPanel, and
     *              can be used to create resources that control the
     *              layout of the component and its label.
     * @param label The label that describes this component
     */
    public void addOptionalTaskComponent(Component component, 
					 String name, 
					 Component label) {
	addTaskComponent(component, name, label);
    }
    
    /** 
     * Removes a component and any label associated with it.
     *
     * @param component The component to remove
     */
    public void removeTaskComponent(Component component) {
	_fields.removeComponent(component);
    }
}
