//
// TaskPageLayoutManager.java
//
//      A LayoutManager for the TaskPage
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import java.awt.*;
import java.util.*;
import com.sgi.sysadm.util.*;

/**
 * Implements the layout described in TaskPage.
 * 
 * @see TaskPage
 */
/*package*/ class TaskPageLayoutManager implements LayoutManager2,
                                                   DynamicSizeLayoutManager {
    /**
     * The resource <i>TaskPageLayoutManager.iconToTitleSpacing</i> is
     * an integer that specifies the width (in points) of the space to
     * put between the icon and the title.
     */
    public static final String ICON_TO_TITLE_SPACING =
        "TaskPageLayoutManager.iconToTitleSpacing";
    /**
     * The resource <i>TaskPageLayoutManager.titleToIntroSpacing</i>
     * is an integer that specifies the height (in points) of the
     * space between the bottom of the title and the top of the intro text.
     */
    public static final String TITLE_TO_INTRO_SPACING =
        "TaskPageLayoutManager.titleToIntroSpacing";

    /**
     * The resource <i>TaskPageLayoutManager.introToFieldsSpacing</i>
     * is an integer that specifies the height (in points) of the
     * space between the bottom of the intro text and the fields below it.
     */
    public static final String INTRO_TO_FIELDS_SPACING =
        "TaskPageLayoutManager.introToFieldsSpacing";

    public static final String ICON = "Icon";
    public static final String TITLE = "Title";
    public static final String FIELDS = "Fields";
    public static final String INTRO = "Intro";

    private Component _icon = null;
    private Component _title = null;
    private Component _fields = null;
    private Component _intro = null;

    private int _iconToTitleSpacing ;
    private int _titleToIntroSpacing;
    private int _introToFieldsSpacing;

    /**
     * The default constructor for TaskPageLayoutManager
     */
    public TaskPageLayoutManager(ResourceStack rs) {
	_iconToTitleSpacing = rs.getPixels(ICON_TO_TITLE_SPACING);
	_titleToIntroSpacing = rs.getPixels(TITLE_TO_INTRO_SPACING);
	_introToFieldsSpacing = rs.getPixels(INTRO_TO_FIELDS_SPACING);
    }

    /**
     * Called when a new component is being added to the
     * container.
     *
     * @param name A name associated with the component.
     * @param comp The component to add to the layout
     */
    public void addLayoutComponent(String name, Component comp) {
	if (name.equals(ICON)) {
	    _icon = comp;
	}
	else if (name.equals(TITLE)) {
	    _title = comp;
	}
	else if (name.equals(FIELDS)) {
	    _fields = comp;
	}
	else if (name.equals(INTRO)) {
	    _intro = comp;
	} else {
	    Log.fatal("Unsupported name.");
	}
    }

    /**
     * Figure out the height that <tt>parent</tt> wants to be given
     * <tt>width</tt>.
     *
     * @param parent Container to determine height of.
     * @param width proposed width.
     *
     * @return desired height.
     */
    public int getPreferredHeight(Container parent, int width) {
	Insets insets = parent.getInsets();
	int vInsets = insets.top + insets.bottom;
	int hInsets = insets.left + insets.right;
	Dimension size;
	int titleHeight = 0;
	if (_title != null) {
	    titleHeight =  _title.getPreferredSize().height;
	}

	int iconWidth = 0;
	int iconHeight = 0;
	if (_icon != null) {
	    size = _icon.getPreferredSize();
	    iconWidth = size.width + _iconToTitleSpacing;
	    iconHeight = size.height;
	}

	int infoHeight = 0;
	if (_intro != null) {
	    infoHeight =  getComponentSize(_intro,
					   width - hInsets - iconWidth).height;
	}
	int fieldsHeight = 0;
	if (_fields != null) {
	    fieldsHeight = getComponentSize(_fields,
					   width - hInsets - iconWidth).height;
	}
	if (titleHeight != 0 && infoHeight != 0) {
	    titleHeight += _titleToIntroSpacing;
	}
	if (infoHeight != 0 && fieldsHeight != 0) {
	    infoHeight += _introToFieldsSpacing;
	}

	return  vInsets + Math.max(iconHeight, infoHeight + titleHeight) +
	    fieldsHeight;

    }

    private Dimension getComponentSize(Component c,
				       int width) {
	boolean dynamic = c instanceof DynamicSize;
	if (!dynamic) {
	    return c.getPreferredSize();
	} else {
	    return new Dimension(width,
				 ((DynamicSize)c).getPreferredHeight(width));
	}
    }

    /**
     * A method to determine the size of the container.  It's called
     * by all of the get*Size methods.
     */
    private Dimension getSize(Container parent) {
	Insets insets = parent.getInsets();
	int vInsets = insets.top + insets.bottom;
	int hInsets = insets.left + insets.right;
	int width = hInsets;
	int height = vInsets;
	Dimension size = null;

	int iconHeight = 0;

	if (_icon != null) {
	    size = _icon.getPreferredSize();
	    iconHeight = size.height;
	    if (_title == null && _intro == null) {
		width += size.width;
	    } else {
		// If both the title or intro  and icon are present, then we
		// need to add some spacing between them
		width += size.width + _iconToTitleSpacing;
	    }
	}
	if (_icon != null) {
	}

	int titleHeight = 0;
	int titleWidth = 0;
	if (_title != null) {
	    size = _title.getPreferredSize();
	    titleWidth = size.width;
	    titleHeight = size.height;
	    height += _titleToIntroSpacing;
	}
	int introHeight = 0;
	int introWidth = 0;
	if (_intro != null) {
	    size = _intro.getPreferredSize();
	    introWidth = size.width;
	    introHeight = size.height;
	}

	int fieldsHeight = 0;
	int fieldsWidth = 0;
	if (_fields != null) {
	    // add the spacing between the bottom of the intro and the
	    // fields
	    if (_intro != null) {
		fieldsHeight += _titleToIntroSpacing;
	    }

	    size = _fields.getPreferredSize();
	    fieldsWidth = size.width;
	    fieldsHeight = size.height;
	}
	height += Math.max(iconHeight, titleHeight + introHeight) +
	    fieldsHeight;
	width += Math.max(introWidth, Math.max(titleWidth, fieldsWidth));

	return new Dimension(width,height);
    }

    /**
     * Does the layout.
     *
     * @param parent The container containing the objects to lay out.
     */
    public void layoutContainer(Container parent) {
	Insets i = parent.getInsets();
	int currentYPos =  i.top;
	int currentXPos =  i.left;
	Dimension iconSize = new Dimension(0,0);
	Dimension titleSize = new Dimension(0,0);
	Dimension fieldsSize = new Dimension(0,0);
	Dimension containerSize = parent.getSize();
	// Don't include i.top and i.left; these are already
	// accounted for in currentYPos and currentXPos;
	containerSize.height -= i.bottom;
	containerSize.width -= i.right;

	if (_icon != null) {
	    iconSize = _icon.getPreferredSize();
	    _icon.setBounds(currentXPos,
			    currentYPos,
			    iconSize.width,
			    iconSize.height);
	    currentXPos += iconSize.width;
	}
	if (_title != null) {
	    titleSize = _title.getPreferredSize();
	    _title.setBounds(currentXPos +
			     ((_icon == null) ? 0 : _iconToTitleSpacing),
			     currentYPos,
			     titleSize.width,
			     titleSize.height);
	    currentYPos += titleSize.height;
	    currentYPos += _titleToIntroSpacing;
	}

	if (_intro != null) {
	    Dimension introSize = getComponentSize(_intro,
						   containerSize.width
						   - currentXPos);
	    _intro.setBounds(currentXPos +
			     ((_icon == null) ? 0 :
			      _iconToTitleSpacing),
			     currentYPos,
			     introSize.width,
			     introSize.height);
	    currentYPos += introSize.height + _introToFieldsSpacing;
	}

	if (_fields != null) {

	    if (_icon != null) {
		currentXPos += _iconToTitleSpacing;
	    }
	    _fields.setBounds(currentXPos,
			      currentYPos,
			      containerSize.width - currentXPos,
			      containerSize.height - currentYPos);
	}
    }

    /**
     * Returns the minumum size of this component.
     * @see java.awt.Component#getMaximumSize()
     * @see java.awt.LayoutManager#minimumLayoutSize
     */
    public Dimension minimumLayoutSize(Container parent) {
        return getSize(parent);
    }

    /**
     * Returns the preferred size of this component.
     * @see java.awt.Component#getPreferredSize()
     * @see java.awt.LayoutManager#preferredLayoutSize
     */
    public Dimension preferredLayoutSize(Container parent) {
        return getSize(parent);
    }

    /**
     * Returns the maximum size of this component.
     * @see java.awt.Component#getMaximumSize()
     * @see java.awt.Component#getPreferredSize()
     * @see java.awt.LayoutManager2#maximumLayoutSize
     */
    public Dimension maximumLayoutSize(Container target) {
        return getSize(target);
    }

    /**
     * Removes the specified component from the layout.
     * @param comp the component to be removed
     */
    public void removeLayoutComponent(Component comp) {
	if (comp == _icon) {
	    _icon = null;
	    return;
	}
	if (comp == _title) {
	    _title = null;
	    return;
	}
	if (comp == _fields) {
	    _fields = null;
	    return;
	}
	if (comp == _intro) {
	    _intro = null;
	    return;
	}
	Log.fatal("component was not icon nor title nor fields");
    }

    /**
     * Adds the specified component to the layout, using the specified
     * constraint object.
     * @param comp the component to be added
     * @param constraints  where/how the component is added to the layout.
     * @see java.awt.LayoutManager2#addLayoutComponent
     */
    public void addLayoutComponent( Component comp, Object constraints) {
	Log.assert(constraints instanceof String,
	       "constraint must be a String");
	addLayoutComponent((String)constraints,comp);
    }

    /**
     * Returns the alignment along the x axis.  This specifies how
     * the component would like to be aligned relative to other
     * components.  The value should be a number between 0 and 1
     * where 0 represents alignment along the origin, 1 is aligned
     * the furthest away from the origin, 0.5 is centered, etc.
     */
    public float getLayoutAlignmentX(Container target) {
        return 0.5F;
    }

    /**
     * Returns the alignment along the y axis.  This specifies how
     * the component would like to be aligned relative to other
     * components.  The value should be a number between 0 and 1
     * where 0 represents alignment along the origin, 1 is aligned
     * the furthest away from the origin, 0.5 is centered, etc.
     */
    public float getLayoutAlignmentY(Container target) {
        return 0.5F;
    }

    /**
     * Invalidates the layout, indicating that if the layout manager
     * has cached information it should be discarded.
     */
    public void invalidateLayout(Container target) {
    }
}
