//
// TaskRegistry.java
//
//	Determines tasks pertaining to CategoryView and ItemView
//
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//
package com.sgi.sysadm.ui;

import java.util.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.util.*;

/**
 * TaskRegistry determines tasks pertaining to CategoryView and ItemView
 * <p>
 * If a task is applicable to a certain category of items, use inst to
 * add an entry to the directory /var/sysadm/taskRegistry/{Category}/.
 * The entry should include the fully qualified name of the Task, for
 * example: com.sgi.sysadm.ui.tests.AddUserTask
 * <p>
 * To specify the item state that the task is applicable to, the task
 * must first have a directory entry as specified above for the 
 * category of the item under consideration. In addition, add a
 * property with key = <I>Task.itemTester</I> in {Task}P.java or any of the
 * property files associated with the task. 
 * <p>
 * The value of the <I>Task.itemTester</I> property should be a String
 * containing the fully qualified name of a class that implements the 
 * ItemTester interface. ItemTester.testItem() should be defined to
 * perform whatever checks are required to identify if the task is
 * applicable to the item under consideration and return true, if
 * applicable and false otherwise.
 * <p>
 * Typical tests might be:
 * <ol>
 * <li>Applicable to all items
 * <pre>
 * public boolean testItem(Item item) {
 *     return true;
 * }
 * </pre></li>
 * <li>
 * Applicable to items of type "User"
 * <pre>
 * public boolean testItem(Item item) {
 * 	if (item.getType().equals("User")) {
 *		return true;
 *	}
 * }
 * </pre></li>
 * <li>
 * Applicable to items of type "FS" if filesystem name = "foo"
 * and filesystem is mounted and total number of blocks &gt; some
 * limit
 * <pre>
 * public boolean testItem(Item item) {
 *	if (item.getType().equals("FS") &&
 *	    item.getSelector().equals("foo") &&
 *	    item.getBoolean("mounted") &&
 *	    item.getLong("numBlocks") &gt; LIMIT) {
 *       	return true;  
 * 	}
 * }
 * </pre></li></ol>
 * <p>
 * In order to get a task list from the TaskRegistry, you must use a
 * ResultListener to get the result from <tt>getTaskList</tt>.  In the
 * <tt>succeeded</tt> method of the ResultListener, use the
 * <tt>getResult</tt> method of ResultEvent to get the list of tasks.
 * The Object returned by <tt>getResult</tt> should be cast to a
 * TaskLoader[].  For example:
 * <pre>
 * getTaskList("Test", new ResultListener() {
 *     public void succeeded(ResultEvent event) {
 *         TaskLoader[] taskList = (TaskLoader[])event.getResult());
 *         printTaskList(taskList);
 *     }
 *     public void failed(ResultEvent event) {
 *         System.out.println("getTaskList failed.");
 *     }
 * });
 * </pre>
 */
 
public abstract class TaskRegistry {

    protected static final String TASK_REGISTRY = "taskRegistry";

    private Hashtable _taskLists = new Hashtable();
    private int _nextCookie = 0;
    private Vector _listeners = new Vector();
    
    // ListenerElem is an element in our _listeners Vector.  Each
    // element corresponds to a getTaskList call that needs to have a
    // result returned via ResultEvent.
    private class ListenerElem {
	String _catName;
	ResultListener _listener;

	ListenerElem(String catName, ResultListener listener) {
	    _catName = catName;
	    _listener = listener;
	}
    };

    /**
     * Set the HostContext needed for creating TaskLoader instances.
     * 
     * @param hostContext HostContext for creating TaskLoader
     *                    instances.
     */
    public void setHostContext(HostContext hostContext) {
    }

    /**
     * Determines the tasks relevant to a Category.
     * 
     * @param catName Category Name.
     * @param listener Gets result of this operation.  The
     *                 <tt>getResult</tt> method should be called on
     *                 the ResultEvent passed to <tt>listener.succeeded</tt>
     *                 to get the task list for "catName".  The task
     *                 list takes the form of an array of TaskLoaders.
     */
    public synchronized void getTaskList(String catName,
					 ResultListener listener) {
	TaskLoader[] taskArray = (TaskLoader[])_taskLists.get(catName);
	if (taskArray != null) {
	    ResultEvent event = new ResultEvent(TaskRegistry.this, taskArray);
	    listener.succeeded(event);
	    return;
	}
	_listeners.addElement(new ListenerElem(catName, listener));
	computeTaskList(catName);
    }

    /**
     * Determines the tasks relevant to the state of an item.
     * 
     * @param item Item under consideration.
     * @param listener Gets result of this operation.  The
     *                 <tt>getResult</tt> method should be called on
     *                 the ResultEvent passed to <tt>listener.succeeded</tt>
     *                 to get the task list for "catName".  The task
     *                 list takes the form of an array of TaskLoaders.
     */
    public void getTaskList(final Item item, final ResultListener listener) {
	getTaskList(item.getType(), new ResultListener() {
	    public void succeeded(ResultEvent event) {
		TaskLoader[] taskList = (TaskLoader[])event.getResult();
		ItemTester itemTester;
		Vector applicableTasks = new Vector();
		for (int ii = 0 ; ii < taskList.length ; ii++) {
		    try {
			itemTester = taskList[ii].getItemTester();
		    } catch (TaskLoaderException exception) {
			Log.error(TASK_REGISTRY, 
				  "Unable to getItemTester.  " 
				  + exception.getMessage());
			continue;
		    }

		    if (itemTester.testItem(item).passed()) {
			applicableTasks.addElement(taskList[ii]);
		    }
		}

		TaskLoader[] tlList = new TaskLoader[applicableTasks.size()];
		applicableTasks.copyInto(tlList);

		listener.succeeded(new ResultEvent(TaskRegistry.this,
						   tlList));
	    }
	    public void failed(ResultEvent event) {
		listener.failed(event);
	    }
	});
    }
    
    /**
     * Subclasses override this method that TaskRegistry calls when it
     * needs the list of tasks for a particular category.  Subclass
     * must eventually call setTaskList when it is done computing the
     * task list.
     * 
     * @param catName Name of category to compute task list for.
     */
    protected abstract void computeTaskList(String catName);

    /**
     * Called by subclass to report result of computeTaskList.
     * 
     * @param catName Name of category this is a task list for.
     * @param taskArray List of tasks for this category.
     */
    protected synchronized void setTaskList(String catName,
					    TaskLoader[] taskArray) {
	_taskLists.put(catName, taskArray);
	ResultEvent event = new ResultEvent(TaskRegistry.this, taskArray);
	Enumeration enum = _listeners.elements();
	Vector toRemove = new Vector();
	while (enum.hasMoreElements()) {
	    ListenerElem elem = (ListenerElem)enum.nextElement();
	    if (elem._catName.equals(catName)) {
		elem._listener.succeeded(event);
		toRemove.addElement(elem);
	    }
	}
	
	// We need a separate loop to remove the Listeners that we
	// notify because calling Vector.removeElement() while
	// enumerating messes up the enumeration.
	enum = toRemove.elements();
	while (enum.hasMoreElements()) {
	    _listeners.removeElement(enum.nextElement());
	}
    }
}
