//
// TaskShelfController.java
//
//	A class that controls the AbstractTableModel that controls the
//	TaskShelf
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import com.sgi.sysadm.category.*;
import com.sgi.sysadm.util.*;
import java.util.*;

/** 
 * This class will drive a TaskShelfTable by giving the task
 * shelf a list of tasks appropriate to a given item in a given
 * state.
 */
public class TaskShelfController {
    private TaskShelfPanel _tsp;
    private TaskRegistry _tr;
    private Category _category;
    private HostContext _hc;
    private TaskLoader[] _loaders;
    private CategoryListener _catListener = new CatListener();

    /**
     * The constructor
     *
     * @param hc The HostContext
     */
    public TaskShelfController(HostContext hc, TaskLoader[] primary) {
	_tr = hc.getTaskRegistry();
	_hc = hc;
	_loaders = primary;	    
    }
     
    /** Sets the TaskShelfPanel to control
     *
     * @param taskShelfPanel The TaskShelfPanel that this controller will
     *                       control.  
     */
    public void setTaskShelfPanel(TaskShelfPanel tsp) {
	_tsp = tsp;
	if (_loaders != null) {
	    _tsp.setTaskList(_loaders);
	}
    }

    private TaskLoader[] mergeLists(TaskLoader[] primary,
				    TaskLoader[] secondary) {
	int pLen = primary == null ? 0 : primary.length;
	int sLen =secondary == null ? 0 : secondary.length;
	Vector merged = new Vector(pLen + sLen);
	for (int ii = 0; ii < pLen; ii++) {
	    merged.insertElementAt(primary[ii], ii);
	}
	outer: for (int ii = 0; ii < sLen; ii++) {
	    for (int jj = 0; jj < pLen; jj++) {
		if (primary[jj].equals(secondary[ii])) {
		    continue outer;
		}
	    }
	    merged.addElement(secondary[ii]);
	}
	TaskLoader[] array = new TaskLoader[merged.size()];
	merged.copyInto(array);
	return array;
    }
    
	    

    /** 
     * Sets the item whose state will control the tasks shown in
     * the TaskShelf
     * 
     * @param category The category that contains the item whose state
     *                 determines the tasks shown.
     * @param itemSelector The item to use to determine the tasks
     *                     shown.
     */
    public void setItem(Category category, String itemSelector) {
	destroy();
	_category = category;
	if (_category != null) {
	    NotificationFilter nf = new NotificationFilter();
	    nf.monitorItem(itemSelector);
	    _category.addCategoryListener(_catListener, nf);
	}
    } 

    /**
     * Sets the category that determines the tasks shown in the
     * TaskShelf
     *
     * @param category The category that determines the tasks shown.
     */
    public void setCategory(Category category) {
	destroy();
	_tsp.setItem(null);
	_tr.getTaskList(category.getSelector(), new ResultListener() {
	    public void succeeded(ResultEvent event) {
		_tsp.setTaskList(mergeLists(
		    _loaders,(TaskLoader[])event.getResult()));
	    }
	    public void failed(ResultEvent event) {
	    }
	});
    }
    
    /**
     * Releases the category that this object is using.
     */
    public void destroy() {
	if (_category != null) {
	    _category.removeCategoryListener(_catListener);
	    _category = null;
	}
    }
    
    /**
     * Returns the TaskShelfPanel that this controller is controlling
     * @return the TaskShelfPanel
     */
    public TaskShelfPanel getTaskShelfPanel() {
	return _tsp;
    }
    

    private class CatListener extends CategoryAdapter {

	/**
	 * Called by Category after an Item has been added or 
	 * detected at startup to the Category.
	 * 
	 * @param item The item that was added.
	 */
	public void itemAdded(Item item) {
	    _tsp.setItem(item);
	    _tr.getTaskList(item, new ResultListener() {
		public void succeeded(ResultEvent event) {
		    _tsp.setTaskList(mergeLists(
			_loaders,(TaskLoader[])event.getResult()));
		}
		public void failed(ResultEvent event) {
		}
	    });
	}
	
	
	/**
	 * Called by Category when an Item changes. 
	 * 
	 * @param oldItem The old item.
	 * @param newItem The new item.
	 */
	public void itemChanged(Item oldItem, Item newItem){
	    itemAdded(newItem);
	}
	
	/**
	 * Called by Category when an Item is removed.
	 * 
	 * @param item The item that was removed from the list of items
	 */
	public void itemRemoved(Item item){
	    _tsp.setItem(null);
	    _tsp.setTaskList(new TaskLoader[0]);
	} 
    }
}
