//
// TaskShelfPanel.java
//
//	A JPanel that displays a task shelf
//
//  Copyright (c) 1998, 2000 Silicon Graphics, Inc.  All Rights Reserved.
//  
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of version 2.1 of the GNU Lesser General Public
//  License as published by the Free Software Foundation.
//  
//  This program is distributed in the hope that it would be useful, but
//  WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//  
//  Further, this software is distributed without any warranty that it is
//  free of the rightful claim of any third person regarding infringement
//  or the like.  Any license provided herein, whether implied or
//  otherwise, applies only to this software file.  Patent licenses, if
//  any, provided herein do not apply to combinations of this program
//  with other software, or any other product whatsoever.
//  
//  You should have received a copy of the GNU Lesser General Public
//  License along with this program; if not, write the Free Software
//  Foundation, Inc., 59 Temple Place - Suite 330, Boston MA 02111-1307,
//  USA.
//  
//  Contact information: Silicon Graphics, Inc., 1600 Amphitheatre Pkwy,
//  Mountain View, CA 94043, or http://www.sgi.com/
//  
//  For further information regarding this notice, see:
//  http://oss.sgi.com/projects/GenInfo/NoticeExplan/
//

package com.sgi.sysadm.ui;

import java.awt.*;
import java.awt.event.*; 
import java.util.*;
import com.sgi.sysadm.category.*;
import com.sgi.sysadm.ui.event.*;
import com.sgi.sysadm.ui.richText.*;
import com.sgi.sysadm.util.*;
import javax.swing.*;
import javax.swing.table.*;
import javax.swing.border.*;

/** 
 * This class uses a JTable and an associated TabelModel to display a
 * TaskShelf
 */
public class TaskShelfPanel extends JPanel {
    
    /**
     * A resource <i>TaskShelfPanel.width</i> that specifies the
     * desired width (in points) of the TaskShelf
     */
    public static final String WIDTH = "TaskShelfPanel.width";
    
    /**
     * A resource <i>TaskShelfPanel.rows</i> that specifies the
     * desired height (in rows) of the TaskShelf
     */
    public static final String ROWS = "TaskShelfPanel.rows";

    /**
     * A resource <i>TaskShelfPanel.IntercellSpacing.width</i> that
     * specifies the width (in points) of the gap to put between cells
     * in the table.
     */
     public static final String INTERCELL_WIDTH = 
     "TaskShelfPanel.IntercellSpacing.width";

    /**
     * A resource <i>TaskShelfPanel.IntercellSpacing.height</i> that
     * specifies the height (in points) of the gap to put between
     * cells in the table.
     */
     public static final String INTERCELL_HEIGHT = 
     "TaskShelfPanel.IntercellSpacing.height";

    /**
     * A resource <i>TaskShelfPanel.rowHeight</i> that specifies the
     * height of the rows in the panel in points.
     */
    public static final String ROW_HEIGHT = "TaskShelfPanel.rowHeight";

    /**
     * A resource <i>TaskShelfPanel.iconColumnWidth</i> that specifies
     * the width in points of columns that contain icons.
     */
    public static final String ICON_COLUMN_WIDTH =
        "TaskShelfPanel.iconColumnWidth";

    private TaskShelfTableModel _tm;
    private JTable _table;
    private UIContext _uic;
    private Vector _launchListenerList = new Vector(); 
    private Item _item;

    private class LaunchHandler implements TaskLaunchRequestListener {

	public void taskLaunchRequested(TaskLaunchRequestEvent event)
	{
	    if (_item != null &&
		event.getTaskLoader().acceptsOperandType(_item.getType())) {
		Vector operands = new Vector();
		operands.addElement(_item.getSelector());
		event.setOperands(operands);
	    }

	    int size = _launchListenerList.size();
	    if (size == 0) {
		TaskFrame.launchTaskFrame(event,_uic, TaskFrame.POST_RESULT_PANEL);
	    } 
	    for (int ii = 0 ; ii < size ; ii++) {
		((TaskLaunchRequestListener)
		 _launchListenerList.
		 elementAt(ii)).taskLaunchRequested(event);
	    }
	}
    }
    
    private LaunchHandler _launchHandler = new LaunchHandler();
    /** 
     * Constructor that uses a TaskShelfTableModel for the model
     *
     * @param uic The UIContext that the Panel can use to display
     *      messages.
     */
    public TaskShelfPanel(UIContext uic) {
	this(uic,new TaskShelfTableModel(uic.getResourceStack()));
    }	
   
    /** 
     * Constructor that allows you to specify an alternate table model
     *
     * @param uic The UIContext that the Panel can use to display
     *           messages.
     * @param tm The TaskShelfTableModel that will be used to control
     *            the JTable.
     */
    public TaskShelfPanel(final UIContext uic,
			  TaskShelfTableModel tm) {
        // Set backgound color to the Table's background color to work
        // around a Swing 1.1.1 bug.
        setBackground((Color)UIManager.getLookAndFeelDefaults().get("Table.background"));
	setLayout(new BorderLayout());
	_uic = uic;
	ResourceStack rs = uic.getResourceStack();
	_tm = tm;
	_table  = new ComponentTable(_tm);
	JScrollPane scrollpane = new JScrollPane(_table);
	int width = rs.getPixels(WIDTH); 
	int rows = rs.getInt(ROWS);
	add(scrollpane, BorderLayout.CENTER);
	_table.getSelectionModel().setSelectionMode(
	    ListSelectionModel.SINGLE_SELECTION);
	_table.setShowVerticalLines(false);
	_table.setShowHorizontalLines(false);
	int intercellWidth = rs.getPixels(INTERCELL_WIDTH); 
	int interCellHeight = rs.getPixels(INTERCELL_HEIGHT); 
	_table.setIntercellSpacing(new Dimension(intercellWidth,
						 interCellHeight));
	_table.setAutoResizeMode(JTable.AUTO_RESIZE_ALL_COLUMNS);
	_table.setColumnSelectionAllowed(false);
	_table.setRowSelectionAllowed(false);
	_table.setCellSelectionEnabled(false);

	int rowHeight = rs.getPixels(ROW_HEIGHT);
	_table.setRowHeight(rowHeight);
        scrollpane.setPreferredSize(new Dimension(
	    width,  
	    rowHeight * rows + (interCellHeight + 1) * rows));
	
	// Set the width of even numbered columns to the
	// ICON_COLUMN_WIDTH.
	int iconColumnWidth = rs.getPixels(ICON_COLUMN_WIDTH);
	TableColumnModel colModel = _table.getColumnModel();
	int numColumns = tm.getColumnCount();
	for (int col = 0; col < numColumns; col += 2) {
	    TableColumn tableColumn = colModel.getColumn(col);
	    tableColumn.setResizable(false);
	    tableColumn.setPreferredWidth(iconColumnWidth);
	    tableColumn.setMaxWidth(iconColumnWidth);
	}

	// Use a custom renderer for icons, so that we can change the
	// icon color to indicate selection and focus.
	TableCellComponentRenderer iconRenderer = 
	    new TableCellComponentRenderer() {
		public Component getTableCellRendererComponent(
		JTable table, Object value, boolean isSelected,
		boolean hasFocus, int row, int column) {
		JLabel c = (JLabel)super.getTableCellRendererComponent(
		    table, value, false, hasFocus, row, column);
		if (c != null) {
		    c.setHorizontalAlignment(JLabel.CENTER);
		    c.setBorder(new EmptyBorder(1, 2, 1, 2));
		    try {
			FtrIcon icon = ((FtrIcon)c.getIcon());
			icon.setSelected(isSelected);
			icon.setLocated(hasFocus);
		    } catch (Exception ex) {
		    }
		}
		return c;
	    }
	};

	_table.setDefaultRenderer(JLabel.class, iconRenderer);
	_table.setDefaultRenderer(RichTextComponent.class, new
				  TableCellComponentRenderer());

	_table.setTableHeader(null);
    }
   
    /** 
     * Sets the TaskList to display a list of tasks.
     *
     * @param taskLoaderList The list of tasks to display
     */
    public void setTaskList(TaskLoader[] taskLoaderList) {
	_tm.setTaskList(taskLoaderList, _launchHandler);
    }

    /** 
     * Adds a listener to the list that will be notified when a user
     * requests that a task be launched.
     *
     * @param listener The listener to add.
     */
    public void addTaskLaunchRequestListener
                 (TaskLaunchRequestListener listener) {
	_launchListenerList.addElement(listener);
    }

     /** 
     * Removes a TaskLaunchRequestListener
     *
     * @param listener The listener to remove.
     */
    public void removeTaskLaunchRequestListener
                 (TaskLaunchRequestListener listener) {
	_launchListenerList.removeElement(listener);
    }

    /** 
     * Sets the Item that should be passed in the
     * TaskLaunchRequestEvent if the user launches a task.
     * 
     * @param item The item.  Can be null if there is no applicable item.
     */
    public void setItem(Item item) {
	_item = item;
    }
}
